/* -*- C -*- */
/* solvers.c
 */

#include <lfc/lfci.h>

LFC_BEGIN_C_DECLS

extern int dgels_(char *trans, integer *m, integer *n, integer * nrhs,
  doublereal *a, integer *lda, doublereal *b, integer *ldb, doublereal *work,
  integer *lwork, integer *info, ftnlen trans_len);
extern int dgesv_(integer *n, integer *nrhs, doublereal *a, integer *lda,
  integer *ipiv, doublereal *b, integer *ldb, integer *info);
extern int dposv_(char *uplo, integer *n, integer *nrhs, doublereal *a,
  integer *lda, doublereal *b, integer *ldb, integer *info, ftnlen uplo_len);
extern int sgels_(char *trans, integer *m, integer *n, integer * nrhs,
  real *a, integer *lda, real *b, integer *ldb, real *work,
  integer *lwork, integer *info, ftnlen trans_len);
extern int sgesv_(integer *n, integer *nrhs, real *a, integer *lda,
  integer *ipiv, real *b, integer *ldb, integer *info);
extern int sposv_(char *uplo, integer *n, integer *nrhs, real *a,
  integer *lda, real *b, integer *ldb, integer *info, ftnlen uplo_len);

int
LFC_linear_solve(enum LFC_Solver op, enum LFC_UpLo uplo, enum LFC_Transpose trans,
  enum LFC_Datatype dtype, int m, int n, int nrhs, void *a, int lda, int *piv,
  void *b, int ldb, int *info) {
  int i, mn;
  char chTrans, chUpLo;

  chTrans = e2chTrans(trans);
  chUpLo = e2chUpLo(uplo);

  mn = MAX( m, n );

  if (! a || ! b || ! piv) return LFC_FAILURE;

  if (LFC_DOUBLE == dtype) {
    double *work, query_work, *da = (double *)a, *db = (double *)b;
    int lwork;

    switch (op) {
      case LFC_GELS:
	/* query optimal size for 'work' */
	lwork = -1;
	dgels_( &chTrans, &m, &n, &nrhs, da, &lda, db, &ldb, &query_work, &lwork, &i, 1 );
	lwork = (int)query_work;

	work = LFC_MALLOC( double, lwork );
	if (work) {
	  dgels_( &chTrans, &m, &n, &nrhs, da, &lda, db, &ldb, work, &lwork, &i, 1 );

	  LFC_FREE( work );
	} else
	  i = -9;
	/* FIXME: try less optimal work space */
      break;

      case LFC_GESV:
	dgesv_( &n, &nrhs, da, &lda, piv, db, &ldb, &i );
      break;

      case LFC_POSV:
	dposv_( &chUpLo, &n, &nrhs, da, &lda, db, &ldb, &i, 1 );
      break;
    }
  } else if (LFC_FLOAT == dtype) {
    float *work, query_work, *sa = (float *)a, *sb = (float *)b;
    int lwork;

    switch (op) {
      case LFC_GELS:
	/* query optimal size for 'work' */
	lwork = -1;
	sgels_( &chTrans, &m, &n, &nrhs, sa, &lda, sb, &ldb, &query_work, &lwork, &i, 1 );
	lwork = (int)query_work;

	work = LFC_MALLOC( float, lwork );
	if (work) {
	  sgels_( &chTrans, &m, &n, &nrhs, sa, &lda, sb, &ldb, work, &lwork, &i, 1 );

	  LFC_FREE( work );
	} else
	  i = -9;
	/* FIXME: try less optimal work space */
      break;

      case LFC_GESV:
	sgesv_( &n, &nrhs, sa, &lda, piv, sb, &ldb, &i );
      break;

      case LFC_POSV:
	sposv_( &chUpLo, &n, &nrhs, sa, &lda, sb, &ldb, &i, 1 );
      break;
    }
  }

  *info = i;

  if (i) return LFC_FAILURE;

  return LFC_SUCCESS;
}	/* LFC_linear_solve */

int
LFC_run_mpi(char **argv) {
  pid_t kid; int fds[2];

  if (pipe( fds )) return -1;

  kid = fork();

  if (kid == 0) { /* child process */
    close( fds[0] );
    close( 1 );
    dup( fds[1] );
    close( fds[1] );

    execvp( argv[0], argv );

    perror( "execvp()" );
    exit( EXIT_FAILURE );
  } else if (kid > 0) { /* parent process */
    int c, new0, rv, pid;

    close( fds[1] );
    new0 = dup( 0 );
    close( 0 );
    dup( fds[0] );
    close( fds[0] );

    for (c = getchar(); c != EOF; c = getchar()) putchar( c );

    close( 0 );
    dup( new0 );
    close( new0 );

    pid = wait( &rv );
    if (pid != kid) {
      if (pid < 0) perror( "wait()" );
      return -1;
    }

    if (WIFEXITED(rv)) {
      int extsts = WEXITSTATUS(rv);
      if (extsts != 0) return -1;
      return 0;
    } else return -1;
  } else { /* fork() failed */
    perror( "fork()" );
    return -1;
  }

  return 0;
}	/* LFC_run_mpi */

LFC_END_C_DECLS
