/* -*- C -*- */
/* schedulr.c
 */

#include <lfc/lfci.h>

LFC_BEGIN_C_DECLS

/* the bigger the memory the "smaller" the host - the host with the largest
   memory will end up at position 0 */
static int
hostMemCmp(const void *e1, const void *e2) {
  Host *h1 = (Host *)e1, *h2 = (Host *)e2;
  if (h1->freeMem > h2->freeMem) return -1;
  else if (h1->freeMem < h2->freeMem) return 1;
  return 0;
}

/* if available CPU cycles less than 2 than we have 1 CPU otherwise truncate */
#define avCPU2int(f) ((f)<2.0 ? 1.0 : (int)(f))

static int
countCPUs(Host *hosts, int nhosts) {
  int i, cntCPU;

  cntCPU = 0;
  for (i = 0; i < nhosts; i++) cntCPU += avCPU2int( hosts[i].freeCPU );

  /* FIXME: no CPUs available somewhere */
  if (cntCPU < nhosts) cntCPU = nhosts;

  return cntCPU;
}

static int
bestGrid(Host *hosts, int nhosts, int *gp, int *gq, double *qlty) {
  int rows, cols, srows, cntCPU; double f, sf, bestf = 0.25;

  if (nhosts < 0 || ! gp || ! gq) return LFC_FAILURE;

  cntCPU = countCPUs( hosts, nhosts );

  srows = 1; sf = 1.0 / cntCPU;
  for (rows = 2; rows < cntCPU; rows++) {
    cols = cntCPU / rows;
    if (rows >= cols) break;
    if (rows * cols != cntCPU) continue;
    f = (double)rows / (double)cols;

    /* higher numbers of rows are preferred so <= is used instead of < */
    if (fabs(f - bestf) <= fabs(sf - bestf)) {
      sf = f;
      srows = rows;
    }
  }
 
  *gp = srows;
  *gq = cntCPU / srows;
  if (qlty) *qlty = 1.0 - fabs( sf - bestf );

  return LFC_SUCCESS;
}	/* bestGrid */

#ifdef HAVE_STRINGIZE
#define LFC_Assert(e) LFC_AssertImpl((e)?1:0, #e, __FILE__, __LINE__ )
#else
#define LFC_Assert(e) LFC_AssertImpl((e)?1:0, "", __FILE__, __LINE__ )
#endif /* HAVE_STRINGIZE */

static int
LFC_AssertImpl(int expr, const char *sExpr, const char *fileName,
		int lineNumber) {
  if (! expr) {
    printf( "Assertion failed at %s(%d): %s", fileName, lineNumber, sExpr );
    exit( EXIT_FAILURE );
    return LFC_FAILURE;
  }
  return LFC_SUCCESS;
}

static int
createMachinefile(Host *hosts, int nhosts, char **machinefile) {
  int i, j, cntCPU, mypid, n; char *mfile, lname[4096+1], *path; FILE *f;

  n = (sizeof lname) / (sizeof *lname) - 1;

  LFC_Assert( machinefile );

  /* FIXME: directory separator */
  mypid = getpid();
  mfile = LFC_StrSubst1( "/tmp/lfc%d.mch", &mypid );
  if (! mfile) return LFC_FAILURE;

  gethostname( lname, n );
  lname[n] = '\0';

  f = fopen( mfile, "w" );
  if (! f) {
    LFC_FREE( mfile );
    return LFC_FAILURE;
  }
  *machinefile = mfile;

  path = LFC_getPDSLVpath();
  if (! path) {
    fclose( f );
    LFC_FREE( mfile );
    *machinefile = NULL;
    return LFC_FAILURE;
  }

  fprintf( f, "%s 0 %s\n", lname, path );

  for (i = 0; i < nhosts; i++) {
    cntCPU = countCPUs( hosts + i, 1 );

    for (j = 0; j < cntCPU; j++)
      fprintf( f, "%s 1 %s\n", hosts[i].name, path );
  }

  fclose( f );

  return LFC_SUCCESS;
}	/* createMachinefile */

int
LFC_memRule(int ndim, int nrhs, int size, int *gp, int *gq,
	    char **machinefile) {
  Host *hosts;
  int i, rv, nhosts, curHosts, cntCPU;
  double totCMem, mtxMem, curMem, qlty, qlty1;

  int lgp1, *gp1, lgq1, *gq1;
  gp1 = &lgp1;
  gq1 = &lgq1;

  if (ndim < 1 || nrhs < 1 || size < 1 || ! gp || ! gq || ! machinefile)
    return LFC_FAILURE;

  /* if matrix is small (not many flops for factorization) and area of matrix
     and right hand side is small */
  if (ndim <= 1000 && ndim < 1e6 / nrhs) {
    *gp = *gq = 1;
    *machinefile = NULL;
    return LFC_SUCCESS;
  }

  rv = LFC_getHosts( &hosts );
  if (LFC_FAILURE == rv) {
    *gp = *gq = 1;
    *machinefile = NULL;
    return LFC_SUCCESS;
  }

  totCMem = 0.0;
  for (i = 0; ! hosts[i].last; i++) totCMem += hosts[i].freeMem;
  nhosts = i;

  if (nhosts < 1) { /* something went wrong, stick to one host */
    *gp = *gq = 1;
    *machinefile = NULL;
    LFC_FREE( hosts );
    return LFC_SUCCESS;
  }

  cntCPU = countCPUs( hosts, nhosts );
  if (cntCPU < 2) { /* FIXME: may be the code should be shipped to some other CPU */
    *gp = *gq = 1;
    *machinefile = NULL;
    LFC_FREE( hosts );
    return LFC_SUCCESS;
  }

  mtxMem = (double)size / 1024.0 * (double)ndim / 1024.0 * ((double)ndim + (double)nrhs);
  mtxMem /= 0.8; /* matrix should occupy 80% of memory */

  /* memory required is all (or more) what's currently available */
  if (totCMem <= mtxMem) {
    rv = createMachinefile( hosts, nhosts, machinefile );

    if (rv) {
      LFC_FREE( hosts );
      *gp = *gq = 1;
      *machinefile = NULL;
      return LFC_SUCCESS;
    }

    bestGrid( hosts, nhosts, gp, gq, &qlty );
    LFC_FREE( hosts );

    return LFC_SUCCESS;
  }

  /* sort with descending memory sizes */
  qsort( hosts, nhosts, sizeof *hosts, hostMemCmp );

  /* FIXME: what if the file is on the local file system? include this node
     (play with -nolocal) */

  /* FIXME: what if the host with the most memory is not local? */
  if (hosts[0].freeMem >= mtxMem) {
    LFC_FREE( hosts );
    *gp = *gq = 1;
    *machinefile = NULL;
    return LFC_SUCCESS;
  }

  /* go through hosts with decreasing amount of memory per host */
  for (i = 0; ; i++) {
    curHosts = i + 1;

    /* assume each host has as much memory as this one */
    curMem = curHosts * hosts[i].freeMem;

    if (curMem >= mtxMem || hosts[curHosts].last) {

      if (hosts[curHosts].last) { /* if all of the nodes should be used */
	rv = createMachinefile( hosts, curHosts, machinefile );

	if (rv) {
	  LFC_FREE( hosts );
	  *gp = *gq = 1;
	  *machinefile = NULL;
	  return LFC_SUCCESS;
	}

	bestGrid( hosts, curHosts, gp, gq, &qlty );
	LFC_FREE( hosts );

	return LFC_SUCCESS;
      }

      /* there were some hosts that were not included */

      /* Try better */
      bestGrid( hosts, curHosts, gp, gq, &qlty );
      bestGrid( hosts, curHosts + 1, gp1, gq1, &qlty1 );

      if (qlty1 > qlty && *gp * *gq > 10) {
	curHosts++;
	*gp = *gp1;
	*gq = *gq1;
      }

      rv = createMachinefile( hosts, curHosts, machinefile );
      LFC_FREE( hosts );

      if (rv) {
	*gp = *gq = 1;
	*machinefile = NULL;
	return LFC_SUCCESS;
      }

      return LFC_SUCCESS;
    }
  }

  return LFC_SUCCESS;
}	/* LFC_memRule */

void prnt(const char *s) {printf("%s\n", s); fflush(stdout);}
#define P(x) printf( "%s(%d):%s=%g\n", __FILE__ , __LINE__ , #x , (double)(x) )

int
LFC_mfile(int gp, int gq, char **machinefile) {
  int i, rv, nhosts, cntCPU; Host *hosts;

  rv = LFC_getHosts( &hosts );
  if (rv) {
    prnt( "No parallel environment!" );
    return LFC_FAILURE;
  }

  for (i = 0; ! hosts[i].last; i++) ;
  nhosts = i;

  cntCPU = countCPUs( hosts, nhosts );
  if (cntCPU < gp * gq) {
    P(cntCPU);
    P(gp);
    P(gq);
    prnt( "Not enough CPUs!" );
    LFC_FREE( hosts );
    return LFC_FAILURE;
  }
  if (cntCPU != nhosts) {
    prnt( "More than one CPU per host!" );
    LFC_FREE( hosts );
    return LFC_FAILURE;
  }

  rv = createMachinefile( hosts, gp * gq, machinefile );
  if (rv) {
    prnt( "Couldn't create machine file!" );
    return LFC_FAILURE;
  }
  return LFC_SUCCESS;
}

/* Count number of new hosts - it will be equal to number of CPUs; freeCPU
   value for a host is used in the follwing way:
   freeCPU
   (0, 1.9)   -> 1
   (1.9, 2.9) -> 2, and so on
*/
static int
flattenCPU(double cpu) {
  int i, cnt;

  if (cpu < 1.9) {
    cnt = 1;
  } else {
    cnt = (int)cpu; /* this is ceiling function of `cpu' */
    if (cpu - cnt >= 0.9) cnt += 1; /* this is in case at least 90% remained */
  }

  return cnt;
}

int
LFC_flattenHosts(Host *oldHosts, Host **uNewHosts) {
  int i, j, cnt, curCnt; Host *h, *nh, *newHosts; double mem, f;

  if (! oldHosts || ! uNewHosts || oldHosts[0].last) return LFC_FAILURE;

  cnt = 0;
  for (i = 0; ! oldHosts[i].last; i++) {
    h = oldHosts + i;
    curCnt = flattenCPU( h->freeCPU );
    cnt += curCnt;
  }

  newHosts = LFC_MALLOC( Host, cnt + 1 );
  if (! newHosts) return LFC_FAILURE;

  cnt = 0;
  for (i = 0; ! oldHosts[i].last; i++) {
    h = oldHosts + i;
    curCnt = flattenCPU( h->freeCPU );

    nh = newHosts + cnt;
    if (1 == curCnt) {
      LFC_strcpy( nh->name, h->name );
      nh->freeCPU = h->freeCPU;
      nh->freeMem = h->freeMem;
      nh->last = 0;
    } else {
      mem = h->freeMem / h->freeCPU;
      for (j = 0; j < curCnt; j++) {
	nh = newHosts + cnt + j;
	LFC_strcpy( nh->name, h->name );
	nh->freeCPU = 1.0;
	nh->freeMem = mem;
	nh->last = 0;
      }
      f = curCnt - h->freeCPU;
      if (f > 0.0) nh->freeCPU = f;
    }

    cnt += curCnt;
  }

  newHosts[cnt].last = 1;
  *uNewHosts = newHosts;
  return LFC_SUCCESS;
}	/* LFC_flattenHosts */

static int
hostCPUCmp(const void *e1, const void *e2) {
  Host *h1 = (Host *)e1, *h2 = (Host *)e2;
  if (h1->freeCPU > h2->freeCPU) return -1;
  else if (h1->freeCPU < h2->freeCPU) return 1;
  return 0;
}

int
LFC_schedule(int m, int n, int nrhs, int size, int *gp, int *gq,
  char **machinefile) {
  int i, rv, nhosts; Host *hosts, *newHosts; double totCMem, mtxMem, qlty;
  int ndim;

  if (m > n)
    ndim = m;
  else
    ndim = n;

  if (ndim < 1 || nrhs < 1 || size < 1 || ! gp || ! gq || ! machinefile)
    return LFC_FAILURE;

  /* if matrix is small (not many flops for factorization) and area of matrix
     and right hand side is small use one node */
  if (ndim <= 1000 && ndim < 1e6 / nrhs) {
    *gp = *gq = 1;
    *machinefile = NULL;
    return LFC_SUCCESS;
  }

  rv = LFC_getHosts( &hosts );
  if (LFC_FAILURE == rv) {
    *gp = *gq = 1;
    *machinefile = NULL;
    return LFC_SUCCESS;
  }

  rv = LFC_flattenHosts( hosts, &newHosts );
  if (LFC_FAILURE == rv) {
    *gp = *gq = 1;
    *machinefile = NULL;
    LFC_FREE( hosts );
    return LFC_SUCCESS;
  }
  LFC_FREE( hosts );
  hosts = newHosts;

  totCMem = 0.0;
  for (i = 0; ! hosts[i].last; i++) totCMem += hosts[i].freeMem;
  nhosts = i;

  if (nhosts < 1) { /* something went wrong, stick to one host */
    *gp = *gq = 1;
    *machinefile = NULL;
    LFC_FREE( hosts );
    return LFC_SUCCESS;
  }

  mtxMem = (double)size / 1024.0 * (double)ndim / 1024.0 * ((double)ndim + (double)nrhs);
  mtxMem /= 0.8; /* matrix should occupy no more than 80% of memory */

  /* memory required is all (or more) what's currently available */
  if (totCMem > 0.0 && totCMem <= mtxMem) {
    rv = createMachinefile( hosts, nhosts, machinefile );

    if (rv) {
      LFC_FREE( hosts );
      *gp = *gq = 1;
      *machinefile = NULL;
      return LFC_SUCCESS;
    }

    bestGrid( hosts, nhosts, gp, gq, &qlty );
    LFC_FREE( hosts );

    return LFC_SUCCESS;
  }

  /* sort with descending CPU availability */
  qsort( hosts, nhosts, sizeof *hosts, hostCPUCmp );

  /*
    for (i = 0; ! hosts[i].last; i++)
      printf( "%s %g %g\n", hosts[i].name, hosts[i].freeCPU, hosts[i].freeMem);
  */

  /* FIXME: what about `nrhs' */
  /* select 20 panels per process column and 1/4 process grid aspect ratio */
  *gq = (int)(ndim / 80.0 / 20.0);
  *gp = (int)(0.25 * *gq + 0.5);
  if (*gq < 1) *gq = 1;
  if (*gp < 1) *gp = 1;

  if (*gp == 1 && *gq == 1) {
    LFC_FREE( hosts );
    *machinefile = NULL;
    return LFC_SUCCESS;
  }

  if (*gp * *gq >= nhosts) {
    rv = createMachinefile( hosts, nhosts, machinefile );
    if (rv) {
      *gp = *gq = 1;
      *machinefile = NULL;
      LFC_FREE( hosts );
      return LFC_SUCCESS;
    }

    bestGrid( hosts, nhosts, gp, gq, &qlty );
    LFC_FREE( hosts );
    return LFC_SUCCESS;
  }

  rv = createMachinefile( hosts, *gp * *gq, machinefile );
  LFC_FREE( hosts );
  if (rv) {
    *gp = *gq = 1;
    *machinefile = NULL;
    return LFC_SUCCESS;
  }

  return LFC_SUCCESS;
}	/* LFC_schedule */

LFC_END_C_DECLS
