/* -*- C -*- */
/* argvutil.c
 */

#include <lfc/lfci.h>

LFC_BEGIN_C_DECLS

#define VPF(f,v) fprintf(f, "%s(%03d)@%d:%s=%g\n", __FILE__, __LINE__, getpid(), #v, (double)(v) )
#define PIPEBUFLEN (4 * 1024)
static int io(int fd, void *buf, size_t count, int inout) {
  int rv, icnt;

  for (icnt = count; icnt > 0; buf = (char *)buf + rv) {
    if (inout) rv = write( fd, buf, icnt );
    else rv = read( fd, buf, icnt );
    if (0 == rv) {
      VPF( stderr, inout );
      return 0;
    } else if (rv < 0) {
      if (inout) perror( "write()" ); else perror( "read()" );
      return icnt - count;
    }
    icnt -= rv;
  }

  return count;
}

static int
vpipe_vector(int n, void *a, int siz, int inout, int fd) {
  int pbl = PIPEBUFLEN, nd = pbl / siz; int i; char *aptr = a;

  for (i = n; i >= nd; i -= nd, aptr += pbl)
    if (pbl != io(  fd, aptr, pbl, inout )) {
      VPF(stderr, i);
      return -1;
    }

  if (i > 0) {
    i *= siz;
    if (i != io( fd, aptr, i, inout )) {
      VPF(stderr, i);
      return -1;
    }
  }

  return 0;
}	/* vpipe_vector */

int
LFC_vpipe_matrix(int m, int n, int siz, void *a, int lda, int inout, int fd) {
  int i, j, ldp = lda * siz; char *pa = a;
  for (i = 0; i < n; i++, pa += ldp) {
    j = vpipe_vector( m, pa, siz, inout, fd );
    if (j) return j;
  }
  return 0;
}	/* LFC_vpipe_matrix */

#if 0
int
LFC_dpipe_matrix(int m, int n, double *a, int lda, int inout, int fd) {
  int i, j; double *pa = a;
  for (i = 0; i < n; i++, pa += lda) {
    j = vpipe_vector( m, pa, sizeof *a, inout, fd );
    if (j) return j;
  }
  return 0;
}	/* LFC_dpipe_matrix */
#endif

int
LFC_dpipe_matrix(int m, int n, double *a, int lda, int inout, int fd) {
  LFC_vpipe_matrix( m, n, sizeof *a, a, lda, inout, fd );
  return 0;
}
int
LFC_ipipe_matrix(int m, int n, int *a, int lda, int inout, int fd) {
  LFC_vpipe_matrix( m, n, sizeof *a, a, lda, inout, fd );
  return 0;
}

int
LFC_pipeArgv(char **argv, LFC_PipeBuf *pbuf) {
  int chid, to_child[2], to_parent[2];

  pipe( to_child );
  pipe( to_parent );

  chid = fork();
  if (0 == chid) { /* child process */
    close( to_child[1] );
    close( to_parent[0] );

    close( 0 );
    dup( to_child[0] );

    close( 1 );
    dup( to_parent[1] );

    execvp( argv[0], argv );
    perror( "execvp()" );

    exit( EXIT_FAILURE );
    return LFC_FAILURE; /* so the compiler knows if it cares */
  } else if (chid > 0) { /* parent process */
    close( to_child[0] );
    close( to_parent[1] );

    pbuf->fdRead  = to_parent[0];
    pbuf->fdWrite = to_child[1];
    pbuf->chid = chid;
  } else { /* fork() failed */
    perror( "fork()" );
    return LFC_FAILURE;
  }

  return LFC_SUCCESS;
}	/* LFC_pipeArgv */

char *
LFC_pipeReadLine(LFC_PipeBuf *pbuf) {
  static char line[2048+1];
  int i, j, n;

  /* FIXME: deal with long lines */
  /* FIXME: call read() less times */

  n = (sizeof line) / (sizeof *line) - 1;

  for (i = 0; i < n; i++) {
    j = read( pbuf->fdRead, line + i, 1 );
    if (j < 0) {
      perror( "read()" );
      return NULL;
    }
    if (j == 0) return NULL;
    if ('\n' == line[i]) break;
  }

  line[i + 1] = '\0';
  return line;
}	/* LFC_pipeReadLine */

int
LFC_pipeClose(LFC_PipeBuf *pbuf) {
  int chid, pid, rv, extsts;

  if (! pbuf) return LFC_FAILURE;

  chid = pbuf->chid;

  if (close( pbuf->fdRead ))  perror( "close()" );
  if (close( pbuf->fdWrite )) perror( "close()" );

  do {
    pid = wait( &rv );
    if (pid < 0) perror( "wait()" );
  } while (pid != chid);

  if (WIFEXITED(rv)) {
    extsts = WEXITSTATUS(rv);
    if (extsts != 0) return -1;
    return LFC_SUCCESS;
  } else return LFC_FAILURE;

  return LFC_FAILURE;
}	/* LFC_pipeClose */

int
LFC_pipe_solve(enum LFC_Solver op, enum LFC_UpLo uplo, enum LFC_Transpose trans,
  enum LFC_Datatype dtype, int m, int n, int nrhs, void *a, int lda, int *piv,
  void *b, int ldb, int *info) {
  pid_t chid; int mn, to_child[2], to_parent[2], linfo = 0;
  char fexec[] = "lfcslvpp";
#define IBUF (sizeof(int)*3+2)
  char mbuf[IBUF], nbuf[IBUF], nrhsbuf[IBUF], opbuf[2], uplobuf[2], transbuf[2];
  char typebuf[2];
  char *efile, *argv[9];

  mn = MAX( m, n );

  efile = LFC_strmrg3( LFC_get_libexec_dir(), "/", fexec );
  if (! efile) {
    return LFC_FAILURE;
  }

  opbuf[1] = uplobuf[1] = transbuf[1] = typebuf[1] = '\0';

  opbuf[0]    = e2chSolver( op );
  uplobuf[0]  = e2chUpLo(  uplo );
  transbuf[0] = e2chTrans( trans );
  typebuf[0]  = e2chDtype( dtype );
  sprintf( mbuf, "%d", m  );
  sprintf( nbuf, "%d", n  );
  sprintf( nrhsbuf, "%d", nrhs  );

  argv[0] = efile;
  argv[1] = opbuf;
  argv[2] = uplobuf;
  argv[3] = transbuf;
  argv[4] = typebuf;
  argv[5] = mbuf;
  argv[6] = nbuf;
  argv[7] = nrhsbuf;
  argv[8] = NULL;

  pipe( to_child );
  pipe( to_parent );

  chid = fork();
  if (0 == chid) { /* child process */
    close( to_child[1] );
    close( to_parent[0] );

    close( 0 );
    dup( to_child[0] );

    close( 1 );
    dup( to_parent[1] );

    execvp( efile, argv );
    LFC_FREE( efile );
    perror( "execvp()" );

    linfo = 1;
    write( to_parent[1], &linfo, sizeof(int) ); /* notify parent of failure */

    exit( EXIT_FAILURE );
    return -1; /* so the compiler knows if it cares */
  } else if (chid > 0) { /* parent process */
    int rv;
    close( to_child[0] );
    close( to_parent[1] );

    read( to_parent[0], &linfo, sizeof(int) ); /* see if execvp() succeeded in child */

    if (0 == linfo) {
      LFC_vpipe_matrix( m, n,    e2szDtype(dtype), a, lda, 1, to_child[1] );
      LFC_vpipe_matrix( mn,nrhs, e2szDtype(dtype), b, ldb, 1, to_child[1] );
      close( to_child[1] );

      LFC_vpipe_matrix( m, n,    e2szDtype(dtype), a, lda, 0, to_parent[0] );
      if (LFC_GESV == op) LFC_ipipe_matrix( n, 1,  piv,   n, 0, to_parent[0] );
      LFC_vpipe_matrix( mn,nrhs, e2szDtype(dtype), b, ldb, 0, to_parent[0] );

      read( to_parent[0], &linfo, sizeof(int) ); /* get solver's 'info' value */
      close( to_parent[0] );
    } else {
      close( to_child[1] );
      close( to_parent[0] );
      linfo = n + 1;
    }

    wait( &rv );
  } else { /* fork() failed */
    perror( "fork()" );
    return LFC_FAILURE;
  }
  if (info) *info = linfo;
  if (linfo) return LFC_FAILURE;

  return LFC_SUCCESS;
}	/* LFC_pipe_solve */

LFC_END_C_DECLS
