/* -*- C -*- */
/* lfccblas.c
   LFC interface to BLAS with CBLAS API.

   Piotr Luszczek, February 8, 2003
*/

#include <lfc/lfci.h>

#define ACX_DASUM LFC_BLASF(dasum,DASUM,dasum_,DASUM_,dasum__,DASUM__)
#define ACX_DTRSV LFC_BLASF(dtrsv,DTRSV,dtrsv_,DTRSV_,dtrsv__,DTRSV__)
#define ACX_DGBMV LFC_BLASF(dgbmv,DGBMV,dgbmv_,DGBMV_,dgbmv__,DGBMV__)
#define ACX_DTBSV LFC_BLASF(dtbsv,DTBSV,dtbsv_,DTBSV_,dtbsv__,DTBSV__)
#define ACX_DSYR2K LFC_BLASF(dsyr2k,DSYR2K,dsyr2k_,DSYR2K_,dsyr2k__,DSYR2K__)
#define ACX_DGER LFC_BLASF(dger,DGER,dger_,DGER_,dger__,DGER__)
#define ACX_DROT LFC_BLASF(drot,DROT,drot_,DROT_,drot__,DROT__)
#define ACX_DSYMV LFC_BLASF(dsymv,DSYMV,dsymv_,DSYMV_,dsymv__,DSYMV__)
#define ACX_DSYR LFC_BLASF(dsyr,DSYR,dsyr_,DSYR_,dsyr__,DSYR__)
#define ACX_DSYR2 LFC_BLASF(dsyr2,DSYR2,dsyr2_,DSYR2_,dsyr2__,DSYR2__)
#define ACX_DSBMV LFC_BLASF(dsbmv,DSBMV,dsbmv_,DSBMV_,dsbmv__,DSBMV__)
#define ACX_DAXPY LFC_BLASF(daxpy,DAXPY,daxpy_,DAXPY_,daxpy__,DAXPY__)
#define ACX_DCOPY LFC_BLASF(dcopy,DCOPY,dcopy_,DCOPY_,dcopy__,DCOPY__)
#define ACX_DDOT LFC_BLASF(ddot,DDOT,ddot_,DDOT_,ddot__,DDOT__)
#define ACX_DNRM2 LFC_BLASF(dnrm2,DNRM2,dnrm2_,DNRM2_,dnrm2__,DNRM2__)
#define ACX_DSCAL LFC_BLASF(dscal,DSCAL,dscal_,DSCAL_,dscal__,DSCAL__)
#define ACX_DSWAP LFC_BLASF(dswap,DSWAP,dswap_,DSWAP_,dswap__,DSWAP__)
#define ACX_DGEMV LFC_BLASF(dgemv,DGEMV,dgemv_,DGEMV_,dgemv__,DGEMV__)
#define ACX_DTRMV LFC_BLASF(dtrmv,DTRMV,dtrmv_,DTRMV_,dtrmv__,DTRMV__)
#define ACX_DGEMM LFC_BLASF(dgemm,DGEMM,dgemm_,DGEMM_,dgemm__,DGEMM__)
#define ACX_DTRMM LFC_BLASF(dtrmm,DTRMM,dtrmm_,DTRMM_,dtrmm__,DTRMM__)
#define ACX_DTRSM LFC_BLASF(dtrsm,DTRSM,dtrsm_,DTRSM_,dtrsm__,DTRSM__)
#define ACX_DSYRK LFC_BLASF(dsyrk,DSYRK,dsyrk_,DSYRK_,dsyrk__,DSYRK__)
#define ACX_DSYMM LFC_BLASF(dsymm,DSYMM,dsymm_,DSYMM_,dsymm__,DSYMM__)
#define ACX_SASUM LFC_BLASF(sasum,SASUM,sasum_,SASUM_,sasum__,SASUM__)
#define ACX_STRSV LFC_BLASF(strsv,STRSV,strsv_,STRSV_,strsv__,STRSV__)
#define ACX_SGBMV LFC_BLASF(sgbmv,SGBMV,sgbmv_,SGBMV_,sgbmv__,SGBMV__)
#define ACX_STBSV LFC_BLASF(stbsv,STBSV,stbsv_,STBSV_,stbsv__,STBSV__)
#define ACX_SSYR2K LFC_BLASF(ssyr2k,SSYR2K,ssyr2k_,SSYR2K_,ssyr2k__,SSYR2K__)
#define ACX_SGER LFC_BLASF(sger,SGER,sger_,SGER_,sger__,SGER__)
#define ACX_SROT LFC_BLASF(srot,SROT,srot_,SROT_,srot__,SROT__)
#define ACX_SSYMV LFC_BLASF(ssymv,SSYMV,ssymv_,SSYMV_,ssymv__,SSYMV__)
#define ACX_SSYR LFC_BLASF(ssyr,SSYR,ssyr_,SSYR_,ssyr__,SSYR__)
#define ACX_SSYR2 LFC_BLASF(ssyr2,SSYR2,ssyr2_,SSYR2_,ssyr2__,SSYR2__)
#define ACX_SSBMV LFC_BLASF(ssbmv,SSBMV,ssbmv_,SSBMV_,ssbmv__,SSBMV__)
#define ACX_SAXPY LFC_BLASF(saxpy,SAXPY,saxpy_,SAXPY_,saxpy__,SAXPY__)
#define ACX_SCOPY LFC_BLASF(scopy,SCOPY,scopy_,SCOPY_,scopy__,SCOPY__)
#define ACX_SDOT LFC_BLASF(sdot,SDOT,sdot_,SDOT_,sdot__,SDOT__)
#define ACX_SNRM2 LFC_BLASF(snrm2,SNRM2,snrm2_,SNRM2_,snrm2__,SNRM2__)
#define ACX_SSCAL LFC_BLASF(sscal,SSCAL,sscal_,SSCAL_,sscal__,SSCAL__)
#define ACX_SSWAP LFC_BLASF(sswap,SSWAP,sswap_,SSWAP_,sswap__,SSWAP__)
#define ACX_SGEMV LFC_BLASF(sgemv,SGEMV,sgemv_,SGEMV_,sgemv__,SGEMV__)
#define ACX_STRMV LFC_BLASF(strmv,STRMV,strmv_,STRMV_,strmv__,STRMV__)
#define ACX_SGEMM LFC_BLASF(sgemm,SGEMM,sgemm_,SGEMM_,sgemm__,SGEMM__)
#define ACX_STRMM LFC_BLASF(strmm,STRMM,strmm_,STRMM_,strmm__,STRMM__)
#define ACX_STRSM LFC_BLASF(strsm,STRSM,strsm_,STRSM_,strsm__,STRSM__)
#define ACX_SSYRK LFC_BLASF(ssyrk,SSYRK,ssyrk_,SSYRK_,ssyrk__,SSYRK__)
#define ACX_SSYMM LFC_BLASF(ssymm,SSYMM,ssymm_,SSYMM_,ssymm__,SSYMM__)
#define ACX_SCASUM LFC_BLASF(scasum,SCASUM,scasum_,SCASUM_,scasum__,SCASUM__)
#define ACX_CTRSV LFC_BLASF(ctrsv,CTRSV,ctrsv_,CTRSV_,ctrsv__,CTRSV__)
#define ACX_CGBMV LFC_BLASF(cgbmv,CGBMV,cgbmv_,CGBMV_,cgbmv__,CGBMV__)
#define ACX_CTBSV LFC_BLASF(ctbsv,CTBSV,ctbsv_,CTBSV_,ctbsv__,CTBSV__)
#define ACX_CSYR2K LFC_BLASF(csyr2k,CSYR2K,csyr2k_,CSYR2K_,csyr2k__,CSYR2K__)
#define ACX_CAXPY LFC_BLASF(caxpy,CAXPY,caxpy_,CAXPY_,caxpy__,CAXPY__)
#define ACX_CCOPY LFC_BLASF(ccopy,CCOPY,ccopy_,CCOPY_,ccopy__,CCOPY__)
#define ACX_SCNRM2 LFC_BLASF(scnrm2,SCNRM2,scnrm2_,SCNRM2_,scnrm2__,SCNRM2__)
#define ACX_CSCAL LFC_BLASF(cscal,CSCAL,cscal_,CSCAL_,cscal__,CSCAL__)
#define ACX_CSWAP LFC_BLASF(cswap,CSWAP,cswap_,CSWAP_,cswap__,CSWAP__)
#define ACX_CGEMV LFC_BLASF(cgemv,CGEMV,cgemv_,CGEMV_,cgemv__,CGEMV__)
#define ACX_CTRMV LFC_BLASF(ctrmv,CTRMV,ctrmv_,CTRMV_,ctrmv__,CTRMV__)
#define ACX_CGEMM LFC_BLASF(cgemm,CGEMM,cgemm_,CGEMM_,cgemm__,CGEMM__)
#define ACX_CTRMM LFC_BLASF(ctrmm,CTRMM,ctrmm_,CTRMM_,ctrmm__,CTRMM__)
#define ACX_CTRSM LFC_BLASF(ctrsm,CTRSM,ctrsm_,CTRSM_,ctrsm__,CTRSM__)
#define ACX_CSYRK LFC_BLASF(csyrk,CSYRK,csyrk_,CSYRK_,csyrk__,CSYRK__)
#define ACX_CSYMM LFC_BLASF(csymm,CSYMM,csymm_,CSYMM_,csymm__,CSYMM__)
#define ACX_CSSCAL LFC_BLASF(csscal,CSSCAL,csscal_,CSSCAL_,csscal__,CSSCAL__)
#define ACX_CGERU LFC_BLASF(cgeru,CGERU,cgeru_,CGERU_,cgeru__,CGERU__)
#define ACX_CGERC LFC_BLASF(cgerc,CGERC,cgerc_,CGERC_,cgerc__,CGERC__)
#define ACX_CHER LFC_BLASF(cher,CHER,cher_,CHER_,cher__,CHER__)
#define ACX_CHER2 LFC_BLASF(cher2,CHER2,cher2_,CHER2_,cher2__,CHER2__)
#define ACX_CHERK LFC_BLASF(cherk,CHERK,cherk_,CHERK_,cherk__,CHERK__)
#define ACX_CHER2K LFC_BLASF(cher2k,CHER2K,cher2k_,CHER2K_,cher2k__,CHER2K__)
#define ACX_CHEMV LFC_BLASF(chemv,CHEMV,chemv_,CHEMV_,chemv__,CHEMV__)
#define ACX_CHEMM LFC_BLASF(chemm,CHEMM,chemm_,CHEMM_,chemm__,CHEMM__)
#define ACX_CHBMV LFC_BLASF(chbmv,CHBMV,chbmv_,CHBMV_,chbmv__,CHBMV__)
#define ACX_DZASUM LFC_BLASF(dzasum,DZASUM,dzasum_,DZASUM_,dzasum__,DZASUM__)
#define ACX_ZTRSV LFC_BLASF(ztrsv,ZTRSV,ztrsv_,ZTRSV_,ztrsv__,ZTRSV__)
#define ACX_ZGBMV LFC_BLASF(zgbmv,ZGBMV,zgbmv_,ZGBMV_,zgbmv__,ZGBMV__)
#define ACX_ZTBSV LFC_BLASF(ztbsv,ZTBSV,ztbsv_,ZTBSV_,ztbsv__,ZTBSV__)
#define ACX_ZSYR2K LFC_BLASF(zsyr2k,ZSYR2K,zsyr2k_,ZSYR2K_,zsyr2k__,ZSYR2K__)
#define ACX_ZAXPY LFC_BLASF(zaxpy,ZAXPY,zaxpy_,ZAXPY_,zaxpy__,ZAXPY__)
#define ACX_ZCOPY LFC_BLASF(zcopy,ZCOPY,zcopy_,ZCOPY_,zcopy__,ZCOPY__)
#define ACX_DZNRM2 LFC_BLASF(dznrm2,DZNRM2,dznrm2_,DZNRM2_,dznrm2__,DZNRM2__)
#define ACX_ZSCAL LFC_BLASF(zscal,ZSCAL,zscal_,ZSCAL_,zscal__,ZSCAL__)
#define ACX_ZSWAP LFC_BLASF(zswap,ZSWAP,zswap_,ZSWAP_,zswap__,ZSWAP__)
#define ACX_ZGEMV LFC_BLASF(zgemv,ZGEMV,zgemv_,ZGEMV_,zgemv__,ZGEMV__)
#define ACX_ZTRMV LFC_BLASF(ztrmv,ZTRMV,ztrmv_,ZTRMV_,ztrmv__,ZTRMV__)
#define ACX_ZGEMM LFC_BLASF(zgemm,ZGEMM,zgemm_,ZGEMM_,zgemm__,ZGEMM__)
#define ACX_ZTRMM LFC_BLASF(ztrmm,ZTRMM,ztrmm_,ZTRMM_,ztrmm__,ZTRMM__)
#define ACX_ZTRSM LFC_BLASF(ztrsm,ZTRSM,ztrsm_,ZTRSM_,ztrsm__,ZTRSM__)
#define ACX_ZSYRK LFC_BLASF(zsyrk,ZSYRK,zsyrk_,ZSYRK_,zsyrk__,ZSYRK__)
#define ACX_ZSYMM LFC_BLASF(zsymm,ZSYMM,zsymm_,ZSYMM_,zsymm__,ZSYMM__)
#define ACX_ZDSCAL LFC_BLASF(zdscal,ZDSCAL,zdscal_,ZDSCAL_,zdscal__,ZDSCAL__)
#define ACX_ZGERU LFC_BLASF(zgeru,ZGERU,zgeru_,ZGERU_,zgeru__,ZGERU__)
#define ACX_ZGERC LFC_BLASF(zgerc,ZGERC,zgerc_,ZGERC_,zgerc__,ZGERC__)
#define ACX_ZHER LFC_BLASF(zher,ZHER,zher_,ZHER_,zher__,ZHER__)
#define ACX_ZHER2 LFC_BLASF(zher2,ZHER2,zher2_,ZHER2_,zher2__,ZHER2__)
#define ACX_ZHERK LFC_BLASF(zherk,ZHERK,zherk_,ZHERK_,zherk__,ZHERK__)
#define ACX_ZHER2K LFC_BLASF(zher2k,ZHER2K,zher2k_,ZHER2K_,zher2k__,ZHER2K__)
#define ACX_ZHEMV LFC_BLASF(zhemv,ZHEMV,zhemv_,ZHEMV_,zhemv__,ZHEMV__)
#define ACX_ZHEMM LFC_BLASF(zhemm,ZHEMM,zhemm_,ZHEMM_,zhemm__,ZHEMM__)
#define ACX_ZHBMV LFC_BLASF(zhbmv,ZHBMV,zhbmv_,ZHBMV_,zhbmv__,ZHBMV__)
#define ACX_IDAMAX LFC_BLASF(idamax,IDAMAX,idamax_,IDAMAX_,idamax__,IDAMAX__)
#define ACX_ISAMAX LFC_BLASF(isamax,ISAMAX,isamax_,ISAMAX_,isamax__,ISAMAX__)
#define ACX_ICAMAX LFC_BLASF(icamax,ICAMAX,icamax_,ICAMAX_,icamax__,ICAMAX__)
#define ACX_IZAMAX LFC_BLASF(izamax,IZAMAX,izamax_,IZAMAX_,izamax__,IZAMAX__)

#ifdef LFC_BLAS_FNAME_ONE_
#define LFC_BLASF(l,U,l_,U_,l__,U__) LFC_BLASFC(l_,U_)
#endif
#ifdef LFC_BLAS_FNAME_NO_
#define LFC_BLASF(l,U,l_,U_,l__,U__) LFC_BLASFC(l,U)
#endif

#ifdef LFC_BLAS_FNAME_LOWER
#define LFC_BLASFC(l,U) l
#endif
#ifdef LFC_BLAS_FNAME_UPPER
#define LFC_BLASFC(l,U) U
#endif

#define f77DASUM ACX_DASUM
#define f77DTRSV ACX_DTRSV
#define f77DGBMV ACX_DGBMV
#define f77DTBSV ACX_DTBSV
#define f77DSYR2K ACX_DSYR2K
#define f77DGER ACX_DGER
#define f77DROT ACX_DROT
#define f77DSYMV ACX_DSYMV
#define f77DSYR ACX_DSYR
#define f77DSYR2 ACX_DSYR2
#define f77DSBMV ACX_DSBMV
#define f77DAXPY ACX_DAXPY
#define f77DCOPY ACX_DCOPY
#define f77DDOT ACX_DDOT
#define f77DNRM2 ACX_DNRM2
#define f77DSCAL ACX_DSCAL
#define f77DSWAP ACX_DSWAP
#define f77DGEMV ACX_DGEMV
#define f77DTRMV ACX_DTRMV
#define f77DGEMM ACX_DGEMM
#define f77DTRMM ACX_DTRMM
#define f77DTRSM ACX_DTRSM
#define f77DSYRK ACX_DSYRK
#define f77DSYMM ACX_DSYMM
#define f77SASUM ACX_SASUM
#define f77STRSV ACX_STRSV
#define f77SGBMV ACX_SGBMV
#define f77STBSV ACX_STBSV
#define f77SSYR2K ACX_SSYR2K
#define f77SGER ACX_SGER
#define f77SROT ACX_SROT
#define f77SSYMV ACX_SSYMV
#define f77SSYR ACX_SSYR
#define f77SSYR2 ACX_SSYR2
#define f77SSBMV ACX_SSBMV
#define f77SAXPY ACX_SAXPY
#define f77SCOPY ACX_SCOPY
#define f77SDOT ACX_SDOT
#define f77SNRM2 ACX_SNRM2
#define f77SSCAL ACX_SSCAL
#define f77SSWAP ACX_SSWAP
#define f77SGEMV ACX_SGEMV
#define f77STRMV ACX_STRMV
#define f77SGEMM ACX_SGEMM
#define f77STRMM ACX_STRMM
#define f77STRSM ACX_STRSM
#define f77SSYRK ACX_SSYRK
#define f77SSYMM ACX_SSYMM
#define f77SCASUM ACX_SCASUM
#define f77CTRSV ACX_CTRSV
#define f77CGBMV ACX_CGBMV
#define f77CTBSV ACX_CTBSV
#define f77CSYR2K ACX_CSYR2K
#define f77CAXPY ACX_CAXPY
#define f77CCOPY ACX_CCOPY
#define f77SCNRM2 ACX_SCNRM2
#define f77CSCAL ACX_CSCAL
#define f77CSWAP ACX_CSWAP
#define f77CGEMV ACX_CGEMV
#define f77CTRMV ACX_CTRMV
#define f77CGEMM ACX_CGEMM
#define f77CTRMM ACX_CTRMM
#define f77CTRSM ACX_CTRSM
#define f77CSYRK ACX_CSYRK
#define f77CSYMM ACX_CSYMM
#define f77CSSCAL ACX_CSSCAL
#define f77CGERU ACX_CGERU
#define f77CGERC ACX_CGERC
#define f77CHER ACX_CHER
#define f77CHER2 ACX_CHER2
#define f77CHERK ACX_CHERK
#define f77CHER2K ACX_CHER2K
#define f77CHEMV ACX_CHEMV
#define f77CHEMM ACX_CHEMM
#define f77CHBMV ACX_CHBMV
#define f77DZASUM ACX_DZASUM
#define f77ZTRSV ACX_ZTRSV
#define f77ZGBMV ACX_ZGBMV
#define f77ZTBSV ACX_ZTBSV
#define f77ZSYR2K ACX_ZSYR2K
#define f77ZAXPY ACX_ZAXPY
#define f77ZCOPY ACX_ZCOPY
#define f77DZNRM2 ACX_DZNRM2
#define f77ZSCAL ACX_ZSCAL
#define f77ZSWAP ACX_ZSWAP
#define f77ZGEMV ACX_ZGEMV
#define f77ZTRMV ACX_ZTRMV
#define f77ZGEMM ACX_ZGEMM
#define f77ZTRMM ACX_ZTRMM
#define f77ZTRSM ACX_ZTRSM
#define f77ZSYRK ACX_ZSYRK
#define f77ZSYMM ACX_ZSYMM
#define f77ZDSCAL ACX_ZDSCAL
#define f77ZGERU ACX_ZGERU
#define f77ZGERC ACX_ZGERC
#define f77ZHER ACX_ZHER
#define f77ZHER2 ACX_ZHER2
#define f77ZHERK ACX_ZHERK
#define f77ZHER2K ACX_ZHER2K
#define f77ZHEMV ACX_ZHEMV
#define f77ZHEMM ACX_ZHEMM
#define f77ZHBMV ACX_ZHBMV
#define f77IDAMAX ACX_IDAMAX
#define f77ISAMAX ACX_ISAMAX
#define f77ICAMAX ACX_ICAMAX
#define f77IZAMAX ACX_IZAMAX

/* FIXME: provide fall back routine */
#define f77assert(expr) if (!(expr)) exit(EXIT_FAILURE)

#define side2char(s) (LFC_CblasRight==(s)?'R':'L')
#define uplo2char(u) (LFC_CblasLower==(u)?'L':'U')
#define trans2char(t) (LFC_CblasConjTrans==(t)?'C':((LFC_CblasTrans==(t))?'T':'N'))
#define diag2char(u) (LFC_CblasNonUnit==(u)?'N':'U')

LFC_BEGIN_C_DECLS

#if ! defined(HAVE_ATLAS) && ! defined(HAVE_CBLAS) && ! defined(USE_LFC_BLAS)
static void
#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
f77XERBLA(char *srname, BLAS_INTEGER *uinfo, int len) {
  char nm[7]; int i, info = 0;
  if (uinfo) info = *uinfo;
  for (i = 0; i < 6; i++) nm[i] = srname[i];
  nm[i] = '\0';
#else
f77XERBLA() {
  char nm[] = "NONE"; int info = -1;
#endif
  fprintf( stderr,
	   " ** On entry to %s parameter number %2d had an illegal value",
	   nm, info );
  fflush( stderr );
}

static void 
xerbla_impl(char *srname, int *uinfo) {
  char nm[7]; int i, info = 0;
  if (uinfo) info = *uinfo;
  for (i = 0; i < 6; i++) nm[i] = srname[i];
  nm[i] = '\0';
  fprintf( stderr,
	   " ** On entry to %s parameter number %2d had an illegal value",
	   nm, info );
  fflush( stderr );
  exit( EXIT_FAILURE );
}

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
#define XRBLARGS (char*sr,int*info,int l)
#define XRBLBODY {xerbla_impl(sr,info);}
#else 
#define XRBLARGS ()
#define XRBLBODY {char nm[8] = "UNKNOWN";int info=-32768; xerbla_impl(sr,&info);}
#endif 

/* This comes from the LAPACK library */ 
/*void xerbla_  XRBLARGS XRBLBODY*/

void xerbla   XRBLARGS XRBLBODY
void xerbla__ XRBLARGS XRBLBODY
void XERBLA   XRBLARGS XRBLBODY
void XERBLA_  XRBLARGS XRBLBODY
void XERBLA__ XRBLARGS XRBLBODY

void
lfc_cblas_saxpy(int N, float alpha, /*const*/ float *X,int incX, float *Y,
		int incY) {
  BLAS_INTEGER n = N, incx = incX, incy = incY;
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  f77SAXPY( &n, &alpha, X, &incx, Y, &incy );
}

void
lfc_cblas_scopy(int N, /*const*/ float *X, int incX, float *Y, int incY) {
  BLAS_INTEGER n = N, incx = incX, incy = incY;
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  f77SCOPY( &n, X, &incx, Y, &incy );
}

#if 0
float
lfc_cblas_sdot(int N, /*const*/ float *X, int incX, /*const*/ float *Y,
	       int incY) {
  BLAS_INTEGER n = N, incx = incX, incy = incY;
  BLAS_REAL rv;
  extern BLAS_REAL f77SDOT(BLAS_INTEGER *, BLAS_REAL *,
    BLAS_INTEGER *, BLAS_REAL *, BLAS_INTEGER *);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  rv = f77SDOT( &n, X, &incx, Y, &incy );
  return rv;
}
#endif

void
lfc_cblas_srot(/*const*/ int N, float *X, /*const*/ int incX,
                float *Y, /*const*/ int incY, /*const*/ float c, /*const*/ float s) {
  BLAS_INTEGER n = N, incx = incX, incy = incY;
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  f77SROT( &n, X, &incx, Y, &incy, &c, &s );
}

void
lfc_cblas_sger(enum LFC_CBLAS_ORDER Order, int M, int N, float alpha,
	       /*const*/ float *X, int incX, float *Y, int incY, float *A,
	       int ldA) {
  BLAS_INTEGER m = M, n = N, incx = incX, incy = incY, lda = ldA;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  f77SGER( &m, &n, &alpha, X, &incx, Y, &incy, A, &lda );
}

float
lfc_cblas_snrm2(int N, float *X, int incX) {
  BLAS_INTEGER n = N, incx = incX;
  BLAS_REAL rv;
  extern BLAS_REAL f77SNRM2(BLAS_INTEGER *, BLAS_REAL *,
    BLAS_INTEGER *);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  rv = f77SNRM2( &n, X, &incx );
  return rv;
}

float
lfc_cblas_sasum(/*const*/ int N, /*const*/ float *X, /*const*/ int incX) {
  BLAS_INTEGER n = N, incx = incX;
  BLAS_REAL rv;
  extern BLAS_REAL f77SASUM(BLAS_INTEGER *, BLAS_REAL *,
    BLAS_INTEGER *);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  rv = f77SASUM( &n, X, &incx );
  return rv;
}

void
lfc_cblas_sscal(int N, float alpha, float *X, int incX) {
  BLAS_INTEGER n = N, incx = incX;
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  f77SSCAL( &n, &alpha, X, &incx );
}

void
lfc_cblas_sswap(int N, float *X, int incX, float *Y, int incY) {
  BLAS_INTEGER n = N, incx = incX, incy = incY;
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  f77SSWAP( &n, X, &incx, Y, &incy );
}

int
lfc_cblas_isamax(int N, /*const*/ float *X, int incX) {
  BLAS_INTEGER n = N, incx = incX, r;
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  r = f77ISAMAX( &n, X, &incx ) - 1;
  return r;
}

void lfc_cblas_ssyr(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                /*const*/ int N, /*const*/ float alpha, /*const*/ float *X,
                /*const*/ int incX, float *A, /*const*/ int ldA) {
  BLAS_INTEGER n = N, incx = incX, lda = ldA;
  char uplo;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  f77SSYR( &uplo, &n, &alpha, X, &incx, A, &lda, 1 );
#else
  f77assert(0);
#endif
}

void lfc_cblas_ssyr2(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                /*const*/ int N, /*const*/ float alpha, /*const*/ float *X,
                /*const*/ int incX, /*const*/ float *Y, /*const*/ int incY, float *A,
                /*const*/ int ldA) {
  BLAS_INTEGER n = N, incx = incX, incy = incY, lda = ldA;
  char uplo;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  f77SSYR2( &uplo, &n, &alpha, X, &incx, Y, &incy, A, &lda, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_ssymv(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                /*const*/ int N, /*const*/ float alpha, /*const*/ float *A,
                /*const*/ int ldA, /*const*/ float *X, /*const*/ int incX,
                /*const*/ float beta, float *Y, /*const*/ int incY) {
  BLAS_INTEGER n = N, lda = ldA, incx = incX, incy = incY;
  char uplo;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  f77SSYMV( &uplo, &n, &alpha, A, &lda, X, &incx, &beta, Y, &incy, 1 );
#else
  f77assert(0);
#endif
}

void lfc_cblas_ssbmv(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                 /*const*/ int N, /*const*/ int K, /*const*/ float alpha, /*const*/ float *A,
                 /*const*/ int ldA, /*const*/ float *X, /*const*/ int incX,
                 /*const*/ float beta, float *Y, /*const*/ int incY) {
  BLAS_INTEGER n = N, k = K, lda = ldA, incx = incX, incy = incY;
  char uplo;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  f77SSBMV( &uplo, &n, &k, &alpha, A, &lda, X, &incx, &beta, Y, &incy, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_sgbmv(/*const*/ enum LFC_CBLAS_ORDER Order,
                 /*const*/ enum LFC_CBLAS_TRANSPOSE TransA, /*const*/ int M, /*const*/ int N,
                 /*const*/ int KL, /*const*/ int KU, /*const*/ float alpha,
                 /*const*/ float *A, /*const*/ int ldA, /*const*/ float *X,
                 /*const*/ int incX, /*const*/ float beta, float *Y, /*const*/ int incY) {
  BLAS_INTEGER m = M, n = N, kl = KL, ku = KU, lda = ldA, incx = incX, incy = incY;
  char ta;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  ta = trans2char(TransA);
  f77SGBMV( &ta, &m, &n, &kl, &ku, &alpha, A, &lda, X, &incx, &beta, Y, &incy, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_sgemv(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_TRANSPOSE TransA,
		int M, int N, float alpha, /*const*/ float  *A, int ldA,
		/*const*/ float *X, int incX, float beta, float  *Y,
		int incY) {
  BLAS_INTEGER m = M, n = N, lda = ldA, incx = incX, incy = incY;
  char ta;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  ta = trans2char(TransA);
  f77SGEMV( &ta, &m, &n, &alpha, A, &lda, X, &incx, &beta, Y, &incy, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_ssyr2k(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                  /*const*/ enum LFC_CBLAS_TRANSPOSE Trans, /*const*/ int N, /*const*/ int K,
                  /*const*/ float alpha, /*const*/ float *A, /*const*/ int ldA,
                  /*const*/ float *B, /*const*/ int ldB, /*const*/ float beta,
                  float *C, /*const*/ int ldC) {
  BLAS_INTEGER n = N, k = K, lda = ldA, ldb = ldB, ldc = ldC;
  char uplo, trans;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(Trans);
  f77SSYR2K( &uplo, &trans, &n, &k, &alpha, A, &lda, B, &ldb, &beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_stbsv(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                 /*const*/ enum LFC_CBLAS_TRANSPOSE TransA, /*const*/ enum LFC_CBLAS_DIAG Diag,
                 /*const*/ int N, /*const*/ int K, /*const*/ float *A, /*const*/ int ldA,
                 float *X, /*const*/ int incX) {
  BLAS_INTEGER n = N, k = K, lda = ldA, incx = incX;
  char uplo, trans, diag;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77STBSV( &uplo, &trans, &diag, &n, &k, A, &lda, X, &incx, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_strsv(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                 /*const*/ enum LFC_CBLAS_TRANSPOSE TransA, /*const*/ enum LFC_CBLAS_DIAG Diag,
                 /*const*/ int N, /*const*/ float *A, /*const*/ int ldA, float *X,
                 /*const*/ int incX) {
  BLAS_INTEGER n = N, lda = ldA, incx = incX;
  char uplo, trans, diag;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77STRSV( &uplo, &trans, &diag, &n, A, &lda, X, &incx, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void lfc_cblas_strmv(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_UPLO Uplo,
  enum LFC_CBLAS_TRANSPOSE TransA, enum LFC_CBLAS_DIAG Diag, int N,
  /* const */ float *A, int ldA, float *X, int incX) {
  BLAS_INTEGER n = N, lda = ldA, incx = incX;
  char uplo, trans, diag;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77STRMV( &uplo, &trans, &diag, &n, A, &lda, X, &incx, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_sgemm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_TRANSPOSE TransA,
                enum LFC_CBLAS_TRANSPOSE TransB, int M, int N, int K,
                float alpha, /*const*/ float *A, int ldA, /*const*/ float *B,
                int ldB, float beta, float *C, int ldC) {
  BLAS_INTEGER m = M, n = N, k = K, lda = ldA, ldb = ldB, ldc = ldC;
  char ta, tb;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  ta = trans2char(TransA);
  tb = trans2char(TransB);
  f77SGEMM( &ta, &tb, &m, &n, &k, &alpha, A, &lda, B, &ldb, &beta, C, &ldc,
            1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_strmm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_SIDE Side,
                enum LFC_CBLAS_UPLO Uplo, enum LFC_CBLAS_TRANSPOSE TransA,
                enum LFC_CBLAS_DIAG Diag, int M, int N, float alpha,
                /*const*/ float *A, int ldA, float *B, int ldB) {
  BLAS_INTEGER m = M, n = N, lda = ldA, ldb = ldB;
  char side, uplo, trans, diag;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  side = side2char(Side);
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77STRMM( &side, &uplo, &trans, &diag, &m, &n, &alpha, A, &lda, B, &ldb,
            1, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_strsm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_SIDE Side,
                enum LFC_CBLAS_UPLO Uplo, enum LFC_CBLAS_TRANSPOSE TransA,
                enum LFC_CBLAS_DIAG Diag, int M, int N, float alpha,
                /*const*/ float *A, int ldA, float *B, int ldB) {
  BLAS_INTEGER m = M, n = N, lda = ldA, ldb = ldB;
  char side, uplo, trans, diag;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  side = side2char(Side);
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77STRSM( &side, &uplo, &trans, &diag, &m, &n, &alpha, A, &lda, B, &ldb,
            1, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_ssyrk(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_UPLO Uplo,
		enum LFC_CBLAS_TRANSPOSE Trans, int N, int K, float alpha,
		/*const*/ float *A, int ldA, float beta, float *C, int ldC){
  BLAS_INTEGER n = N, k = K, lda = ldA, ldc = ldC;
  char uplo, trans;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(Trans);
  f77SSYRK( &uplo, &trans, &n, &k, &alpha, A, &lda, &beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_ssymm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_SIDE Side,
		enum LFC_CBLAS_UPLO Uplo, int M, int N, float alpha,
		/*const*/ float *A, int ldA,  /*const*/ float *B, int ldB,
		float beta, float *C, int ldC) {
  BLAS_INTEGER n = N, m = M, lda = ldA, ldb = ldB, ldc = ldC;
  char side, uplo;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  side = side2char(Side);
  uplo = uplo2char(Uplo);
  f77SSYMM( &side, &uplo, &m, &n, &alpha, A, &lda, B, &ldb, &beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_daxpy(int N, double alpha, /*const*/ double *X,int incX, double *Y,
		int incY) {
  BLAS_INTEGER n = N, incx = incX, incy = incY;
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  f77DAXPY( &n, &alpha, X, &incx, Y, &incy );
}

void
lfc_cblas_dcopy(int N, /*const*/ double *X, int incX, double *Y, int incY) {
  BLAS_INTEGER n = N, incx = incX, incy = incY;
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  f77DCOPY( &n, X, &incx, Y, &incy );
}

#if 0
double
lfc_cblas_ddot(int N, /*const*/ double *X, int incX, /*const*/ double *Y,
	       int incY) {
  BLAS_INTEGER n = N, incx = incX, incy = incY;
  BLAS_DOUBLE_PRECISION rv;
  extern BLAS_DOUBLE_PRECISION f77DDOT(BLAS_INTEGER *, BLAS_DOUBLE_PRECISION *,
    BLAS_INTEGER *, BLAS_DOUBLE_PRECISION *, BLAS_INTEGER *);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  rv = f77DDOT( &n, X, &incx, Y, &incy );
  return rv;
}
#endif

void
lfc_cblas_drot(/*const*/ int N, double *X, /*const*/ int incX,
                double *Y, /*const*/ int incY, /*const*/ double c, /*const*/ double s) {
  BLAS_INTEGER n = N, incx = incX, incy = incY;
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  f77DROT( &n, X, &incx, Y, &incy, &c, &s );
}

void
lfc_cblas_dger(enum LFC_CBLAS_ORDER Order, int M, int N, double alpha,
	       /*const*/ double *X, int incX, double *Y, int incY, double *A,
	       int ldA) {
  BLAS_INTEGER m = M, n = N, incx = incX, incy = incY, lda = ldA;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  f77DGER( &m, &n, &alpha, X, &incx, Y, &incy, A, &lda );
}

double
lfc_cblas_dnrm2(int N, double *X, int incX) {
  BLAS_INTEGER n = N, incx = incX;
  BLAS_DOUBLE_PRECISION rv;
  extern BLAS_DOUBLE_PRECISION f77DNRM2(BLAS_INTEGER *, BLAS_DOUBLE_PRECISION *,
    BLAS_INTEGER *);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  rv = f77DNRM2( &n, X, &incx );
  return rv;
}

double
lfc_cblas_dasum(/*const*/ int N, /*const*/ double *X, /*const*/ int incX) {
  BLAS_INTEGER n = N, incx = incX;
  BLAS_DOUBLE_PRECISION rv;
  extern BLAS_DOUBLE_PRECISION f77DASUM(BLAS_INTEGER *, BLAS_DOUBLE_PRECISION *,
    BLAS_INTEGER *);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  rv = f77DASUM( &n, X, &incx );
  return rv;
}

void
lfc_cblas_dscal(int N, double alpha, double *X, int incX) {
  BLAS_INTEGER n = N, incx = incX;
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  f77DSCAL( &n, &alpha, X, &incx );
}

void
lfc_cblas_dswap(int N, double *X, int incX, double *Y, int incY) {
  BLAS_INTEGER n = N, incx = incX, incy = incY;
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  f77DSWAP( &n, X, &incx, Y, &incy );
}

int
lfc_cblas_idamax(int N, /*const*/ double *X, int incX) {
  BLAS_INTEGER n = N, incx = incX, r;
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  r = f77IDAMAX( &n, X, &incx ) - 1;
  return r;
}

void lfc_cblas_dsyr(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                /*const*/ int N, /*const*/ double alpha, /*const*/ double *X,
                /*const*/ int incX, double *A, /*const*/ int ldA) {
  BLAS_INTEGER n = N, incx = incX, lda = ldA;
  char uplo;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  f77DSYR( &uplo, &n, &alpha, X, &incx, A, &lda, 1 );
#else
  f77assert(0);
#endif
}

void lfc_cblas_dsyr2(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                /*const*/ int N, /*const*/ double alpha, /*const*/ double *X,
                /*const*/ int incX, /*const*/ double *Y, /*const*/ int incY, double *A,
                /*const*/ int ldA) {
  BLAS_INTEGER n = N, incx = incX, incy = incY, lda = ldA;
  char uplo;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  f77DSYR2( &uplo, &n, &alpha, X, &incx, Y, &incy, A, &lda, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_dsymv(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                /*const*/ int N, /*const*/ double alpha, /*const*/ double *A,
                /*const*/ int ldA, /*const*/ double *X, /*const*/ int incX,
                /*const*/ double beta, double *Y, /*const*/ int incY) {
  BLAS_INTEGER n = N, lda = ldA, incx = incX, incy = incY;
  char uplo;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  f77DSYMV( &uplo, &n, &alpha, A, &lda, X, &incx, &beta, Y, &incy, 1, 1 );
#else
  f77assert(0);
#endif
}

void lfc_cblas_dsbmv(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                 /*const*/ int N, /*const*/ int K, /*const*/ double alpha, /*const*/ double *A,
                 /*const*/ int ldA, /*const*/ double *X, /*const*/ int incX,
                 /*const*/ double beta, double *Y, /*const*/ int incY) {
  BLAS_INTEGER n = N, k = K, lda = ldA, incx = incX, incy = incY;
  char uplo;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  f77DSBMV( &uplo, &n, &k, &alpha, A, &lda, X, &incx, &beta, Y, &incy, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_dgbmv(/*const*/ enum LFC_CBLAS_ORDER Order,
                 /*const*/ enum LFC_CBLAS_TRANSPOSE TransA, /*const*/ int M, /*const*/ int N,
                 /*const*/ int KL, /*const*/ int KU, /*const*/ double alpha,
                 /*const*/ double *A, /*const*/ int ldA, /*const*/ double *X,
                 /*const*/ int incX, /*const*/ double beta, double *Y, /*const*/ int incY) {
  BLAS_INTEGER m = M, n = N, kl = KL, ku = KU, lda = ldA, incx = incX, incy = incY;
  char ta;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  ta = trans2char(TransA);
  f77DGBMV( &ta, &m, &n, &kl, &ku, &alpha, A, &lda, X, &incx, &beta, Y, &incy, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_dgemv(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_TRANSPOSE TransA,
		int M, int N, double alpha, /*const*/ double  *A, int ldA,
		/*const*/ double *X, int incX, double beta, double  *Y,
		int incY) {
  BLAS_INTEGER m = M, n = N, lda = ldA, incx = incX, incy = incY;
  char ta;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  ta = trans2char(TransA);
  f77DGEMV( &ta, &m, &n, &alpha, A, &lda, X, &incx, &beta, Y, &incy, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_dsyr2k(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                  /*const*/ enum LFC_CBLAS_TRANSPOSE Trans, /*const*/ int N, /*const*/ int K,
                  /*const*/ double alpha, /*const*/ double *A, /*const*/ int ldA,
                  /*const*/ double *B, /*const*/ int ldB, /*const*/ double beta,
                  double *C, /*const*/ int ldC) {
  BLAS_INTEGER n = N, k = K, lda = ldA, ldb = ldB, ldc = ldC;
  char uplo, trans;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(Trans);
  f77DSYR2K( &uplo, &trans, &n, &k, &alpha, A, &lda, B, &ldb, &beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_dtbsv(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                 /*const*/ enum LFC_CBLAS_TRANSPOSE TransA, /*const*/ enum LFC_CBLAS_DIAG Diag,
                 /*const*/ int N, /*const*/ int K, /*const*/ double *A, /*const*/ int ldA,
                 double *X, /*const*/ int incX) {
  BLAS_INTEGER n = N, k = K, lda = ldA, incx = incX;
  char uplo, trans, diag;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77DTBSV( &uplo, &trans, &diag, &n, &k, A, &lda, X, &incx, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_dtrsv(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                 /*const*/ enum LFC_CBLAS_TRANSPOSE TransA, /*const*/ enum LFC_CBLAS_DIAG Diag,
                 /*const*/ int N, /*const*/ double *A, /*const*/ int ldA, double *X,
                 /*const*/ int incX) {
  BLAS_INTEGER n = N, lda = ldA, incx = incX;
  char uplo, trans, diag;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77DTRSV( &uplo, &trans, &diag, &n, A, &lda, X, &incx, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void lfc_cblas_dtrmv(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_UPLO Uplo,
  enum LFC_CBLAS_TRANSPOSE TransA, enum LFC_CBLAS_DIAG Diag, int N,
  /* const */ double *A, int ldA, double *X, int incX) {
  BLAS_INTEGER n = N, lda = ldA, incx = incX;
  char uplo, trans, diag;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77DTRMV( &uplo, &trans, &diag, &n, A, &lda, X, &incx, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_dgemm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_TRANSPOSE TransA,
                enum LFC_CBLAS_TRANSPOSE TransB, int M, int N, int K,
                double alpha, /*const*/ double *A, int ldA, /*const*/ double *B,
                int ldB, double beta, double *C, int ldC) {
  BLAS_INTEGER m = M, n = N, k = K, lda = ldA, ldb = ldB, ldc = ldC;
  char ta, tb;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  ta = trans2char(TransA);
  tb = trans2char(TransB);
  f77DGEMM( &ta, &tb, &m, &n, &k, &alpha, A, &lda, B, &ldb, &beta, C, &ldc,
            1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_dtrmm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_SIDE Side,
                enum LFC_CBLAS_UPLO Uplo, enum LFC_CBLAS_TRANSPOSE TransA,
                enum LFC_CBLAS_DIAG Diag, int M, int N, double alpha,
                /*const*/ double *A, int ldA, double *B, int ldB) {
  BLAS_INTEGER m = M, n = N, lda = ldA, ldb = ldB;
  char side, uplo, trans, diag;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  side = side2char(Side);
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77DTRMM( &side, &uplo, &trans, &diag, &m, &n, &alpha, A, &lda, B, &ldb,
            1, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_dtrsm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_SIDE Side,
                enum LFC_CBLAS_UPLO Uplo, enum LFC_CBLAS_TRANSPOSE TransA,
                enum LFC_CBLAS_DIAG Diag, int M, int N, double alpha,
                /*const*/ double *A, int ldA, double *B, int ldB) {
  BLAS_INTEGER m = M, n = N, lda = ldA, ldb = ldB;
  char side, uplo, trans, diag;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  side = side2char(Side);
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77DTRSM( &side, &uplo, &trans, &diag, &m, &n, &alpha, A, &lda, B, &ldb,
            1, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_dsyrk(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_UPLO Uplo,
		enum LFC_CBLAS_TRANSPOSE Trans, int N, int K, double alpha,
		/*const*/ double *A, int ldA, double beta, double *C, int ldC){
  BLAS_INTEGER n = N, k = K, lda = ldA, ldc = ldC;
  char uplo, trans;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(Trans);
  f77DSYRK( &uplo, &trans, &n, &k, &alpha, A, &lda, &beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_dsymm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_SIDE Side,
		enum LFC_CBLAS_UPLO Uplo, int M, int N, double alpha,
		/*const*/ double *A, int ldA,  /*const*/ double *B, int ldB,
		double beta, double *C, int ldC) {
  BLAS_INTEGER n = N, m = M, lda = ldA, ldb = ldB, ldc = ldC;
  char side, uplo;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  side = side2char(Side);
  uplo = uplo2char(Uplo);
  f77DSYMM( &side, &uplo, &m, &n, &alpha, A, &lda, B, &ldb, &beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

/* */ 
void
lfc_cblas_caxpy(int N, void *alpha, /*const*/ void *X,int incX, void *Y,
		int incY) {
  BLAS_INTEGER n = N, incx = incX, incy = incY;
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  f77CAXPY( &n, alpha, X, &incx, Y, &incy );
}

void
lfc_cblas_ccopy(int N, /*const*/ void *X, int incX, void *Y, int incY) {
  BLAS_INTEGER n = N, incx = incX, incy = incY;
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  f77CCOPY( &n, X, &incx, Y, &incy );
}

void
lfc_cblas_cgerc(enum LFC_CBLAS_ORDER Order, int M, int N, void *alpha,
	       /*const*/ void *X, int incX, void *Y, int incY, void *A,
	       int ldA) {
  BLAS_INTEGER m = M, n = N, incx = incX, incy = incY, lda = ldA;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  f77CGERC( &m, &n, alpha, X, &incx, Y, &incy, A, &lda );
}

void
lfc_cblas_cgeru(enum LFC_CBLAS_ORDER Order, int M, int N, void *alpha,
	       /*const*/ void *X, int incX, void *Y, int incY, void *A,
	       int ldA) {
  BLAS_INTEGER m = M, n = N, incx = incX, incy = incY, lda = ldA;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  f77CGERU( &m, &n, alpha, X, &incx, Y, &incy, A, &lda );
}

float
lfc_cblas_scnrm2(int N, void *X, int incX) {
  BLAS_INTEGER n = N, incx = incX;
  BLAS_REAL rv;
  extern BLAS_REAL f77SCNRM2(BLAS_INTEGER *, void *, BLAS_INTEGER *);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  rv = f77SCNRM2( &n, X, &incx );
  return rv;
}

float
lfc_cblas_scasum(/*const*/ int N, /*const*/ void *X, /*const*/ int incX) {
  BLAS_INTEGER n = N, incx = incX;
  BLAS_REAL rv;
  extern BLAS_REAL f77CASUM(BLAS_INTEGER *, void *, BLAS_INTEGER *);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  rv = f77SCASUM( &n, X, &incx );
  return rv;
}

void
lfc_cblas_cscal(int N, void *alpha, void *X, int incX) {
  BLAS_INTEGER n = N, incx = incX;
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  f77CSCAL( &n, alpha, X, &incx );
}

void
lfc_cblas_csscal(int N, float alpha, void *X, int incX) {
  BLAS_INTEGER n = N, incx = incX;
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  f77CSCAL( &n, &alpha, X, &incx );
}

void
lfc_cblas_cswap(int N, void *X, int incX, void *Y, int incY) {
  BLAS_INTEGER n = N, incx = incX, incy = incY;
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  f77CSWAP( &n, X, &incx, Y, &incy );
}

int
lfc_cblas_icamax(int N, /*const*/ void *X, int incX) {
  BLAS_INTEGER n = N, incx = incX, r;
  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  r = f77ICAMAX( &n, X, &incx ) - 1;
  return r;
}

void
lfc_cblas_cher(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                /*const*/ int N, /*const*/ float alpha, /*const*/ void *X,
                /*const*/ int incX, void *A, /*const*/ int ldA) {
  BLAS_INTEGER n = N, incx = incX, lda = ldA;
  char uplo;

  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  f77assert(Order == LFC_CblasColMajor);
#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  f77CHER( &uplo, &n, &alpha, X, &incx, A, &lda, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_cher2(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                /*const*/ int N, /*const*/ void *alpha, /*const*/ void *X,
                /*const*/ int incX, /*const*/ void *Y, /*const*/ int incY, void *A,
                /*const*/ int ldA) {
  BLAS_INTEGER n = N, incx = incX, incy = incY, lda = ldA;
  char uplo;

  f77assert(sizeof(BLAS_REAL)==sizeof(float));
  f77assert(Order == LFC_CblasColMajor);
#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  f77CHER2( &uplo, &n, alpha, X, &incx, Y, &incy, A, &lda, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_chemv(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                /*const*/ int N, /*const*/ void *alpha, /*const*/ void *A,
                /*const*/ int ldA, /*const*/ void *X, /*const*/ int incX,
                /*const*/ void *beta, void *Y, /*const*/ int incY) {
  BLAS_INTEGER n = N, lda = ldA, incx = incX, incy = incY;
  char uplo;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  f77CHEMV( &uplo, &n, alpha, A, &lda, X, &incx, beta, Y, &incy, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_chbmv(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                 /*const*/ int N, /*const*/ int K, /*const*/ void *alpha, /*const*/ void *A,
                 /*const*/ int ldA, /*const*/ void *X, /*const*/ int incX,
                 /*const*/ void *beta, void *Y, /*const*/ int incY) {
  BLAS_INTEGER n = N, k = K, lda = ldA, incx = incX, incy = incY;
  char uplo;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  f77CHBMV( &uplo, &n, &k, alpha, A, &lda, X, &incx, beta, Y, &incy, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_cgbmv(/*const*/ enum LFC_CBLAS_ORDER Order,
                 /*const*/ enum LFC_CBLAS_TRANSPOSE TransA, /*const*/ int M, /*const*/ int N,
                 /*const*/ int KL, /*const*/ int KU, /*const*/ void *alpha,
                 /*const*/ void *A, /*const*/ int ldA, /*const*/ void *X,
                 /*const*/ int incX, /*const*/ void *beta, void *Y, /*const*/ int incY) {
  BLAS_INTEGER m = M, n = N, kl = KL, ku = KU, lda = ldA, incx = incX, incy = incY;
  char ta;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  ta = trans2char(TransA);
  f77CGBMV( &ta, &m, &n, &kl, &ku, alpha, A, &lda, X, &incx, beta, Y, &incy, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_cgemv(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_TRANSPOSE TransA,
		int M, int N, void *alpha, /*const*/ void *A, int ldA,
		/*const*/ void *X, int incX, void *beta, void *Y,
		int incY) {
  BLAS_INTEGER m = M, n = N, lda = ldA, incx = incX, incy = incY;
  char ta;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  ta = trans2char(TransA);
  f77CGEMV( &ta, &m, &n, alpha, A, &lda, X, &incx, beta, Y, &incy, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_csyr2k(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                  /*const*/ enum LFC_CBLAS_TRANSPOSE Trans, /*const*/ int N, /*const*/ int K,
                  /*const*/ void *alpha, /*const*/ void *A, /*const*/ int ldA,
                  /*const*/ void *B, /*const*/ int ldB, /*const*/ void *beta,
                  void *C, /*const*/ int ldC) {
  BLAS_INTEGER n = N, k = K, lda = ldA, ldb = ldB, ldc = ldC;
  char uplo, trans;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(Trans);
  f77CSYR2K( &uplo, &trans, &n, &k, alpha, A, &lda, B, &ldb, beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_cher2k(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                  /*const*/ enum LFC_CBLAS_TRANSPOSE Trans, /*const*/ int N, /*const*/ int K,
                  /*const*/ void *alpha, /*const*/ void *A, /*const*/ int ldA,
                  /*const*/ void *B, /*const*/ int ldB, /*const*/ float beta,
                  void *C, /*const*/ int ldC) {
  BLAS_INTEGER n = N, k = K, lda = ldA, ldb = ldB, ldc = ldC;
  char uplo, trans;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(Trans);
  f77CHER2K( &uplo, &trans, &n, &k, alpha, A, &lda, B, &ldb, &beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_ctbsv(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                 /*const*/ enum LFC_CBLAS_TRANSPOSE TransA, /*const*/ enum LFC_CBLAS_DIAG Diag,
                 /*const*/ int N, /*const*/ int K, /*const*/ void *A, /*const*/ int ldA,
                 void *X, /*const*/ int incX) {
  BLAS_INTEGER n = N, k = K, lda = ldA, incx = incX;
  char uplo, trans, diag;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77CTBSV( &uplo, &trans, &diag, &n, &k, A, &lda, X, &incx, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_ctrsv(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                 /*const*/ enum LFC_CBLAS_TRANSPOSE TransA, /*const*/ enum LFC_CBLAS_DIAG Diag,
                 /*const*/ int N, /*const*/ void *A, /*const*/ int ldA, void *X,
                 /*const*/ int incX) {
  BLAS_INTEGER n = N, lda = ldA, incx = incX;
  char uplo, trans, diag;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77CTRSV( &uplo, &trans, &diag, &n, A, &lda, X, &incx, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void lfc_cblas_ctrmv(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_UPLO Uplo,
  enum LFC_CBLAS_TRANSPOSE TransA, enum LFC_CBLAS_DIAG Diag, int N,
  /* const */ void *A, int ldA, void *X, int incX) {
  BLAS_INTEGER n = N, lda = ldA, incx = incX;
  char uplo, trans, diag;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77CTRMV( &uplo, &trans, &diag, &n, A, &lda, X, &incx, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_cgemm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_TRANSPOSE TransA,
                enum LFC_CBLAS_TRANSPOSE TransB, int M, int N, int K,
                void *alpha, /*const*/ void *A, int ldA, /*const*/ void *B,
                int ldB, void *beta, void *C, int ldC) {
  BLAS_INTEGER m = M, n = N, k = K, lda = ldA, ldb = ldB, ldc = ldC;
  char ta, tb;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  ta = trans2char(TransA);
  tb = trans2char(TransB);
  f77CGEMM( &ta, &tb, &m, &n, &k, alpha, A, &lda, B, &ldb, beta, C, &ldc,
            1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_chemm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_SIDE Side,
                enum LFC_CBLAS_UPLO Uplo, int M, int N,
                void *alpha, /*const*/ void *A, int ldA, /*const*/ void *B,
                int ldB, void *beta, void *C, int ldC) {
  BLAS_INTEGER m = M, n = N, lda = ldA, ldb = ldB, ldc = ldC;
  char side, uplo;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  side = side2char(Side);
  uplo = uplo2char(Uplo);
  f77CHEMM( &side, &uplo, &m, &n, alpha, A, &lda, B, &ldb, beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_ctrmm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_SIDE Side,
                enum LFC_CBLAS_UPLO Uplo, enum LFC_CBLAS_TRANSPOSE TransA,
                enum LFC_CBLAS_DIAG Diag, int M, int N, void *alpha,
                /*const*/ void *A, int ldA, void *B, int ldB) {
  BLAS_INTEGER m = M, n = N, lda = ldA, ldb = ldB;
  char side, uplo, trans, diag;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  side = side2char(Side);
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77CTRMM( &side, &uplo, &trans, &diag, &m, &n, alpha, A, &lda, B, &ldb,
            1, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_ctrsm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_SIDE Side,
                enum LFC_CBLAS_UPLO Uplo, enum LFC_CBLAS_TRANSPOSE TransA,
                enum LFC_CBLAS_DIAG Diag, int M, int N, void *alpha,
                /*const*/ void *A, int ldA, void *B, int ldB) {
  BLAS_INTEGER m = M, n = N, lda = ldA, ldb = ldB;
  char side, uplo, trans, diag;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  side = side2char(Side);
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77CTRSM( &side, &uplo, &trans, &diag, &m, &n, alpha, A, &lda, B, &ldb,
            1, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_csyrk(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_UPLO Uplo,
		enum LFC_CBLAS_TRANSPOSE Trans, int N, int K, void *alpha,
		/*const*/ void *A, int ldA, void *beta, void *C, int ldC){
  BLAS_INTEGER n = N, k = K, lda = ldA, ldc = ldC;
  char uplo, trans;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(Trans);
  f77CSYRK( &uplo, &trans, &n, &k, alpha, A, &lda, beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_cherk(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_UPLO Uplo,
		enum LFC_CBLAS_TRANSPOSE Trans, int N, int K, float alpha,
		/*const*/ void *A, int ldA, float beta, void *C, int ldC){
  BLAS_INTEGER n = N, k = K, lda = ldA, ldc = ldC;
  char uplo, trans;
  f77assert(Order == LFC_CblasColMajor);

  f77assert(sizeof(BLAS_REAL)==sizeof(float));
#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(Trans);
  f77CHERK( &uplo, &trans, &n, &k, &alpha, A, &lda, &beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_csymm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_SIDE Side,
		enum LFC_CBLAS_UPLO Uplo, int M, int N, void *alpha,
		/*const*/ void *A, int ldA,  /*const*/ void *B, int ldB,
		void *beta, void *C, int ldC) {
  BLAS_INTEGER n = N, m = M, lda = ldA, ldb = ldB, ldc = ldC;
  char side, uplo;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_REAL)==sizeof(float));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  side = side2char(Side);
  uplo = uplo2char(Uplo);
  f77CSYMM( &side, &uplo, &m, &n, alpha, A, &lda, B, &ldb, beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}
/* */ 

void
lfc_cblas_zaxpy(int N, void *alpha, /*const*/ void *X,int incX, void *Y,
		int incY) {
  BLAS_INTEGER n = N, incx = incX, incy = incY;
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  f77ZAXPY( &n, alpha, X, &incx, Y, &incy );
}

void
lfc_cblas_zcopy(int N, /*const*/ void *X, int incX, void *Y, int incY) {
  BLAS_INTEGER n = N, incx = incX, incy = incY;
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  f77ZCOPY( &n, X, &incx, Y, &incy );
}

void
lfc_cblas_zgerc(enum LFC_CBLAS_ORDER Order, int M, int N, void *alpha,
	       /*const*/ void *X, int incX, void *Y, int incY, void *A,
	       int ldA) {
  BLAS_INTEGER m = M, n = N, incx = incX, incy = incY, lda = ldA;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  f77ZGERC( &m, &n, alpha, X, &incx, Y, &incy, A, &lda );
}

void
lfc_cblas_zgeru(enum LFC_CBLAS_ORDER Order, int M, int N, void *alpha,
	       /*const*/ void *X, int incX, void *Y, int incY, void *A,
	       int ldA) {
  BLAS_INTEGER m = M, n = N, incx = incX, incy = incY, lda = ldA;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  f77ZGERU( &m, &n, alpha, X, &incx, Y, &incy, A, &lda );
}

double
lfc_cblas_dznrm2(int N, void *X, int incX) {
  BLAS_INTEGER n = N, incx = incX;
  BLAS_DOUBLE_PRECISION rv;
  extern BLAS_DOUBLE_PRECISION f77DZNRM2(BLAS_INTEGER *, void *, BLAS_INTEGER *);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  rv = f77DZNRM2( &n, X, &incx );
  return rv;
}

double
lfc_cblas_dzasum(/*const*/ int N, /*const*/ void *X, /*const*/ int incX) {
  BLAS_INTEGER n = N, incx = incX;
  BLAS_DOUBLE_PRECISION rv;
  extern BLAS_DOUBLE_PRECISION f77ZASUM(BLAS_INTEGER *, void *, BLAS_INTEGER *);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  rv = f77DZASUM( &n, X, &incx );
  return rv;
}

void
lfc_cblas_zscal(int N, void *alpha, void *X, int incX) {
  BLAS_INTEGER n = N, incx = incX;
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  f77ZSCAL( &n, alpha, X, &incx );
}

void
lfc_cblas_zdscal(int N, double alpha, void *X, int incX) {
  BLAS_INTEGER n = N, incx = incX;
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  f77ZDSCAL( &n, &alpha, X, &incx );
}

void
lfc_cblas_zswap(int N, void *X, int incX, void *Y, int incY) {
  BLAS_INTEGER n = N, incx = incX, incy = incY;
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  f77ZSWAP( &n, X, &incx, Y, &incy );
}

int
lfc_cblas_izamax(int N, /*const*/ void *X, int incX) {
  BLAS_INTEGER n = N, incx = incX, r;
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  r = f77IZAMAX( &n, X, &incx ) - 1;
  return r;
}

void
lfc_cblas_zher(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                /*const*/ int N, /*const*/ double alpha, /*const*/ void *X,
                /*const*/ int incX, void *A, /*const*/ int ldA) {
  BLAS_INTEGER n = N, incx = incX, lda = ldA;
  char uplo;

  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  f77assert(Order == LFC_CblasColMajor);
#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  f77ZHER( &uplo, &n, &alpha, X, &incx, A, &lda, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_zher2(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                /*const*/ int N, /*const*/ void *alpha, /*const*/ void *X,
                /*const*/ int incX, /*const*/ void *Y, /*const*/ int incY, void *A,
                /*const*/ int ldA) {
  BLAS_INTEGER n = N, incx = incX, incy = incY, lda = ldA;
  char uplo;

  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
  f77assert(Order == LFC_CblasColMajor);
#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  f77ZHER2( &uplo, &n, alpha, X, &incx, Y, &incy, A, &lda, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_zhemv(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                /*const*/ int N, /*const*/ void *alpha, /*const*/ void *A,
                /*const*/ int ldA, /*const*/ void *X, /*const*/ int incX,
                /*const*/ void *beta, void *Y, /*const*/ int incY) {
  BLAS_INTEGER n = N, lda = ldA, incx = incX, incy = incY;
  char uplo;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  f77ZHEMV( &uplo, &n, alpha, A, &lda, X, &incx, beta, Y, &incy, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_zhbmv(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                 /*const*/ int N, /*const*/ int K, /*const*/ void *alpha, /*const*/ void *A,
                 /*const*/ int ldA, /*const*/ void *X, /*const*/ int incX,
                 /*const*/ void *beta, void *Y, /*const*/ int incY) {
  BLAS_INTEGER n = N, k = K, lda = ldA, incx = incX, incy = incY;
  char uplo;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  f77ZHBMV( &uplo, &n, &k, alpha, A, &lda, X, &incx, beta, Y, &incy, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_zgbmv(/*const*/ enum LFC_CBLAS_ORDER Order,
                 /*const*/ enum LFC_CBLAS_TRANSPOSE TransA, /*const*/ int M, /*const*/ int N,
                 /*const*/ int KL, /*const*/ int KU, /*const*/ void *alpha,
                 /*const*/ void *A, /*const*/ int ldA, /*const*/ void *X,
                 /*const*/ int incX, /*const*/ void *beta, void *Y, /*const*/ int incY) {
  BLAS_INTEGER m = M, n = N, kl = KL, ku = KU, lda = ldA, incx = incX, incy = incY;
  char ta;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  ta = trans2char(TransA);
  f77ZGBMV( &ta, &m, &n, &kl, &ku, alpha, A, &lda, X, &incx, beta, Y, &incy, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_zgemv(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_TRANSPOSE TransA,
		int M, int N, void *alpha, /*const*/ void *A, int ldA,
		/*const*/ void *X, int incX, void *beta, void *Y,
		int incY) {
  BLAS_INTEGER m = M, n = N, lda = ldA, incx = incX, incy = incY;
  char ta;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  ta = trans2char(TransA);
  f77ZGEMV( &ta, &m, &n, alpha, A, &lda, X, &incx, beta, Y, &incy, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_zsyr2k(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                  /*const*/ enum LFC_CBLAS_TRANSPOSE Trans, /*const*/ int N, /*const*/ int K,
                  /*const*/ void *alpha, /*const*/ void *A, /*const*/ int ldA,
                  /*const*/ void *B, /*const*/ int ldB, /*const*/ void *beta,
                  void *C, /*const*/ int ldC) {
  BLAS_INTEGER n = N, k = K, lda = ldA, ldb = ldB, ldc = ldC;
  char uplo, trans;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(Trans);
  f77ZSYR2K( &uplo, &trans, &n, &k, alpha, A, &lda, B, &ldb, beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_zher2k(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                  /*const*/ enum LFC_CBLAS_TRANSPOSE Trans, /*const*/ int N, /*const*/ int K,
                  /*const*/ void *alpha, /*const*/ void *A, /*const*/ int ldA,
                  /*const*/ void *B, /*const*/ int ldB, /*const*/ double beta,
                  void *C, /*const*/ int ldC) {
  BLAS_INTEGER n = N, k = K, lda = ldA, ldb = ldB, ldc = ldC;
  char uplo, trans;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(Trans);
  f77ZHER2K( &uplo, &trans, &n, &k, alpha, A, &lda, B, &ldb, &beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_ztbsv(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                 /*const*/ enum LFC_CBLAS_TRANSPOSE TransA, /*const*/ enum LFC_CBLAS_DIAG Diag,
                 /*const*/ int N, /*const*/ int K, /*const*/ void *A, /*const*/ int ldA,
                 void *X, /*const*/ int incX) {
  BLAS_INTEGER n = N, k = K, lda = ldA, incx = incX;
  char uplo, trans, diag;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77ZTBSV( &uplo, &trans, &diag, &n, &k, A, &lda, X, &incx, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_ztrsv(/*const*/ enum LFC_CBLAS_ORDER Order, /*const*/ enum LFC_CBLAS_UPLO Uplo,
                 /*const*/ enum LFC_CBLAS_TRANSPOSE TransA, /*const*/ enum LFC_CBLAS_DIAG Diag,
                 /*const*/ int N, /*const*/ void *A, /*const*/ int ldA, void *X,
                 /*const*/ int incX) {
  BLAS_INTEGER n = N, lda = ldA, incx = incX;
  char uplo, trans, diag;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77ZTRSV( &uplo, &trans, &diag, &n, A, &lda, X, &incx, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void lfc_cblas_ztrmv(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_UPLO Uplo,
  enum LFC_CBLAS_TRANSPOSE TransA, enum LFC_CBLAS_DIAG Diag, int N,
  /* const */ void *A, int ldA, void *X, int incX) {
  BLAS_INTEGER n = N, lda = ldA, incx = incX;
  char uplo, trans, diag;

  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77ZTRMV( &uplo, &trans, &diag, &n, A, &lda, X, &incx, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_zgemm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_TRANSPOSE TransA,
                enum LFC_CBLAS_TRANSPOSE TransB, int M, int N, int K,
                void *alpha, /*const*/ void *A, int ldA, /*const*/ void *B,
                int ldB, void *beta, void *C, int ldC) {
  BLAS_INTEGER m = M, n = N, k = K, lda = ldA, ldb = ldB, ldc = ldC;
  char ta, tb;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  ta = trans2char(TransA);
  tb = trans2char(TransB);
  f77ZGEMM( &ta, &tb, &m, &n, &k, alpha, A, &lda, B, &ldb, beta, C, &ldc,
            1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_zhemm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_SIDE Side,
                enum LFC_CBLAS_UPLO Uplo, int M, int N,
                void *alpha, /*const*/ void *A, int ldA, /*const*/ void *B,
                int ldB, void *beta, void *C, int ldC) {
  BLAS_INTEGER m = M, n = N, lda = ldA, ldb = ldB, ldc = ldC;
  char side, uplo;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  side = side2char(Side);
  uplo = uplo2char(Uplo);
  f77ZHEMM( &side, &uplo, &m, &n, alpha, A, &lda, B, &ldb, beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_ztrmm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_SIDE Side,
                enum LFC_CBLAS_UPLO Uplo, enum LFC_CBLAS_TRANSPOSE TransA,
                enum LFC_CBLAS_DIAG Diag, int M, int N, void *alpha,
                /*const*/ void *A, int ldA, void *B, int ldB) {
  BLAS_INTEGER m = M, n = N, lda = ldA, ldb = ldB;
  char side, uplo, trans, diag;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  side = side2char(Side);
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77ZTRMM( &side, &uplo, &trans, &diag, &m, &n, alpha, A, &lda, B, &ldb,
            1, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_ztrsm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_SIDE Side,
                enum LFC_CBLAS_UPLO Uplo, enum LFC_CBLAS_TRANSPOSE TransA,
                enum LFC_CBLAS_DIAG Diag, int M, int N, void *alpha,
                /*const*/ void *A, int ldA, void *B, int ldB) {
  BLAS_INTEGER m = M, n = N, lda = ldA, ldb = ldB;
  char side, uplo, trans, diag;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  side = side2char(Side);
  uplo = uplo2char(Uplo);
  trans = trans2char(TransA);
  diag = diag2char(Diag);
  f77ZTRSM( &side, &uplo, &trans, &diag, &m, &n, alpha, A, &lda, B, &ldb,
            1, 1, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_zsyrk(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_UPLO Uplo,
		enum LFC_CBLAS_TRANSPOSE Trans, int N, int K, void *alpha,
		/*const*/ void *A, int ldA, void *beta, void *C, int ldC){
  BLAS_INTEGER n = N, k = K, lda = ldA, ldc = ldC;
  char uplo, trans;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(Trans);
  f77ZSYRK( &uplo, &trans, &n, &k, alpha, A, &lda, beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_zherk(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_UPLO Uplo,
		enum LFC_CBLAS_TRANSPOSE Trans, int N, int K, double alpha,
		/*const*/ void *A, int ldA, double beta, void *C, int ldC){
  BLAS_INTEGER n = N, k = K, lda = ldA, ldc = ldC;
  char uplo, trans;
  f77assert(Order == LFC_CblasColMajor);

  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));
#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  uplo = uplo2char(Uplo);
  trans = trans2char(Trans);
  f77ZHERK( &uplo, &trans, &n, &k, &alpha, A, &lda, &beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

void
lfc_cblas_zsymm(enum LFC_CBLAS_ORDER Order, enum LFC_CBLAS_SIDE Side,
		enum LFC_CBLAS_UPLO Uplo, int M, int N, void *alpha,
		/*const*/ void *A, int ldA,  /*const*/ void *B, int ldB,
		void *beta, void *C, int ldC) {
  BLAS_INTEGER n = N, m = M, lda = ldA, ldb = ldB, ldc = ldC;
  char side, uplo;
  f77assert(Order == LFC_CblasColMajor);
  f77assert(sizeof(BLAS_DOUBLE_PRECISION)==sizeof(double));

#if BLAS_CHARACTER == BLAS_SUN_CHARACTER
  side = side2char(Side);
  uplo = uplo2char(Uplo);
  f77ZSYMM( &side, &uplo, &m, &n, alpha, A, &lda, B, &ldb, beta, C, &ldc, 1, 1 );
#else
  f77assert(0);
#endif
}

#endif 

LFC_END_C_DECLS
