/****************************************************************************
**  Copyright (C) 2004                                                     **
**  University of Tennessee, Innovative Computing Laboratory               **
**                                                                         **
**  See the file COPYRIGHT in the package base directory for details       **
****************************************************************************/

/*******************************
				
	PresenterApp.cpp	
					
********************************/
#include <iostream>
#include <fstream>

#include "PresenterApp.h"
#include "PresenterFrame.h"
#include "SpectrumPane.h"

using namespace std;

const int PresenterApp::square_size;

bool PresenterApp::OnInit() {
  parse_command();
  create_image_list();
  main_frame = new PresenterFrame(NULL, -1, _T("CUBE "), 
				  wxPoint(0, 0), wxSize(800, 600),
				  wxDEFAULT_FRAME_STYLE |
				  wxNO_FULL_REPAINT_ON_RESIZE |
				  wxHSCROLL | wxVSCROLL); 
  main_frame->CentreOnScreen(wxBOTH);
  main_frame->Show(TRUE);
  SetTopWindow(main_frame);
  if (cur_file.empty()) return true; // no filename in command line

  // if user specifid a file in command line.
  std::ifstream in(cur_file.c_str());
  if (!in) {
    wxLogMessage(_T("Open file error: '%s'"), cur_file.c_str());
    return true;
  }
  wxString path = cur_file.c_str();
  main_frame->SetTitle(path.Prepend("CUBE:  "));
  if (main_frame->get_cube() == NULL) {
    main_frame->set_cube(new Cube());
  } else {
    delete main_frame->get_cube();
    main_frame->set_cube(NULL);
  }
  in >> (*main_frame->get_cube());
  set_proc_num(main_frame->get_cube()->num_procs());
  set_thrd_num(main_frame->get_cube()->get_max_thrdnum());
  main_frame->refresh_label_mn();
  // m_thrd = main_frame->get_cube().thrd_per_proc();
  in.close();
  main_frame->build_met_tree();
  if (main_frame->is_calltree())
    main_frame->build_cnode_tree();
  else 
    main_frame->build_func_tree();
  main_frame->build_loc_tree();
  main_frame->get_spectrum_pane()->Refresh();
  return true;
  
} 

PresenterApp::~PresenterApp() {
  delete images;
}

bool parse_option(string option, string usage) {
  if (option.compare("-h") == 0 ||
      option.compare("--help") == 0) {
    cout << usage << endl;
    exit(0);
  } else if (option.compare("-v") == 0 ||
	     option.compare("--version") == 0) {
    cout << "CUBE   Version 1.0.7" << endl << endl;
    cout << "Copyright(c) 2003 ICL at UTK" << endl;
    exit(0);
  }
  else if (option[0] == '-') {
    // invalid option
    cout << "Invalid option, type -h for help" << endl;
    exit(0);
  }
  return false; // not an option at all 
}
 
void PresenterApp::parse_command() {
  int argc = wxApp::argc;
  char** argv = wxApp::argv;
  string usage = "Usage: presenter [options] [filename] \n";
  usage += "  where options are --help(-h), --version(-v)";
  if (argc >= 3) {
    cout << usage << endl;
    exit(0);
  }
  switch (argc) {
  case 1:
    break; // only "presenter" typed
  case 2: 
    if (parse_option(argv[1], usage)) {
      // if a valid option affecting the execution of presenter, do sth.
    }
    else {
      char* fname = argv[1];
      std::ifstream in(fname);
      if (!in) {
	cout << "File open error: " << fname << endl;
	exit(0);
      }
      in.close();
      cur_file = fname; // if success
    }
  }
  return;
}

void PresenterApp::create_image_list() {
  images = new wxImageList(square_size, square_size, true);
  wxMemoryDC mem_dc;
  wxBrush brush;
  wxPen pen;
  wxColour c;
  bg_color     = wxSystemSettings::GetColour(wxSYS_COLOUR_WINDOWFRAME); 
  shadow_color = wxSystemSettings::GetColour(wxSYS_COLOUR_3DDKSHADOW);
  light_color  = wxSystemSettings::GetColour(wxSYS_COLOUR_3DHIGHLIGHT);
  wxColour mygray(bg_color.Red()-40, bg_color.Green()-40, 
		  bg_color.Blue()-40);

  m_process = 0;
  m_thread  = 0;
  int delta = 4; // padding between color box and labels.

  /* 0 is background color, 101 is white(means error), 102 is dark grey */
  for (int i = 0; i <= 101; i++) {
    wxBitmap* box = new wxBitmap(square_size+delta, square_size);
    mem_dc.SelectObject(*box);
    mem_dc.SetPen(*wxBLACK_PEN);
    c = SpectrumPane::severity2color(i);
    brush.SetColour(c);
    mem_dc.SetBrush(brush);
    mem_dc.DrawRectangle(0, 0, square_size, square_size);

    /*
    // raised effect
    pen.SetColour(shadow_color);
    mem_dc.SetPen(pen);
    mem_dc.DrawLine(0, square_size-2, square_size, square_size-2);
    mem_dc.DrawLine(0, square_size-3, square_size, square_size-3);
    mem_dc.DrawLine(square_size-2, 0, square_size-2, square_size);
    pen.SetColour(light_color);
    mem_dc.SetPen(pen);
    mem_dc.DrawLine(1, 1, square_size-2, 1);
    mem_dc.DrawLine(1, 1, 1, square_size-2);
    */

    // draw right part
    brush.SetColour(bg_color);
    pen.SetColour(bg_color);
    mem_dc.SetBrush(brush);
    mem_dc.SetPen(pen);
    mem_dc.DrawRectangle(square_size, 0, delta, square_size);
    images->Add(*box);
  }
  // adding grey color for 0<value<0.5 with index = 102.
  wxBitmap* box = new wxBitmap(square_size+delta, square_size);
  mem_dc.SelectObject(*box);
  mem_dc.SetPen(*wxBLACK_PEN);
  //c = wxColour(_T("GREY"));
  brush.SetColour(mygray);
  mem_dc.SetBrush(brush);
  mem_dc.DrawRectangle(0, 0, square_size, square_size);

  /*
  // raised effect
  pen.SetColour(shadow_color);
  mem_dc.SetPen(pen);
  mem_dc.DrawLine(0, square_size-2, square_size, square_size-2);
  mem_dc.DrawLine(0, square_size-3, square_size, square_size-3);
  mem_dc.DrawLine(square_size-2, 0, square_size-2, square_size);
  pen.SetColour(light_color);
  mem_dc.SetPen(pen);
  mem_dc.DrawLine(1, 1, square_size-2, 1);
  mem_dc.DrawLine(1, 1, 1, square_size-2);
  */

  // draw right part
  brush.SetColour(bg_color);
  pen.SetColour(bg_color);
  mem_dc.SetBrush(brush);
  mem_dc.SetPen(pen);
  mem_dc.DrawRectangle(square_size, 0, delta, square_size);
  images->Add(*box);
}

wxImageList* PresenterApp::get_image_list() {
  return images;
}


IMPLEMENT_APP(PresenterApp)


