/****************************************************************************
**  Copyright (C) 2004                                                     **
**  University of Tennessee, Innovative Computing Laboratory               **
**                                                                         **
**  See the file COPYRIGHT in the package base directory for details       **
****************************************************************************/

/*******************************
				
     SpectrumPane.cpp	
					
********************************/

#include <iostream>
#include <math.h>
#include "SpectrumPane.h"
#include "PresenterFrame.h"
#include "PresenterApp.h"

using namespace std;

BEGIN_EVENT_TABLE(SpectrumPane, wxSashLayoutWindow)
  EVT_PAINT (SpectrumPane::OnPaint)
END_EVENT_TABLE()

SpectrumPane::SpectrumPane(wxWindow* parent, int id)
  : wxSashLayoutWindow(parent, id, wxDefaultPosition, 
		       wxSize(200, 30), wxSW_3D) {
  
  SetDefaultSize(wxSize(100, 30)); // any width size
  SetOrientation(wxLAYOUT_HORIZONTAL);
  SetAlignment(wxLAYOUT_BOTTOM);
  /* Use default font */
  // font.SetFamily(wxDEFAULT);
  // font.SetStyle(wxNORMAL);
  // font.SetWeight(wxNORMAL);
  
}

/* the actual spectrum-drawing method */
void SpectrumPane::paint_spectrum(wxDC& dc) {
  int width, height;
  int boxwidth;
  int box_num = 100;
  int diff = 0;
  int x;
  wxColour color;
  wxString str;
  PresenterFrame* frame = ((PresenterApp*)wxTheApp)->get_frame();
  GetClientSize(&width, &height);
  diff = width % 100; // the remaining empty area
  width -= diff;
  int cheight = height/2; // height of  colored box
  int w;
  boxwidth = width/box_num;
  diff /= 2;
  //font.SetPointSize(height/3); // height of scale value
  //dc.SetFont(font);
  
  dc.BeginDrawing();
  for (int i = 0; i < 100; i++) {
    color = severity2color(i+1);
    pen.SetColour(color);
    dc.SetPen(pen);   
    brush.SetColour(color);
    dc.SetBrush(brush);
    x = diff+i*boxwidth;
    dc.DrawRectangle(x, 0, boxwidth-1, cheight);

    if (i % 10 == 0) {
      // draw scale line.
      dc.SetPen(*wxBLACK_PEN);
      if (frame->view_type() == 0 && (i ==0 || i == 50)) {
	// absolute mode, only 3 lines
	dc.DrawLine(x, 0, x, height);
      };
      if (frame->view_type() != 0) {
	dc.DrawLine(x, 0, x, height);
      }
      if (frame->view_type() != 0 && i > 0) { // don't draw "0"
	str << i; // convert integer to string
	dc.GetTextExtent(str, &w, NULL);
	// draw a text to the left of a line.
	dc.DrawText(str, x-w-3, cheight+2);
	str.Empty();
      }
    }
  }
  
  if (frame->get_cube() != NULL) {
    double max = frame->get_cube()->get_max();
    if (frame->view_type() == 0 && max > 0) {
      int x2;
      // write max
      //str.Printf("%10.2f", max);
      string maxstr = frame->get_cube()->gen_max_label(max);
      str = maxstr.c_str();
      dc.GetTextExtent(str, &w, NULL);
      x2 = diff + 99 * boxwidth;
      dc.DrawText(str, x2-w, cheight+2);
      str.Empty();
      // write max/2
      //str.Printf("%10.2f", max/2.0);
      maxstr = frame->get_cube()->gen_half_max_label(max);
      str = maxstr.c_str();
      dc.GetTextExtent(str, &w, NULL);
      x2 = diff + 49 * boxwidth;
      dc.DrawText(str, x2-w, cheight+2);
      str.Empty();
    }
  }
  
  // draw the 10th line.
  dc.SetPen(*wxBLACK_PEN);
  x += boxwidth-1;
  dc.DrawLine(x, 0, x, height);
  if (frame->view_type() != 0) {
    dc.GetTextExtent(_T("100"), &w, NULL);
    dc.DrawText("100", x-w-3, cheight+2);
  }
  dc.EndDrawing();
}

void SpectrumPane::OnPaint(wxPaintEvent& event) {
  wxPaintDC dc(this);
  paint_spectrum(dc);
}

/* 
   1st argument is a real severity value.
 */ 

wxColour SpectrumPane::severity2color(double val, double min, double max) {
  double range = max - min;
  double perc = (val / range)*100; 
  double hue;
  PresenterApp* app = (PresenterApp*) wxTheApp;
  
  if (val < min || val > max) 
    return *wxWHITE; // error, set "white" color
  else if (perc == 0.0)
    return app->bg_color;
    //return *wxLIGHT_GREY; // alternative
  else if (perc < 0.5)
    return wxColour(_T("DARK GREY"));
  else {
    hue = TWOPI * 0.75 * (100.0 - logconv(perc)) / 100.0;
    // hue = perc*TWOPI; // a full range of color
    return SpectrumPane::hsi2rgb(hue);
  }
}

/* make the beginning blue colors have larger constrast. */
double SpectrumPane::logconv(double x) {
  double x0 = 3;
  double dx = 100;
  double y0 = log(x0);
  double y1 = log(x0 + dx);
  double dy  = y1 - y0;
  double y  = log(x + x0);
  return (y - y0) * 100.0 / dy;
    
}

// static method
wxColour SpectrumPane::hsi2rgb(double hue, 
			       double saturation, 
			       double intensity) {
  double r, g ,b;
  unsigned char R, G, B;
  
  if (saturation == 0) {
    unsigned char t = static_cast<unsigned char>(intensity);
    return wxColour(t, t, t);
  }
  else {
    while(hue < 0) {
      hue = hue + TWOPI;
    }
    while(hue >= TWOPI) {
      hue = hue - TWOPI;
    }
    hue = 3.0 * hue / PI;
    double f = hue - floor(hue);
    double p = intensity * (1.0 - saturation);
    double q = intensity * (1.0 - saturation * f);
    double t = intensity * (1.0 - saturation * (1.0 - f));
    
    int h = int(hue);
    switch(h) {
    case 0: 
      r = intensity; g = t; b = p;
      break;
    case 1: 
      r = q; g = intensity; b = p;
      break;
    case 2: 
      r = p; g = intensity; b = t;
      break;
    case 3: 
      r = p; g = q; b = intensity;
      break;
    case 4: 
      r = t; g = p; b = intensity;
      break;
    case 5: 
      r = intensity; g = p; b = q;
      break;
    }      
    if (r < 0.0) r = 0.0;
    if (r > 1.0) r = 1.0;
    if (g < 0.0) g = 0.0;
    if (g > 1.0) g = 1.0;
    if (b < 0.0) b = 0.0;
    if (b > 1.0) b = 1.0;
  }
  
  /* Shouldn't use "(unsigned char) r*255" here, 
     it just doesn't work in C++. A painful bug! */
  R = static_cast<unsigned char>(r*255);
  G = static_cast<unsigned char>(g*255);
  B = static_cast<unsigned char>(b*255);

  return wxColour(R, G, B);

}
