/****************************************************************************
**  Copyright (C) 2004                                                     **
**  University of Tennessee, Innovative Computing Laboratory               **
**                                                                         **
**  See the file COPYRIGHT in the package base directory for details       **
****************************************************************************/

#ifndef _CUBE__CACHE_H
#define _CUBE__CACHE_H   

/*
 *----------------------------------------------------------------------------
 *
 * class CubeCache
 *
 *----------------------------------------------------------------------------
 */

#include <map>
#include "CubeMatrix.h"

#define CubeCache_Mod_Flag      1
#define CubeCache_Sub_Flag      2

#define TOL 1.0e-12

class Cube;

class CubeCache {

 public:

  CubeCache         (Cube*);
  double get_vsev   (int mode[], int met_id, int prog_id = -1, int loc_id = -1,
		     bool iscalltree = true);    
  void   set_flag   (int flag, bool val);
  
 private:
  
  /****************************************************
   *
   *      Cache Support Private Methods
   *
   ***************************************************/

  // call-tree node severity
  double get_vcsev (int mode[], int met_id, int cnode_id = -1, int loc_id = -1);
  // region severity
  double get_vrsev (int mode[], int met_id, int reg_id, int loc_id = -1);
  // module severity
  double get_vmsev (int mode[], int met_id, int mod_id, int loc_id = -1);
  // subroutines severity
  double get_vssev (int mode[], int met_id, int reg_id, int loc_id = -1);

  // displayed node: b in B; BxNxL
  double     get_vsev_met       (int state, int met_id);
  // displayed node: n in N or R; BxNxL
  double     get_vsev_prog       (int met_state, int met_id, int prog_state, 
				  int prog_id, bool iscalltree);
  // displayed node: l in L; Bx{N,R}xL
  double     get_vsev_loc        (int met_state, int met_id, 
				  int cnode_state, int cnode_id,
				  int loc_state, int loc_id, 
				  bool iscalltree);
  
  void   update_metforest_cache ();
  void   update_cnodeforest_cache(int met_state, int met_id);

  // refresh caches for calltree mode
  CubeMatrix update_met_cache   (int rootid);
  double update_cnode_cache      (int met_state, int met_id, int rootid);
  double update_loc_cache        (int met_state, int cnode_state,
				  int met_id, int cnode_id, int rootid);

  // refresh caches for func profile moe
  void   update_func_cache       (int met_state, int met_id);
  void   update_loc_cache4func   (int met_state, int func_state, int met_id, 
				  int func_id, int rootid);
  void   update_loc_cache4mod    (int met_state, int mod_state,
				  int met_id, int mod_id, int rootid);
  void   update_loc_cache4sub    (int met_state,
				  int met_id, int reg_id, int rootid);
  // called by get_vcsev() method
  double     sev(int x, int y, int z, int b, int n, int l);
  double     sev(       int y, int z, int b, int n, int l);
  double     sev(              int z, int b, int n, int l );

  // Equations for function profile tree
  double sev4func_excl(int met_state, int func_state, int loc_state,
		       int met_id, int func_id, int locid);
  double sev4func_incl(int met_state, int func_state, int loc_state,
		       int met_id, int func_id, int locid);
  
  // to support for exclusive matrix.
  double get_esev_ex  (int met_id, int cnode_id, int loc_id);
  double get_sev_ex   (int met_id, int cnode_id, int thrd_id);
  
  
  /****************************************************
   *
   *      Cache Support Private Data Members
   *
   ***************************************************/
  
  Cube* cube;

  // Store exclusive matrix for performance metrics.
  std::map<int, CubeMatrix > excl_severity;

  // get_vsev( ) calculation flags
  bool isModule;
  bool isSubroutine;

  bool excl_met_f; // whether to use Exclusive Matrix or not.

  // Cache coherence state flags:
  int old_met_id, old_cnode_id, old_loc_id;
  int old_met_state, old_cnode_state, old_loc_state;

  bool met_cache_type, cnode_cache_type, loc_cache_type;
  bool loc_cache_mod; // is it for module?
  bool loc_cache_sub; // is it for subroutines?

  // location dimension cache
  std::map<int, double> loc_buf_in; // inclusive
  std::map<int, double> loc_buf_ex; // exclusive

  // prog dimension cache
  std::map<int, double> cnode_buf_in;
  std::map<int, double> cnode_buf_ex;

  // metric dimension cache
  std::map<int, double> met_buf_in;
  std::map<int, double> met_buf_ex;
  
};

#endif
