/****************************************************************************
**  Copyright (C) 2004                                                     **
**  University of Tennessee, Innovative Computing Laboratory               **
**                                                                         **
**  See the file COPYRIGHT in the package base directory for details       **
****************************************************************************/

/************************************************
             CubeCache.cpp


 ************************************************/

#include "Cube.h"
#include "CubeCache.h"

#include "Metric.h"
#include "Module.h"
#include "Region.h"
#include "Callsite.h"
#include "Cnode.h"

#include "Grid.h"
#include "Machine.h"
#include "Node.h"
#include "Process.h"
#include "Thread.h"


using namespace std;

CubeCache::CubeCache(Cube* cube) : cube(cube) { 
  old_met_id     = -1; // N/A value assigned
  old_cnode_id    = -1;
  old_loc_id      = -1;
  old_met_state  = -2; // N/A value assigned
  old_cnode_state = -2;
  old_loc_state   = -2;

  met_cache_type = true; // is calltree or not
  cnode_cache_type= true;
  loc_cache_type  = true;

  loc_cache_mod   = false; 
  loc_cache_sub   = false;

  isModule        = false;
  isSubroutine    = false;
}

double CubeCache::get_vsev(int mode[], int met_id, int prog_id, int loc_id, 
			   bool iscalltree) {
  if (prog_id == -1 && loc_id == -1) {
    return get_vsev_met(mode[CUBE_MET], met_id);
  } else if (loc_id == -1) {
    return get_vsev_prog(mode[CUBE_MET], met_id,
			 mode[CUBE_PROG], prog_id,
			 iscalltree); 
  } else {
    return get_vsev_loc(mode[CUBE_MET], met_id,
			mode[CUBE_PROG], prog_id,
			mode[CUBE_LOC], loc_id, iscalltree);
  }
}

void CubeCache::set_flag    (int flag, bool val) {
  switch(flag) {
  case CubeCache_Mod_Flag:
    isModule = val;
    break;
  case CubeCache_Sub_Flag:
    isSubroutine = val;
    break;
  }
}

//   Both exclusive and inclusive buffers are filled in.
CubeMatrix CubeCache::update_met_cache(int rootid) {
  Metric* root = (Metric*) cube->get_met(rootid);
  int childnum = root->num_children();
  CubeMatrix mat = cube->severity[rootid];
  int mode[] = {CUBE_INCL, CUBE_INCL, CUBE_INCL};
  if (childnum == 0) {
    met_buf_in[rootid] = get_vcsev(mode, rootid);
    met_buf_ex[rootid] = met_buf_in[rootid];
    excl_severity[rootid] = mat; // assign an exclusive matrix
    return mat;
  } else {
    CubeMatrix tmp = mat;
    for (int i = 0; i < childnum; i++) {
      int child_id = ((Metric*) root->get_child(i))->get_id();
      CubeMatrix child = update_met_cache(child_id);
      tmp = tmp - child;
    }
    excl_severity[rootid] = tmp; // assign an exclusive matrix
    // if (rootid == 1) cout << tmp;
    met_buf_ex[rootid] = tmp.sum();
    met_buf_in[rootid] = get_vcsev(mode, rootid);
    return mat;
  }
}

double CubeCache::update_cnode_cache(int met_state, int met_id, int rootid) {
  Cnode* root = (Cnode*) cube->get_cnode(rootid);
  int childnum = root->num_children();
  int mode[3];
  mode[CUBE_MET] = met_state;
  mode[CUBE_PROG] = CUBE_EXCL;
  mode[CUBE_LOC]  = CUBE_INCL;

  double val = get_vcsev(mode, met_id, rootid);


  if (childnum == 0) {
    val = get_vcsev(mode, met_id, rootid);
    cnode_buf_ex[rootid] = val;
    cnode_buf_in[rootid] = val;
    return val;
  } else {
    for (int i = 0; i < childnum; i++) {
      int child_id = ((Cnode*) root->get_child(i))->get_id();
      val += update_cnode_cache(met_state, met_id, child_id);
    }
    cnode_buf_ex[rootid] = get_vcsev(mode, met_id, rootid);
    cnode_buf_in[rootid] = val;
    return val;
  }
  old_met_state = met_state;
  old_met_id = met_id;
}

double CubeCache::update_loc_cache(int met_state, int cnode_state,
				   int met_id, int cnode_id, int rootid) {
  // refresh the flags for the location cache
  old_met_id = met_id;
  old_met_state = met_state;
  old_cnode_id = cnode_id;
  old_cnode_state = cnode_state;

  Location* root = (Location*) cube->get_loc(rootid);
  int childnum = root->num_children();
  int mode[3];
  mode[CUBE_MET] = met_state;
  mode[CUBE_PROG] = cnode_state;
  mode[CUBE_LOC]  = CUBE_EXCL;
  double val = get_vcsev(mode, met_id, cnode_id, rootid);
  if (childnum == 0) {
    val = get_vcsev(mode, met_id, cnode_id, rootid);
    loc_buf_ex[rootid] = val;
    loc_buf_in[rootid] = val;
    return val;
  } else {
    for (int i = 0; i < childnum; i++) {
      int child_id = ((Location*) root->get_child(i))->get_locid();
      val += update_loc_cache(met_state, cnode_state,
			      met_id, cnode_id, child_id);
    }
    loc_buf_ex[rootid] = get_vcsev(mode, met_id, cnode_id, rootid);
    loc_buf_in[rootid] = val;
    return val;
  }
}
void CubeCache::update_metforest_cache() {
  Metric* root;
  vector<Metric*> rmetv = cube->get_met_roots();
  for(int i = 0; i < rmetv.size(); i++) {
    root = rmetv[i];
    int pid = root->get_id();
    update_met_cache(pid);
  }
  // cout << "Performance Metric cache is updated" << endl;
}

void CubeCache::update_cnodeforest_cache(int met_state, int met_id) {
  Cnode* root;
  vector<Cnode*> rcnodev = cube->get_cnode_roots();
  for(int i = 0; i < rcnodev.size(); i++) {
    root = rcnodev[i];
    int id = root->get_id();
    update_cnode_cache(met_state, met_id, id);
  }
  // cout << "Cnode cache is updated" << endl;
}

// For displayed node: b in B; BxNxL
double CubeCache::get_vsev_met(int state, int met_id) {
  double ret;
  if (old_met_id == -1) {
      update_metforest_cache();
      old_met_id = met_id;
  }
  if (state == CUBE_EXCL) {
    ret = met_buf_ex[met_id];
  }
  else { 
    ret = met_buf_in[met_id];
  }
  return ret;
}

// For displayed node: n in N; BxNxL
double CubeCache::get_vsev_prog(int met_state, int met_id, 
				int prog_state, int prog_id, 
				bool iscalltree) {
  double ret;
  int mode[3];
  mode[CUBE_MET] = met_state;
  mode[CUBE_PROG] = prog_state;
  mode[CUBE_LOC]  = -1;
  // cache miss, update cache
  if (cnode_cache_type != iscalltree || 
      met_state != old_met_state || 
      met_id != old_met_id) {
    if (iscalltree) {
      update_cnodeforest_cache(met_state, met_id);
      update_loc_cache(met_state, prog_state, met_id, prog_id, 0);
      //If cnode doesn't change, but met modified old_cnode_id, 
      //loc_cache still needs update.
    } else {
      update_func_cache(met_state, met_id);
      if (!isModule && !isSubroutine) // regular region
	update_loc_cache4func(met_state, prog_state, met_id, prog_id, 0);
      else if (isModule) { // module
	update_loc_cache4mod(met_state, prog_state, met_id, prog_id, 0);
	cnode_cache_type = iscalltree; // subroutines
	return get_vmsev(mode, met_id, prog_id, -1);
      } else {
	update_loc_cache4sub(met_state, met_id, prog_id, 0);
	cnode_cache_type = iscalltree; // subroutines
	return get_vssev(mode, met_id, prog_id, -1);	
      }
    }
    cnode_cache_type = iscalltree;
  } else {
    // since there is no cache for module
    if (isModule) {
      return get_vmsev(mode, met_id, prog_id, -1);
    } else  if (isSubroutine) { 
      return get_vssev(mode, met_id, prog_id, -1);
    }
  }
  if (prog_state == CUBE_EXCL) 
    ret = cnode_buf_ex[prog_id];
  else 
    ret = cnode_buf_in[prog_id];

  return ret;
}

// For displayed node: l in L; BxNxL
double CubeCache::get_vsev_loc(int met_state, int met_id, 
			       int cnode_state, int cnode_id,
			       int loc_state, int loc_id, 
			       bool iscalltree) {
  double ret;
  // cache miss, update caches
  if (loc_cache_type != iscalltree   || loc_cache_mod != isModule ||
      met_state != old_met_state   || met_id != old_met_id ||
      cnode_state != old_cnode_state || cnode_id != old_cnode_id ||
      loc_cache_sub != isSubroutine) {
    if (iscalltree)
      update_loc_cache(met_state, cnode_state, met_id, cnode_id, 0);
    else {
      if (!isModule && !isSubroutine)
	update_loc_cache4func(met_state, cnode_state, met_id, cnode_id, 0);
      else if (isModule) {
	update_loc_cache4mod(met_state, cnode_state, met_id, cnode_id, 0);
      }
      else {
	update_loc_cache4sub(met_state, met_id, cnode_id, 0);
      }
    }
    loc_cache_type = iscalltree;
    //loc_cache_mod  = isModule;
  }
  if (loc_state == CUBE_EXCL)
    ret = loc_buf_ex[loc_id];
  else 
    ret = loc_buf_in[loc_id];

  return ret;
}

double CubeCache::get_esev_ex(int met_id, int cnode_id, int loc_id) {
  map<int, Thread*>::iterator i;
  Thread* t;
  // see whether a location is a thread
  for (i = cube->thrdv.begin(); i != cube->thrdv.end(); i++) {
    if (i->second->get_locid() == loc_id) break;
  }
  // not a thread, return 0.
  if (i == cube->thrdv.end()) return 0;
  t = i->second;;
  int thrd_id = t->get_id();
  return get_sev_ex(met_id, cnode_id, thrd_id); 
}

double CubeCache::get_sev_ex   (int met_id, int cnode_id, int thrd_id) {
  return excl_severity[met_id][cnode_id][thrd_id];
}

/* x, y, z are inclusive/exclusive state */
double CubeCache::sev(int x, int y, int z, int b, int n, int l) {
  double sum = 0.0;
  Metric* p = (Metric*) cube->get_met(b);
  if (x == CUBE_EXCL) excl_met_f = true;
  else excl_met_f = false;
  sum = sev(y, z, b, n, l);
  // if leaf tree node
  if (p->num_children() == 0) {
    return sum;
  }
  // if inclusive mode
  if (x == CUBE_INCL) return sum;
  // if exclusive mode, utilize exclusive matrix (eee) instead
  return sum;

  /*for (int i = 0; i < p->num_children(); i++) {
    int cid = ((Metric*) p->get_child(i))->get_id();
    double tmp = sev(CUBE_INCL, y, z, cid, n, l);
    sum -= tmp;
  }
  if (sum < 0 && (-sum) <= TOL) {
    sum = 0;
  }

  return sum; */
}

double CubeCache::sev(       int y, int z, int b, int n, int l) {
  double sum = 0.0;
  Cnode* cnode = (Cnode*) cube->get_cnode(n);
  sum = sev(z, b, n, l);
  // if leaf node
  if (cnode->num_children() == 0) {
    return sum;
  }
  // if exclusive mode
  if (y == CUBE_EXCL) return sum;
  // if inclusive mode
  for (int i = 0; i < cnode->num_children(); i++) {
    int cid = ((Cnode*)cnode->get_child(i))->get_id();
    sum += sev(CUBE_INCL, z, b, cid, l);
  }
  return sum;
}

double CubeCache::sev(              int z, int b, int n, int l ) {
  double sum = 0.0;
  Location* loc = (Location*) cube->get_loc(l);

  if (!excl_met_f) 
    sum = cube->get_esev(b, n, l);
  else // if focused metric is in Exclusive state, use exclusive matrix.
    sum = get_esev_ex(b, n, l);


  // arriving at a leaf node
  if (loc->num_children() == 0) {
    return sum;
  }
  // if exclusive mode
  if (z == CUBE_EXCL) return sum;
  // if inclusive mode
  for (int i = 0; i < loc->num_children(); i++) {
    int cid = ((Location*)loc->get_child(i))->get_locid();
    // add its children up
    sum += sev(CUBE_INCL, b, n, cid); 
  }
  return sum;
}

double CubeCache::get_vcsev(int mode[], int met_id, int cnode_id, int loc_id) {
  double sum = 0.0;
  int cnid;
  if (loc_id   < 0) loc_id = 0;
  if (cnode_id < 0) { // sum of the cnode roots
    // cnode_id = 0;
    vector<Cnode*>& cn_roots = cube->rcnodev;
    for (int i = 0; i < cn_roots.size(); i++) {
      Cnode* cur = cn_roots[i];
      cnid = cur->get_id();
      sum += sev(mode[CUBE_MET], CUBE_INCL, CUBE_INCL,
		 met_id, cnid, loc_id);
    }
    return sum;
  }
  // cout << "get_vcsev: " << mode[0] << "," << mode[1] << "," << mode[2] << endl;
  return sev(mode[CUBE_MET], mode[CUBE_PROG], mode[CUBE_LOC],
		  met_id, cnode_id, loc_id);
}

double CubeCache::get_vrsev(int mode[], int met_id, int reg_id, int loc_id) {
  double sum = 0.0;
  if (loc_id < 0) {
    if (mode[CUBE_PROG] == CUBE_EXCL) {
      return sev4func_excl(mode[CUBE_MET], CUBE_EXCL, CUBE_INCL, met_id, reg_id, 0);
    } else {
      return sev4func_incl(mode[CUBE_MET], CUBE_INCL, CUBE_INCL, met_id, reg_id, 0);
    } 
  } 
  else {
    if (mode[CUBE_PROG] == CUBE_EXCL) {
      sum = sev4func_excl(mode[CUBE_MET], CUBE_EXCL, mode[CUBE_LOC], 
			  met_id, reg_id, loc_id);
    } else {
      sum = sev4func_incl(mode[CUBE_MET], CUBE_INCL, mode[CUBE_LOC], 
			  met_id, reg_id, loc_id);
    } 
  }  
  return sum;
  
}
  
double CubeCache::get_vmsev(int mode[], int met_id, int mod_id, int loc_id) {
  Module* mod = (Module*) cube->get_mod(mod_id);
  double sev = 0;
  if (mode[CUBE_PROG] == CUBE_INCL) {
    Region* r;
    // add up all its callees' value
    for (int i = 0; i < mod->num_children(); i++) {
      r = (Region*) mod->get_child(i);
      if (r->get_name().compare("virtual caller") != 0) {
	int mode2[3];
	mode2[CUBE_MET] = mode[CUBE_MET];
	mode2[CUBE_PROG] = CUBE_EXCL;
	mode2[CUBE_LOC]  = mode[CUBE_LOC];
	sev += get_vrsev(mode2, met_id, r->get_id(), loc_id);
      }
    }
  }
  else sev = 0; // If expanded modul

  return sev;
}

double CubeCache::get_vssev (int mode[], int met_id, int reg_id, int loc_id) {
  double val1, val2;
  mode[CUBE_PROG] = CUBE_INCL;
  val1 = get_vrsev(mode, met_id, reg_id, loc_id);
  mode[CUBE_PROG] = CUBE_EXCL;
  val2 = get_vrsev(mode,met_id, reg_id, loc_id);
  return val1-val2;
}

void CubeCache::update_func_cache(int met_state, int met_id) {
  Region* r;
  int rid;
  for (int k = 0; k < cube->num_regs(); k++) {
    r = (Region* )cube->get_region(k);
    if (!r->get_name().compare("virtual caller")) continue;
    rid = r->get_id();
    /*cnode_buf_ex[rid] = sev4func_excl(met_state, CUBE_EXCL, CUBE_INCL, 
				 met_id, rid, 0);
    cnode_buf_in[rid] = sev4func_incl(met_state, CUBE_INCL, CUBE_INCL,
    met_id, rid, 0);*/
    int mode[3];
    mode[CUBE_MET] = met_state;
    mode[CUBE_PROG] = CUBE_EXCL;
    mode[CUBE_LOC]  = -1;
    cnode_buf_ex[rid] = get_vrsev(mode, met_id, rid);
    mode[CUBE_PROG] = CUBE_INCL;
    cnode_buf_in[rid] = get_vrsev(mode, met_id, rid);
  }
  old_met_state = met_state;
  old_met_id = met_id;
}

void CubeCache::update_loc_cache4func(int met_state, int func_state,
				   int met_id, int func_id, int rootid) {
  Location* root = (Location*) cube->get_loc(rootid);
  int childnum = root->num_children();
  double sum = 0.0;
  // refresh the flags for the location cache
  old_met_id = met_id;
  old_met_state = met_state;
  old_cnode_id = func_id;
  old_cnode_state = func_state;
  loc_cache_mod  = isModule;
  loc_cache_sub  = isSubroutine;

  int mode[3];
  mode[CUBE_MET] = met_state;
  mode[CUBE_PROG] = func_state;
  mode[CUBE_LOC]  = CUBE_EXCL;
  sum = get_vrsev(mode, met_id, func_id, rootid);
  if (childnum == 0) {
    // leaf tree node
    loc_buf_ex[rootid] = sum;
    loc_buf_in[rootid] = sum;
    return;
  }   
  // internal tree node
  loc_buf_ex[rootid] = sum;
  mode[CUBE_LOC] = CUBE_INCL;
  loc_buf_in[rootid] = get_vrsev(mode, met_id, func_id, rootid);
  for (int i = 0; i < childnum; i++) {
    int child_id = ((Location*) root->get_child(i))->get_locid();
    update_loc_cache4func(met_state, func_state, met_id, func_id, child_id);
  }
}

void CubeCache::update_loc_cache4mod(int met_state, int mod_state,
				int met_id, int mod_id, int rootid) {
  Location* root = (Location*) cube->get_loc(rootid);
  int childnum = root->num_children();
  double sum = 0.0;
  // refresh the flags for the location cache
  old_met_id = met_id;
  old_met_state = met_state;
  old_cnode_id = mod_id;
  old_cnode_state = mod_state;
  loc_cache_mod  = isModule;
  loc_cache_sub  = isSubroutine;
  int mode[3];
  mode[CUBE_MET] = met_state;
  mode[CUBE_PROG] = mod_state;
  mode[CUBE_LOC]  = CUBE_EXCL;
  sum = get_vmsev(mode, met_id, mod_id, rootid);
  if (childnum == 0) {
    // leaf tree node
    loc_buf_ex[rootid] = sum;
    loc_buf_in[rootid] = sum;
    return;
  }   
  // internal tree node
  loc_buf_ex[rootid] = sum;
  mode[CUBE_LOC] = CUBE_INCL;
  loc_buf_in[rootid] = get_vmsev(mode, met_id, mod_id, rootid);
  for (int i = 0; i < childnum; i++) {
    int child_id = ((Location*) root->get_child(i))->get_locid();
    update_loc_cache4mod(met_state, mod_state, met_id, mod_id, child_id);
  }
}

void CubeCache::update_loc_cache4sub(int met_state,
				     int met_id, int reg_id, int rootid) {
  Location* root = (Location*) cube->get_loc(rootid);
  int childnum = root->num_children();
  double sum = 0.0;
  // refresh the flags for the location cache
  old_met_id = met_id;
  old_met_state = met_state;
  old_cnode_id = reg_id;
  // old_cnode_state = ;
  loc_cache_mod  = isModule;
  loc_cache_sub  = isSubroutine;
  int mode[3];
  mode[CUBE_MET] = met_state;
  mode[CUBE_PROG] = CUBE_EXCL; // not useful
  mode[CUBE_LOC]  = CUBE_EXCL;
  sum = get_vssev(mode, met_id, reg_id, rootid);
  if (childnum == 0) {
    // leaf tree node
    loc_buf_ex[rootid] = sum;
    loc_buf_in[rootid] = sum;
    return;
  }   
  // internal tree node
  loc_buf_ex[rootid] = sum;
  mode[CUBE_LOC] = CUBE_INCL;
  loc_buf_in[rootid] = get_vssev(mode, met_id, reg_id, rootid);
  for (int i = 0; i < childnum; i++) {
    int child_id = ((Location*) root->get_child(i))->get_locid();
    update_loc_cache4sub(met_state, met_id, reg_id, child_id);
  }
}

/*
void Cube::update_loc_cache4func(int met_state, int func_state,
				   int met_id, int func_id, int rootid) {
  // cout << "update function location cache" << endl;
  Location* root = (Location*) get_loc(rootid);
  int childnum = root->num_children();
  double sum = 0.0;
  // refresh the flags for the location cache
  old_met_id = met_id;
  old_met_state = met_state;
  old_cnode_id = func_id;
  old_cnode_state = func_state;

  if (func_state == CUBE_EXCL) {
    sum = sev4func_excl(met_state, CUBE_EXCL, CUBE_EXCL, met_id, func_id, rootid);
  } else {
    sum = sev4func_incl(met_state, CUBE_INCL, CUBE_EXCL, met_id, func_id, rootid);
  }   
  if (childnum == 0) {
    // leaf tree node
    loc_buf_ex[rootid] = sum;
    loc_buf_in[rootid] = sum;
    return;
  }   
  // internal tree node
  loc_buf_ex[rootid] = sum;
  if (func_state == CUBE_INCL)
    loc_buf_in[rootid] = sev4func_incl(met_state, CUBE_INCL, CUBE_INCL,
				       met_id, func_id, rootid);
  else
    loc_buf_in[rootid] = sev4func_excl(met_state, CUBE_EXCL, CUBE_INCL,
				       met_id, func_id, rootid);
  for (int i = 0; i < childnum; i++) {
    int child_id = ((Location*) root->get_child(i))->get_locid();
    update_loc_cache4func(met_state, func_state, met_id, func_id, child_id);
  }
}
*/
double CubeCache::sev4func_excl(int met_state, int func_state, int loc_state,
			   int met_id, int func_id, int locid) {
  double sum = 0.0;
  Location* loc = (Location*) cube->get_loc(locid);
  map<int, Cnode*> cnodev = cube->get_cnodev();
  // find all the cnodes whose callee is just the region f.
  for (int i = 0; i < cnodev.size(); i++) {
    Cnode* cnode = cnodev[i];
    Callsite* csite = cnode->get_csite();
    Region* callee = (Region*) csite->get_callee();
    if (callee->get_id() == func_id) {
      int n = cnode->get_id();
      sum += sev(met_state, CUBE_EXCL, loc_state,
		 met_id, n, locid);
    }
  }
  return sum;
}

double CubeCache::sev4func_incl(int met_state, int func_state, int loc_state,
			   int met_id, int func_id, int locid) {
  double sum = 0.0;
  Location* loc = (Location*) cube->get_loc(locid);
  map<int, Cnode*> cnodev = cube->get_cnodev();
  // find all the cnodes whose callee is just the region f.
  for (int i = 0; i < cnodev.size(); i++) {
    Cnode* cnode = cnodev[i];
    Callsite* csite = cnode->get_csite();
    Region* callee = (Region*) csite->get_callee();
    if (callee->get_id() == func_id) {
      int n = cnode->get_id();
      sum += sev(met_state, CUBE_INCL, loc_state,
		 met_id, n, locid);
    }
  }
  return sum;
}
