/****************************************************************************
**  Copyright (C) 2004                                                     **
**  University of Tennessee, Innovative Computing Laboratory               **
**                                                                         **
**  See the file COPYRIGHT in the package base directory for details       **
****************************************************************************/


#ifndef _CUBE_H
#define _CUBE_H   

/*
 *----------------------------------------------------------------------------
 *
 * class Cube
 *
 *----------------------------------------------------------------------------
 */

#include <iostream>
#include <string>
#include <vector>
#include <map>

#include "CUBException.h"

class Callsite;
class Cnode;
class CubeCache;
class Grid;
class Location;
class Machine;
class Module;
class Node;
class Process;
class Metric;
class Region;
class Thread;


#define CUBE_INCL    0
#define CUBE_EXCL    1

#define CUBE_MET    0
#define CUBE_PROG    1
#define CUBE_LOC     2

/* id manangement: domain types */
enum {
  METRIC,
  MODULE,
  REGION,
  CALLSITE,
  CALLNODE,
  GRID,
  MACHINE,
  NODE,
  PROCESS,
  THREAD,
  LOCATION
};


class Cube {
 public:

  Cube();
  Cube(const Cube& cube);

  ~Cube();
  
  /***************************************
   * write interface 
   ***************************************/

  // The order of children in any hierarchical structure is defined by
  // the order of child creation. For example, the first performance metric
  // created as child of a given metric is the first child.

  // All of the functions below return an id that uniquely
  // distinguishes the created objects from other objects of the same
  // kind (i.e., each object type has its own namespace). An id -1
  // represents a void reference. All id namespaces include the
  // non-negative numbers from zero to n-1, where n is the number of
  // objects.

  // performance metric dimension
  int def_met(std::string name, std::string uom, std::string descr, int parent_id);   
  
  // call-tree dimension
  int def_module(std::string name);
  int def_region(std::string name, long begln, long endln, std::string descr, int mod_id);
  int def_csite(int mod_id, int line, int callee_id);
  int def_cnode(int csite_id, int parent_id);
  
  // location dimension
  int def_grid(std::string name);
  int def_mach(std::string name, int grid_id);
  int def_node(std::string name, int mach_id);
  int def_proc(std::string name, int node_id);
  int def_thrd(std::string name, int proc_id);

  // define severity values
  void set_sev(int met_id, int cnode_id, int thrd_id, double value);
  void add_sev(int met_id, int cnode_id, int thrd_id, double value);
  void sub_sev(int met_id, int cnode_id, int thrd_id, double value);

  // writes data to a file
  void write(std::string path);
  
  /***************************************
   * read interface 
   ***************************************/

  // returns the number of objects that have been defined
  int num_mets()       { return metv.size(); }
  int num_mods()       { return modv.size(); } 
  int num_regs()       { return regionv.size(); }  
  int num_csites()     { return csitev.size(); } 
  int num_cnodes()     { return cnodev.size(); }
  int num_grids()      { return gridv.size(); }
  int num_machs()      { return machv.size(); }
  int num_nodes()      { return nodev.size(); }
  int num_procs()      { return procv.size(); }
  int num_thrds()      { return thrdv.size(); }
  int num_locs()       { return locs.size(); }

  // returns the object itself
  const Metric*   get_met(int met_id); 
  const Module*   get_mod(int mod_id);
  const Region*   get_region(int func_id);
  const Callsite* get_csite(int csite_id);
  const Cnode*    get_cnode(int cnode_id);
  // const Location* get_loc(int loc_id);
  const Grid*     get_grid(int grid_id);
  const Machine*  get_mach(int mach_id);
  const Node*     get_node(int node_id);
  const Process*  get_proc(int proc_id);
  const Thread*   get_thrd(int thrd_id);
  const Location* get_loc(int loc_id);

  //get cube xml version
  float get_version()                    { return version; }
  void  set_version(float v)             { version = v; }

  /* returns the roots of each forest */
  std::vector<Metric*>   get_met_roots()      { return rmetv; }
  std::vector<Module*>   get_mod_roots()      { return rmodv; }
  std::vector<Cnode*>    get_cnode_roots()    { return rcnodev; }
  std::vector<Grid*>     get_loc_roots()      { return rgridv; }




  /****************************************************************
   *  Interface between cube and GUI.
   *
   *  GUI part calls this method to update 3-dimensional trees  
   ****************************************************************/
  double get_vsev(int mode[], int met_id, int prog_id = -1, int loc_id = -1,
		  bool iscalltree = true, bool isModule = false, bool isSub = false);





  

  /**************************************************************
   * 
   *       Following methods are NOT used by Cube users!
   * 
   *************************************************************/ 
  
 public:

  friend class CUBEXMLParser;
  friend class CubeCache;

  /* output severity matrix to XML document */
  void   print_severity (std::ostream& out);


  /*********************************************************
   * used by  the GUI part to access Cube
   ********************************************************/

  // implements  max { n | n is thread number of each process }
  int    get_max_thrdnum ();

  // for different metric trees in the metric forest
  void   compute_max ();        // called once for each metric tree's max
  double get_max ();            // dependent on the selected metric node in GUI
  void   set_cur_met (int pid)         { cur_metid = pid; }


  /***********************************************************************
   * Convert a severity value to a percentage.
   * met_id is used to distinguish metric vertex from other vertices. 
   **********************************************************************/
  int    index (double val, int view=0, double refval=0, int met_id = -1);
  

  //generate labels for different trees
  std::string gen_met_label      (int met_id, double sev, int view = 0, 
				  double refval = 0, bool default_root=true);
  std::string gen_cnode_label    (int met_id, int cnode_id, double sev, 
				  int view = 0, double refval = 0, 
				  bool default_root = true);
  std::string gen_loc_label      (int met_id, int loc_id, 
				  double sev, int view = 0, 
				  double refval = 0, bool default_root=true);
  std::string gen_function_label (int met_id, int func_id,
				  double sev, int view = 0,
				  double refval = 0, bool default_root = true);
  std::string gen_mod_label      (int met_id, int mod_id, double sev, 
				  int view = 0, double refval = 0, 
				  bool default_root = true);
  std::string gen_func_children_label (int met_id, double sev, int view = 0,
				       double refval = 0, bool default_root = true);
  
  // convert a severity value to a mode-dependent value to display.
  double sev4view (double sev, int view, double refval = 0, int met_id = -1);

  // generate a max label displayed in spectrum pane.
  std::string gen_max_label (double max_val);
  std::string gen_half_max_label (double max_val);


 private:
  
  /*********************************************
   *        Private     Methods  
   *
   *********************************************/

  double get_sev(int met_id, int cnode_id, int thrd_id);
  double get_max(int pid);     // for a given metric vertex
  int    get_root(int id);     // get an arbitrary metric's root id

  /*************************************************************
   *  id allocation method:
   *
   * Domain types: METRIC, MODULE, REGION, CALLSITE,
   * CALLNODE, GRID, MACHINE, NODE, PROCESS, THREAD, LOCATION.
   *
   **************************************************************/
  int        allocate_id(int domain);
  void       validate_id(int id, int domaintype); // not being used now.

 
  /* used by CubeCache */
  double      get_esev    (int met_id, int cnode_id, int loc_id);
  std::map<int, Cnode*>   get_cnodev()   { return cnodev; }
  


  /***************************************************************
   *   methods used by  CubeXMLParser (friend class)
   *
   * to create Cube objects.
   * an object's id is already specified before creation. 
   *
   ***************************************************************/

  // metric dimension
  Metric*  create_met(int id, std::string name, std::string uom, std::string des,
		      int parent_id);

  // call-tree dimension
  Module*    create_module(int id, std::string name);
  Region*    create_region(int id, std::string name, long begln, long endln, 
			   std::string descr, int mod_id);
  Callsite*  create_csite(int id, int mod_id, int line, int callee_id, int caller_id);
  Cnode*     create_cnode(int id, int csite_id, int parent_id);

  // location dimension
  Grid*      create_grid(int loc_id, int id, std::string name);
  Machine*   create_mach(int loc_id, int id, std::string name, int grid_id);
  Node*      create_node(int loc_id, int id, std::string name, int mach_id);
  Process*   create_proc(int loc_id, int id, std::string name, int node_id);
  Thread*    create_thrd(int loc_id, int id, std::string name, int proc_id);


  /*****************************************************
   *        Private Data Members
   *
   *****************************************************/

  CubeCache* cache;

  /* maximum id in different domains */
  int met_id;
  int mod_id;
  int reg_id;
  int csite_id;
  int cnode_id;
  int grid_id;
  int mach_id;
  int node_id;
  int proc_id;
  int thrd_id;
  int loc_id;

  int cur_metid; // selected metric's id in GUI

  /* CUBE XML version, default value=0.1 */
  float version;
  std::map<int, double>    max_sevs; // the value of "Total"

  // vectors (forests) of root nodes
  std::vector<Metric*>     rmetv;
  std::vector<Cnode*>      rcnodev;
  std::vector<Module*>     rmodv;
  std::vector<Grid*>       rgridv;

  /* each entity has a corresponding map */
  std::map<int, Metric*>   metv;    /* all metrics */  
  std::map<int, Module*>   modv;    /* all modules */
  std::map<int, Region*>   regionv; /* all regions */
  std::map<int, Callsite*> csitev;  /* all call sites */
  std::map<int, Cnode*>    cnodev;  /* all call nodes */

  std::map<int, Grid*>     gridv;   /* all grids */
  std::map<int, Machine*>  machv;   /* all machines */
  std::map<int, Node*>     nodev;   /* all nodes */
  std::map<int, Process*>  procv;   /* all processes */
  std::map<int, Thread*>   thrdv;   /* all threads */
  std::map<int, Location*> locs;    /* all locations */


  std::map<int, std::map<int, std::map<int, double> > > severity;

  std::map<int, int> NA_callees;/* not available callees when parsing an XML doc */

  // performance algebra operators
  friend Cube& operator-= (Cube& cube1, Cube& cube2);

};

// CUBE input/output operators
std::istream& operator>>(std::istream& in, Cube& cb);
std::ostream& operator<<(std::ostream& out, Cube& cb);



// performance algebra operators (after version 1.0)
Cube& operator-= (Cube& cube1, Cube& cube2);


#endif






