/****************************************************************************
**  Copyright (C) 2004                                                     **
**  University of Tennessee, Innovative Computing Laboratory               **
**                                                                         **
**  See the file COPYRIGHT in the package base directory for details       **
****************************************************************************/

/************************************************
             Cube.cpp

 ************************************************/

#include <iostream>
#include <strstream>
#include <fstream>
#include <exception>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>

#include "Metric.h"
#include "CUBException.h"

#include "Module.h"
#include "Region.h"
#include "Callsite.h"
#include "Cnode.h"

#include "Grid.h"
#include "Machine.h"
#include "Node.h"
#include "Process.h"
#include "Thread.h"

#include "CUBEXMLParser.h"
#include "CubeMatrix.h"

#include "Cube.h"
#include "CubeCache.h"

using namespace std;

/* constructor */
Cube::Cube() {
  met_id  = 0;
  mod_id   = 0;
  reg_id   = 0;
  csite_id = 0;
  cnode_id = 0;
  grid_id  = 0;
  mach_id  = 0;
  node_id  = 0;
  proc_id  = 0;
  thrd_id  = 0;
  loc_id   = 0;

  version  = 1.0;

  cache    = NULL;
  cur_metid = -1;
}


// When ~Cube() is called, destruct deeply the objects stored in maps.
template<class T> 
void delete_map(map<int, T> & entitymap) {
  typename map<int, T>::iterator i;
  for (i = entitymap.begin(); i != entitymap.end(); i++) {
    delete i->second;
  }
}

/* destructor */
Cube::~Cube() {
  delete cache;
  delete_map<Metric*>(metv);
  delete_map<Module*>(modv);
  delete_map<Region*>(regionv);
  delete_map<Callsite*>(csitev);
  delete_map<Cnode*>(cnodev);
  delete_map<Location*>(locs);
}


int Cube::allocate_id(int domain) {
  switch (domain) {
    case METRIC:
      return met_id++;
    case MODULE:
      return mod_id++;
    case REGION:
      return reg_id++;
    case CALLSITE:
      return csite_id++;
    case CALLNODE:
      return cnode_id++;
    case GRID:
      return grid_id++;
    case MACHINE:
      return mach_id++;
    case NODE:
      return node_id++;
    case PROCESS:
      return proc_id++;
    case THREAD:
      return thrd_id++;
    case LOCATION:
      return loc_id++;
    default:
      throw CUBException("domain type error"); // wrong domain type
  }
}

/*****************************************
 *   write interface
 *****************************************/

/* performance metrics dimension */
int Cube::def_met(string name, string uom, string descr, int parent_id) {
  Metric* met;
  Metric* parent;
  int id;

  try {
    /* get the parent metric */
    if (parent_id == -1) parent = NULL;
    else { parent = (Metric*) get_met(parent_id); }

    /* get an id for the new metric */
    id = allocate_id(METRIC);
    /* create a new metric */
    met = new Metric(id, name, uom, descr, parent);

    /* whether it's a root */
    if(parent_id == -1) rmetv.push_back(met);
    else parent->add_child(met);

    // metv.push_back(met); /* append it to vector */
    metv[id] = met;
    return id;    
  } catch(CUBException excep) {
    cerr << excep;
    exit(1);
  }

}


Metric* Cube::create_met(int id, string name, string uom, 
			 string descr, int parent_id){
  Metric* met;
  Metric* parent;

  try {
    /* get the parent metric */
    if (parent_id == -1) parent = NULL;
    else { parent = (Metric*) get_met(parent_id); }

    /* create a new metric */
    met = new Metric(id, name, uom, descr, parent);

    /* whether it's a root */
    if(parent_id == -1) rmetv.push_back(met);
    else parent->add_child(met);

    //metv.push_back(met); /* append it to vector */
    metv[id] = met;
    return met;    
  } catch(CUBException except) {
    cerr << except;
    exit(1);
  }
}


/* call tree dimension */

int Cube::def_module(string name) {  
  Module* mod;
  int id = allocate_id(MODULE);
  mod = new Module(id, name);
  rmodv.push_back(mod);
  modv[id] = mod;
  return id;

}

/* get a Module from module vector
Module* Cube::get_mod(string name, string path) {
  vector<Module*>::iterator i;
  for (i = modv.begin(); i != modv.end(); i++) {
    if((*i)->get_path() == path && (*i)->get_name() == name)
      return (*i);
  }
  return NULL;
}
*/

Module* Cube::create_module(int id, string name) {
  Module* mod;
  mod = new Module(id, name);
  rmodv.push_back(mod);
  modv[id] = mod;
  return mod;
}

int Cube::def_region(string name, long beginln, long endln, string descr, 
		     int modid) {
  Module* mod;
  Region* reg;
  int id;

  try {
    mod = (Module* )get_mod(modid);
    /* new a region object */
    id = allocate_id(REGION);
    reg = new Region(id, name, mod, beginln, endln, descr);
    // regionv.push_back(reg);
    regionv[id] = reg;
    mod->add_region(reg);
    return id;
  } catch(CUBException except) {
    cerr << except;
    exit(1);
  }
}

Region* Cube::create_region(int id, string name, long beginln, long endln, 
			    string descr, int modid) {
  Module* mod;
  Region* reg;

  try {
    mod = (Module* )get_mod(modid);
    /* new a region object */
    reg = new Region(id, name, mod, beginln, endln, descr);
    // regionv.push_back(reg);
    regionv[id] = reg;
    mod->add_region(reg);
    // make up for the callsites
    map<int, int>::iterator i;
    for (i = NA_callees.begin(); i != NA_callees.end(); ++i) {
      //cout << "csite: " << i->first << " callee: " << i->second << endl;
      if (i->second == id) {
	Callsite* csite = (Callsite*) get_csite(i->first);
	csite->set_callee(reg);
	NA_callees[i->first] = -1;
	//NA_callees.erase(i->first); BUG in c++. don't use it!
      }
    }
    return reg;
  } catch(CUBException ex) {
    cerr << ex;
    exit(1);
  }
}

int Cube::def_csite(int modid, int line, int callee_id) {
  // Module* mod;
  Region *caller, *callee;
  Callsite* csite;
  int id;
  
  /* new a call site object */
  try {
    // mod = (Module*) get_mod(modid);
    /* caller will be defined in the def_cnode() */
    // caller = find_caller_region(mod, line);
    caller = NULL;
    callee = (Region* )get_region(callee_id);
    id = allocate_id(CALLSITE);
    csite = new Callsite(id, line, caller, callee);
    // caller->add_csite(csite);
    //csitev.push_back(csite);
    csitev[id] = csite;
    return id;
  } catch (CUBException ex) {
    cerr << ex;
    exit(1);
  }  
}

Callsite* Cube::create_csite(int id, int modid, int line, 
			     int callee_id, int caller_id) {
  // Module* mod;
  Region *caller, *callee;
  Callsite* csite;
  
  /* new a call site object */
  try {
    // mod = (Module*) get_mod(modid);
    caller = (Region*) get_region(caller_id);
    try {
      callee = (Region* )get_region(callee_id);
    } catch (CUBException) {
      // cout << "when parsing a csite, callee not found" << endl;
      callee = NULL;
      NA_callees[id] = callee_id; // csite id needs a callee(callee_id).
    }
    csite = new Callsite(id, line, caller, callee);
    caller->add_csite(csite);
    csitev[id] = csite;
    return csite;
  } catch (CUBException ex) {
    cerr << ex;
    exit(1);
  }  
}

int Cube::def_cnode(int csite_id, int parent_id) {
  Callsite* csite;
  Cnode* parent;
  Cnode* cnode;
  int id;

  try {
    csite = (Callsite* ) get_csite(csite_id);
    if (parent_id == -1) parent = NULL; // root cnode
    else parent = (Cnode* )get_cnode(parent_id);
    id = allocate_id(CALLNODE); 
    cnode = new Cnode(id, csite, parent);
    csite->add_cnode(cnode);
    if (parent_id == -1) {
      rcnodev.push_back(cnode); // root node
      Module* m = (Module*)csite->get_mod();
      int vid = def_region("virtual caller", 0, 0, "not existing", 
			   m->get_id());
      Region* caller = (Region*) get_region(vid);
      csite->set_caller(caller);
      caller->add_csite(csite);
    }
    else {
      parent->add_child(cnode);
      Callsite* psite = parent->get_csite();
      Region* caller = (Region*) psite->get_callee(); // but current one's caller
      csite->set_caller(caller);
      caller->add_csite(csite);
    }
    // cnodev.push_back(cnode);
    cnodev[id] = cnode;
    return id;
  } catch(CUBException ex){
    cerr << ex;
    exit(1);
  }  
}

Cnode* Cube::create_cnode(int id, int csite_id, int parent_id) {
  Callsite* csite;
  Cnode* parent;
  Cnode* cnode;

  try {
    csite = (Callsite* ) get_csite(csite_id);
    if (parent_id == -1) parent = NULL; // root cnode
    else parent = (Cnode* )get_cnode(parent_id);
    cnode = new Cnode(id, csite, parent);
    csite->add_cnode(cnode);
    if (parent_id == -1) rcnodev.push_back(cnode);
    else parent->add_child(cnode); // root entity
    cnodev[id] = cnode;
    return cnode;
  } catch(CUBException ex) {
    cerr << ex;
    exit(1);
  }  
}

/* location dimension */

int Cube::def_grid(string name) {
  int loc_id, id;
  Grid* grid;
  
  loc_id = allocate_id(LOCATION);
  id = allocate_id(GRID);
  grid = new Grid(loc_id, id, name);
  rgridv.push_back(grid);
  gridv[id] = grid;
  locs[loc_id] = grid;
  return id;  
}

Grid* Cube::create_grid(int loc_id, int id, string name) {
  Grid* grid;
  
  grid = new Grid(loc_id, id, name);
  rgridv.push_back(grid);
  gridv[id] = grid;
  locs[loc_id] = grid;
  return grid;  
}

int Cube::def_mach(string name, int grid_id) {
  Grid* grid;
  Machine* m;
  int loc_id, id;
  
  try{
    grid = (Grid*) get_grid(grid_id);
    loc_id = allocate_id(LOCATION);
    id = allocate_id(MACHINE);
    m = new Machine(loc_id, id, name, grid);
    grid->add_mach(m);
    machv[id] = m;
    locs[loc_id] = m;
    return id;
  }catch(CUBException e) {
    cerr << e;
    exit(1);
  }
}

Machine* Cube::create_mach(int loc_id, int id, string name, int grid_id) {
  Grid* grid;
  Machine* m;
  
  try{
    grid = (Grid*) get_grid(grid_id);
    m = new Machine(loc_id, id, name, grid);
    grid->add_mach(m);
    machv[id] = m;
    locs[loc_id] = m;
    return m;
  }catch(CUBException e) {
    cerr << e;
    exit(1);
  }
}

int Cube::def_node(string name, int mach_id) {
  Machine* m;
  Node* node;
  int loc_id, id;
  
  try{
    m = (Machine*) get_mach(mach_id);
    loc_id = allocate_id(LOCATION);
    id = allocate_id(NODE);
    node = new Node(loc_id, id, name, m);
    m->add_node(node);
    nodev[id] = node;
    locs[loc_id] = node;
    return id;
  }catch(CUBException e) {
    cerr << e;
    exit(1);
  }
}

Node* Cube::create_node(int loc_id, int id, string name, int mach_id) {
  Machine* m;
  Node* node;
  
  try{
    m = (Machine*) get_mach(mach_id);
    node = new Node(loc_id, id, name, m);
    m->add_node(node);
    nodev[id] = node;
    locs[loc_id] = node;
    return node;
  }catch(CUBException e) {
    cerr << e;
    exit(1);
  }
}

int Cube::def_proc(string name, int node_id) {
  Node* node;
  Process* proc;
  int loc_id, id;
  
  try{
    node = (Node*) get_node(node_id);
    loc_id = allocate_id(LOCATION);
    id = allocate_id(PROCESS);
    proc = new Process(loc_id, id, name, node);
    node->add_proc(proc);
    procv[id] = proc;
    locs[loc_id] = proc;
    return id;
  }catch(CUBException e) {
    cerr << e;
    exit(1);
  }
}

Process* Cube::create_proc(int loc_id, int id, string name, int node_id) {
  Node* node;
  Process* proc;
  
  try{
    node = (Node*) get_node(node_id);
    proc = new Process(loc_id, id, name, node);
    node->add_proc(proc);
    procv[id] = proc;
    locs[loc_id] = proc;
    return proc;
  }catch(CUBException e) {
    cerr << e;
    exit(1);
  }
}


int Cube::def_thrd(string name, int proc_id) {
  Thread* thrd;
  Process* proc;
  int loc_id, id;
  
  try{
    proc = (Process*) get_proc(proc_id);
    loc_id = allocate_id(LOCATION);
    id = allocate_id(THREAD);
    thrd = new Thread(loc_id, id, name, proc);
    proc->add_thrd(thrd);
    thrdv[id] = thrd;
    locs[loc_id] = thrd;
    return id;
  }catch(CUBException e) {
    cerr << e;
    exit(1);
  }
}

Thread* Cube::create_thrd(int loc_id, int id, string name, int proc_id) {
  Thread* thrd;
  Process* proc;
  
  try{
    proc = (Process*) get_proc(proc_id);
    thrd = new Thread(loc_id, id, name, proc);
    proc->add_thrd(thrd);
    thrdv[id] = thrd;
    locs[loc_id] = thrd;
    return thrd;
  }catch(CUBException e) {
    cerr << e;
    exit(1);
  }
}

/* currently, not being used. Maybe later? */
void Cube::validate_id(int id, int domaintype) {
  if (id < 0) throw CUBException("invalid id value < 0");
  switch(domaintype) {
  case METRIC:
    if (id > met_id) throw CUBException("too big metric id");
    break;
  case CALLNODE:
    if (id > cnode_id) throw CUBException("too big call-tree node id");
    break;
  case THREAD:
    if (id > thrd_id) throw CUBException("too big thread id");
    break;
  case LOCATION:
    if (id > loc_id) throw CUBException("too big location id");
    break;
  }
}

/***********************************************************************
    Severity Calculation Methods

  severity matrix, define values 
************************************************************************/  
void Cube::set_sev(int met_id, int cnode_id, int thrd_id, double value) {
  severity[met_id][cnode_id][thrd_id] = value;
}

void Cube::add_sev(int met_id, int cnode_id, int thrd_id, double value) {
  severity[met_id][cnode_id][thrd_id] += value;
}

void Cube::sub_sev(int met_id, int cnode_id, int thrd_id, double value) {
  severity[met_id][cnode_id][thrd_id] -= value;
}

// get a simple severity value
double Cube::get_sev(int met_id, int cnode_id, int thrd_id) {
  return severity[met_id][cnode_id][thrd_id];
}

// get an extended severity value
double Cube::get_esev(int met_id, int cnode_id, int loc_id) {
  map<int, Thread*>::iterator i;
  Thread* t;
  // see whether a location is a thread
  for (i = thrdv.begin(); i != thrdv.end(); i++) {
    if (i->second->get_locid() == loc_id) break;
  }
  // not a thread, return 0.
  if (i == thrdv.end()) return 0;
  t = i->second;;
  int thrd_id = t->get_id();
  return get_sev(met_id, cnode_id, thrd_id); 
}

double Cube::get_vsev(int mode[], int met_id, 
		      int prog_id, int loc_id,
		      bool iscalltree, bool isModule, 
		      bool isSub) {
  if (cache == NULL) cache = new CubeCache(this);
  cache->set_flag(CubeCache_Mod_Flag, isModule);
  cache->set_flag(CubeCache_Sub_Flag, isSub);
  return cache->get_vsev(mode, met_id, prog_id, loc_id,
			 iscalltree);

}

/*****************************************
 *   read interface
 *****************************************/

const Metric* Cube::get_met(int met_id) {
  Metric* p = metv[met_id];
  if (p == NULL) throw CUBException("invalid metric id");
  return p;  
} 

const Module* Cube::get_mod(int mod_id) {
  Module* m = modv[mod_id];
  if (m == NULL) throw CUBException("invalid module id");
  return m;  
}

const Region* Cube::get_region(int reg_id) {
  Region* r = regionv[reg_id];
  if (r == NULL) {
    throw CUBException("invalid region id");
  }
  return r; 
}

const Callsite* Cube::get_csite(int csite_id) {
  Callsite* cs = csitev[csite_id];
  if (cs == NULL) throw CUBException("invalid call site id");
  return cs; 
}

const Cnode* Cube::get_cnode(int cnode_id) {
  Cnode* cn = cnodev[cnode_id];
  if (cn == NULL) throw CUBException("invalid call-tree node id");
  return cn; 
}

const Grid* Cube::get_grid(int grid_id) {
  Grid* g = gridv[grid_id];
  if (g == NULL) throw CUBException("invalid grid id");
  return g; 
}

const Machine* Cube::get_mach(int mach_id) {
  Machine* m = machv[mach_id];
  if (m == NULL) throw CUBException("invalid machine id");
  return m; 
}

const Node* Cube::get_node(int node_id) {
  Node* n = nodev[node_id];
  if (n == NULL) throw CUBException("invalid node id");
  return n; 
}

const Process* Cube::get_proc(int proc_id) {
  Process* p = procv[proc_id];
  if (p == NULL) throw CUBException("invalid process id");
  return p; 
}

const Thread*   Cube::get_thrd(int thrd_id) {
  Thread* t = thrdv[thrd_id];
  if (t == NULL) throw CUBException("invalid thread id");
  return t; 
}

const Location*  Cube::get_loc(int loc_id) {
  Location* l = locs[loc_id];
  if (l == NULL) throw CUBException("invalid location id");
  return l; 
}




void Cube::print_severity(ostream& out) {
  map<int, Map2D>::iterator i;
  Map2D::iterator j;
  // map<int, double>::iterator k;

  Map2D map2d;
  map<int, double> map1d;
  out.precision(6);
  out.setf(ios::fixed);
  for (i = severity.begin(); i != severity.end(); i++) {
    out << "    <matrix metricId=\"" << i->first; // 4 space indent
    out << "\">" << endl;
    map2d = i->second;
    for (j = map2d.begin(); j!= map2d.end(); j++) {
      out << "      <row cnodeId=\"" << j->first; // 6 space indent
      out << "\">" <<endl;
      map1d = j->second;
      /*for (k = map1d.begin(); k!= map1d.end(); k++) {
	out << "        " << k->second << endl; // 8 space indent
	}*/
      for (int k = 0; k < thrdv.size(); k++) {
	out << "        " <<  map1d[thrdv[k]->get_id()] << endl; // 8 space indent
      }
      out <<"      </row>" << endl;
    }
    out << "    </matrix>" << endl;
  }
}

/******************************************************* 
  Write Cube objects from Memory into XML documents 

********************************************************/
std::ostream& operator<<(std::ostream& out, Cube& cb) {
  int i, level;
  float version = cb.get_version();
  vector<Metric*>   pv = cb.get_met_roots();
  vector<Module*>   mv = cb.get_mod_roots();
  vector<Cnode*>    cv = cb.get_cnode_roots();
  vector<Grid*> lv = cb.get_loc_roots();

  // write XML document head 
  out << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << endl << endl;
  out << "<cube " << "version=\"" << version <<"\">" << endl;  
  level = 2;

  // write <behavior> element
  out << "  <behavior>"   << endl;
  for(i = 0; i < pv.size() ; i++) {
    pv[i]->set_level(level);   
    out << *pv[i];
  }
  out << "  </behavior>"  << endl;
  
  // write <program> element
  out << "  <program>"    << endl;  
  for(i = 0; i < mv.size(); i++) {
    mv[i]->set_level(level);
    out << *mv[i];
  }
  for(i = 0; i < cv.size(); i++) {
    cv[i]->set_level(level);
    out << *cv[i]; 
  }
  out << "  </program>"   << endl;
  
  // write <locations> element
  out << "  <locations>"  << endl;
  for(i = 0; i < lv.size(); i++) {
    lv[i]->set_level(level);
    out << *lv[i];
  }
  out << "  </locations>" << endl;
  
  // write <severity> element
  out << "  <severity>"   << endl;
  cb.print_severity(out);
  out << "  </severity>"  << endl;
  
  out << "</cube>"      << endl;
  
  return out;
} 

void Cube::write(std::string path) {
  ofstream out;
  out.open(path.c_str());
  if (!out.good()) {
    cout << "Open file error: " << path << endl;
    exit(1);
  }
  out << *this;
  out.close();

}

/* compute all the max values for all the metric trees */
void Cube::compute_max() {
  Metric* root;
  int mode[] = {CUBE_INCL, -1, -1};
  for (int i = 0; i < rmetv.size(); i++) {
    root = rmetv[i];
    int id = root->get_id();
    max_sevs[id] = get_vsev(mode, id);
  }
}

// only used for metric dimension
int Cube::get_root(int id) {
  Metric* p = (Metric*)get_met(id);
  while(p->get_parent() != NULL) 
    p = (Metric*) p->get_parent();
 
  return p->get_id();
}

/* given a pid, decide its locating tree's max value */
double Cube::get_max(int pid) {
  if (pid < 0) return 0;
  int rootid = get_root(pid);
  return max_sevs[rootid];
}
// get the max for the selected metric tree
double Cube::get_max() {
  if (cur_metid < 0) return 0;
  return get_max(cur_metid);
}
/**********************************************************
   Read an XML document to create a Cube object in Memory 

**********************************************************/
istream& operator>>(istream& in, Cube& cb) {
  try {
    CUBEXMLParser parser(cb);
    parser.do_parse(in);
    cb.compute_max();
  } catch (CUBException e) {
    cerr << e;
  }  
  return in;
}

/* based upon current view-mode, a severity value is converted
   to a color index */
int Cube::index(double val, int view, double refval, int pid) { 
  double per;
  double max_sev;
  if (val < 0) {
    // cout << "Negative data: " << val << endl;
    return 101;
  }
  if (val >= 0 && refval < 0 && (view ==1 || view == 2)) {
    cout << "Negative reference data: " << refval << endl;
    return 101;
  }
  if (pid < 0)
    max_sev = get_max();
  else // for metric node's index
    max_sev = get_max(pid);

  if (max_sev <= 1.0e-6 ) return 0; // avoid div 0

  if (view == 0 || view == 1) {
    per = (val/max_sev)*100.0;
  } else {
    if (refval <= 1.0e-6) return 0;
    per = (val/refval)*100.0;
  }
  if (per > 0 && per < 0.5) return 102; // dark grey color
  if (per >= 0.5 && per < 1.0) return 1;
  int i = (int) per;
  if (i > 102) {
    i = 101; // program error!!!
    cout << "color index out of range" << endl;
  }
  return i; 
}

/* based upon curent view-mode, a severity value is displayed
   as a different value, such as "absolute", "absolute percent",
   or "relative percent".*/
double Cube::sev4view(double sev, int view, double refval, int pid) {
  if (view == 0) return sev; // absolute mode
  double max_sev;
  if (pid < 0)
    max_sev = get_max();
  else // relative to the specified pid's root metric
    max_sev = get_max(pid);

  if (max_sev <= 1.0e-6) return 0; // avoid div 0

  if (view == 1) {           // percentage mode
    return (sev / max_sev)*100.0;
  }
  else { // relative percentage mode
    if (refval > 0 && refval <= 1.0e-6) {
      // when dividing a very small value, a big error occurs.
      refval = 1;
      sev = 0;
    } else if (refval == 0) {
      refval = max_sev;
    }
    return (sev / refval)*100.0;
  }
}

/*int get_exponent(double max) {
  unsigned long long val = (unsigned long long) max;
  int k = 0;
  while (val > 0) {
    val /= 10;
    k++;
  }
  return k;
  }*/

// for max >= 1
int get_exponent(double max) {
  if (max < 0) max = -max;
  double val = floor(max);
  int k = 0;
  while (val > 0) {
    val /= 10;
    val = floor(val); // only integer part wanted
    k++;
  }
  return k;
}

// for 0<max<1
int get_exponent2(double max) {
  if (max < 0) max = -max;
  double val = max;
  int k = 0;
  while (val != 0 && val < 1) {
    val *= 10;
    k++;
  }
  return k;
}

/****************************************************************         
                 Rules to display labels:

Applies only to the absolute mode:
Floating Point
  (1) Take the maximum
  (2) If 0.10 <= maximum < 100 don't do anything
  (3) Write it in scientific notation: a.bc E XY 
  (4) Scale all values in the tree with respect to E XY
  (5) Wite a.b E XY on the scale
Integer
  (1) Take the maximum
  (2) If 0 <= maximum < 999 don't do anything
  (3) Write it in scientific notation: abc E XY
  (4) Scale all values in the tree with respect to E XY
  (5) Wite abc E XY on the scale
- (realtive) percentage: one digit after decimal point
- absolute non-scientific decimal: one digit after decimal point
- absolute non-scientific integer: no decimal point
- absolute scientific: three significant digits 
******************************************************************/

string Cube::gen_met_label(int met_id, double sev,
			    int view, double refval, bool default_root) {
  Metric* p = (Metric*) get_met(met_id);
  double disp_v;
  bool isInteger = false;
  string uom = p->get_uom();
  double max_sev;
  ostrstream out;

  if (default_root) {
    max_sev = get_max();
    disp_v = sev4view(sev, view, refval, -1);
  }
  else {
    max_sev = get_max(met_id);
    disp_v = sev4view(sev, view, refval, met_id);
  }
  int exponent = get_exponent(max_sev);

  if (uom.compare("occ") == 0) isInteger = true;
  if (view == 0) {// only if absolute view
    if (!isInteger) {
      
      if (max_sev >= 100.0) {
	out.precision(2); // a.bc E xx
	out.setf(ios::fixed); // omit the E xx part 
	for (int m = 0; m < exponent-1; m++) sev /= 10.0;
	out << sev << " " << p->get_name() << " ";
	*(out.str()+out.pcount()) = 0;
	return out.str();
      } else if (max_sev < 0.1) {
	exponent = get_exponent2(max_sev);
	//cout << "max sev:" << max_sev << "exponent: " << exponent << endl;
	out.precision(2); // a.bc E xx
	out.setf(ios::fixed); // omit the E xx part 
	for (int m = 0; m < exponent; m++) sev *= 10.0;
	out << sev << " " << p->get_name() << " ";
	*(out.str()+out.pcount()) = 0;
	return out.str();
      }
    }
    if (isInteger && max_sev >= 1000) {
      out.precision(2); // a.bc (E xx, implied in spectrum)
      //out.setf(ios::scientific);
      out.setf(ios::fixed);
      for (int m = 0; m < exponent-1; m++) sev /= 10.0;
      out << sev << " " << p->get_name() << " ";
      *(out.str()+out.pcount()) = 0;
      return out.str();    
    } else if (isInteger) {
      int z = (int) sev;
      out << z << " " << p->get_name() << " ";
      *(out.str()+out.pcount()) = 0;
      return out.str();  
    }
  }
  out.precision(1);
  out.setf(ios::fixed);
  out << disp_v << " " << p->get_name() << " ";
  *(out.str()+out.pcount()) = 0;  
  return out.str();
}

string Cube::gen_max_label(double max_sev) {
  Metric* p = (Metric*) get_met(cur_metid);
  // double disp_v;
  bool isInteger = false;
  string uom = p->get_uom();
  ostrstream out;

  if (uom.compare("occ") == 0) isInteger = true;
  if (!isInteger) {
    if (max_sev >= 100.0 || max_sev < 0.1) {
      out.precision(2); // a.bc E xx
      out.setf(ios::scientific);
      out << max_sev << " ";
      *(out.str()+out.pcount()) = 0;
      return out.str();
    } 
  }
  if (isInteger && max_sev >= 1000) {
    out.precision(2); // a.bc E xx
    out.setf(ios::scientific);
    out << max_sev << " ";
    *(out.str()+out.pcount()) = 0;
    return out.str();    
  } else if (isInteger) {
    int z = (int) max_sev;
    out << z << " ";
    *(out.str()+out.pcount()) = 0;
    return out.str();  
  }
  out.precision(2);
  out.setf(ios::fixed);
  out << max_sev << " ";
  *(out.str()+out.pcount()) = 0;
  return out.str();
}

string Cube::gen_half_max_label(double max_sev) {
  Metric* p = (Metric*) get_met(cur_metid);
  // double disp_v;
  bool isInteger = false;
  string uom = p->get_uom();
  ostrstream out;

  if (uom.compare("occ") == 0) isInteger = true;
  if (!isInteger) {
    if (max_sev >= 100.0 || max_sev < 0.1) {
      out.precision(2); // a.bc E xx
      out.setf(ios::scientific);
      out << max_sev/2 << " ";
      *(out.str()+out.pcount()) = 0;
      return out.str();
    } 
  }
  if (isInteger && max_sev >= 1000) {
    out.precision(2); // a.bc E xx
    out.setf(ios::scientific);
    out << max_sev/2 << " ";
    *(out.str()+out.pcount()) = 0;
    return out.str();    
  } else if (isInteger) {
    int z = (int) max_sev/2;
    out << z << " ";
    *(out.str()+out.pcount()) = 0;
    return out.str();  
  }
  out.precision(2);
  out.setf(ios::fixed);
  out << max_sev/2 << " ";
  *(out.str()+out.pcount()) = 0;
  return out.str();
}

string Cube::gen_cnode_label(int met_id, int cnode_id, double sev, 
			     int view, double refval, bool default_root) {
  Metric* p = (Metric*) get_met(met_id);
  Cnode* cnode = (Cnode*) get_cnode(cnode_id);
  double disp_v;
  bool isInteger = false;
  string uom = p->get_uom();
  double max_sev;
  ostrstream out;

  if (default_root) {
    max_sev = get_max();
    disp_v = sev4view(sev, view, refval, -1);
  }
  else {
    max_sev = get_max(met_id);
    disp_v = sev4view(sev, view, refval, met_id);
  }
  int exponent = get_exponent(max_sev);
  if (uom.compare("occ") == 0) isInteger = true;
  string name = ((Region*)cnode->get_csite()->get_callee())->get_name();
  if (view == 0) {// only if absolute view
    if (!isInteger) {
      if (max_sev >= 100.0) {
	out.precision(2); // a.bc E xx
	out.setf(ios::fixed);
	for (int m = 0; m < exponent-1; m++) sev /= 10.0;
	out << sev << " " << name << " ";
	*(out.str()+out.pcount()) = 0;
	return out.str();
      } else if (max_sev < 0.1) {
	out.precision(2); // a.bc E xx
	out.setf(ios::fixed); // omit the E xx part 
	exponent = get_exponent2(max_sev);
	for (int m = 0; m < exponent; m++) sev *= 10.0;
	out << sev << " " << p->get_name() << " ";
	*(out.str()+out.pcount()) = 0;
	return out.str();
      }

    }
    if (isInteger && max_sev >= 1000) {
      out.precision(2); // a.bc E xx
      out.setf(ios::fixed);
      for (int m = 0; m < exponent-1; m++) sev /= 10.0;
      out << sev << " " << name << " ";
      *(out.str()+out.pcount()) = 0;
      return out.str();    
    } else if (isInteger) {
      int z = (int) sev;
      out << z << " " << name << " ";
      *(out.str()+out.pcount()) = 0;
      return out.str();  
    }
  }
  out.precision(1);
  out.setf(ios::fixed);
  out << disp_v << " " << name << " ";
  *(out.str()+out.pcount()) = 0;  
  return out.str();

}

string Cube::gen_function_label(int met_id, int func_id, double sev, 
				int view, double refval, bool default_root) {
  Metric* p = (Metric*) get_met(met_id);
  Region* r = (Region*) get_region(func_id);
  double disp_v;
  bool isInteger = false;
  string uom = p->get_uom();
  double max_sev;
  
  ostrstream out;

  if (default_root) {
    max_sev = get_max();
    disp_v = sev4view(sev, view, refval, -1);
  }
  else {
    max_sev = get_max(met_id);
    disp_v = sev4view(sev, view, refval, met_id);
  }
  int exponent = get_exponent(max_sev);
  if (uom.compare("occ") == 0) isInteger = true;
  string name = r->get_name();
  if (view == 0) {// only if absolute view
    if (!isInteger) {
      if (max_sev >= 100.0) {
	out.precision(2); // a.bc E xx
	//out.setf(ios::scientific);
	out.setf(ios::fixed);
	for (int m = 0; m < exponent-1; m++) sev /= 10.0;
	out << sev << " " << name << " ";
	*(out.str()+out.pcount()) = 0;
	return out.str();
      } else if (max_sev < 0.1) {
	out.precision(2); // a.bc E xx
	out.setf(ios::fixed); // omit the E xx part 
	exponent = get_exponent2(max_sev);
	for (int m = 0; m < exponent; m++) sev *= 10.0;
	out << sev << " " << p->get_name() << " ";
	*(out.str()+out.pcount()) = 0;
	return out.str();
      } 
    }
    if (isInteger && max_sev >= 1000) {
      out.clear();
      out.precision(2); // a.bc E xx
      //out.setf(ios::scientific);
      out.setf(ios::fixed);
      for (int m = 0; m < exponent-1; m++) sev /= 10.0;
      out << sev << " " << name << " ";
      *(out.str()+out.pcount()) = 0;
      // cout << "label: " << sev << "/" << out.str() << endl;
      return out.str();    
    } else if (isInteger) {
      int z = (int) sev;
      out << z << " " << name << " ";
      *(out.str()+out.pcount()) = 0;
      return out.str();  
    }
  }
  out.precision(1);
  out.setf(ios::fixed);
  out << disp_v << " " << name << " ";
  *(out.str()+out.pcount()) = 0;  
  return out.str();

}

string Cube::gen_mod_label(int met_id, int mod_id, double sev, 
			   int view, double refval, bool default_root) {
  Metric* p = (Metric*) get_met(met_id);
  Module* mod = (Module*) get_mod(mod_id);
  double disp_v;
  bool isInteger = false;
  string uom = p->get_uom();
  double max_sev;
  
  ostrstream out;

  if (default_root) {
    max_sev = get_max();
    disp_v = sev4view(sev, view, refval, -1);
  }
  else {
    max_sev = get_max(met_id);
    disp_v = sev4view(sev, view, refval, met_id);
  }
  int exponent = get_exponent(max_sev);
  if (uom.compare("occ") == 0) isInteger = true;
  string name = mod->get_name();
  if (view == 0) {// only if absolute view
    if (!isInteger) {
      if (max_sev >= 100.0) {
	out.precision(2); // a.bc E xx
	//out.setf(ios::scientific);
	out.setf(ios::fixed);
	for (int m = 0; m < exponent-1; m++) sev /= 10.0;
	out << sev << " " << name << " ";
	*(out.str()+out.pcount()) = 0;
	return out.str();
      } else if (max_sev < 0.1) {
	out.precision(2); // a.bc E xx
	out.setf(ios::fixed); // omit the E xx part 
	exponent = get_exponent2(max_sev);
	for (int m = 0; m < exponent; m++) sev *= 10.0;
	out << sev << " " << p->get_name() << " ";
	*(out.str()+out.pcount()) = 0;
	return out.str();
      }
    }
    if (isInteger && max_sev >= 1000) {
      out.clear();
      out.precision(2); // a.bc E xx
      //out.setf(ios::scientific);
      out.setf(ios::fixed);
      for (int m = 0; m < exponent-1; m++) sev /= 10.0;
      out << sev << " " << name << " ";
      *(out.str()+out.pcount()) = 0;
      // cout << "label: " << sev << "/" << out.str() << endl;
      return out.str();    
    } else if (isInteger) {
      int z = (int) sev;
      out << z << " " << name << " ";
      *(out.str()+out.pcount()) = 0;
      return out.str();  
    }
  }
  out.precision(1);
  out.setf(ios::fixed);
  out << disp_v << " " << name << " ";
  *(out.str()+out.pcount()) = 0;  
  return out.str();

}

string Cube::gen_func_children_label(int met_id, double sev, int view, 
				     double refval, bool default_root) {
  Metric* p = (Metric*) get_met(met_id);
  double disp_v;
  bool isInteger = false;
  string uom = p->get_uom();
  double max_sev;
  ostrstream out;

  if (default_root) {
    max_sev = get_max();
    disp_v = sev4view(sev, view, refval, -1);
  }
  else {
    max_sev = get_max(met_id);
    disp_v = sev4view(sev, view, refval, met_id);
  }
  int exponent = get_exponent(max_sev);
  if (uom.compare("occ") == 0) isInteger = true;
  string name = "subregions";
  if (view == 0) {// only if absolute view
    if (!isInteger) {
      if (max_sev >= 100.0) {
	out.precision(2); // a.bc E xx
	//out.setf(ios::scientific);
	out.setf(ios::fixed);
	for (int m = 0; m < exponent-1; m++) sev /= 10.0;
	out << sev << " " << name << " ";
	*(out.str()+out.pcount()) = 0;
	return out.str();
      } else if (max_sev < 0.1) {
	out.precision(2); // a.bc E xx
	out.setf(ios::fixed); // omit the E xx part 
	exponent = get_exponent2(max_sev);
	for (int m = 0; m < exponent; m++) sev *= 10.0;
	out << sev << " " << p->get_name() << " ";
	*(out.str()+out.pcount()) = 0;
	return out.str();
      }
    }
    if (isInteger && max_sev >= 1000) {
      out.precision(2); // a.bc E xx
      //out.setf(ios::scientific);
      out.setf(ios::fixed);
      for (int m = 0; m < exponent-1; m++) sev /= 10.0;
      out << sev << " " << name << " ";
      *(out.str()+out.pcount()) = 0;
      return out.str();    
    } else if (isInteger) {
      int z = (int) sev;
      out << z << " " << name << " ";
      *(out.str()+out.pcount()) = 0;
      return out.str();  
    }
  }
  out.precision(1);
  out.setf(ios::fixed);
  out << disp_v << " " << name << " ";
  *(out.str()+out.pcount()) = 0;  
  return out.str();

}

string Cube::gen_loc_label(int met_id, int loc_id, 
			   double sev, int view, 
			   double refval, bool default_root) {
  Metric* p = (Metric*) get_met(met_id);
  Location* loc = (Location*) get_loc(loc_id);

  double disp_v;
  bool isInteger = false;
  string uom = p->get_uom();
  double max_sev;
  ostrstream out;

  if (default_root) {
    max_sev = get_max();
    disp_v = sev4view(sev, view, refval, -1);
  }
  else {
    max_sev = get_max(met_id);
    disp_v = sev4view(sev, view, refval, met_id);
  }
  int exponent = get_exponent(max_sev);
  if (uom.compare("occ") == 0) isInteger = true;
  string name = loc->get_name();
  if (view == 0) {// only if absolute view
    if (!isInteger) {
      if (max_sev >= 100.0) {
	out.precision(2); // a.bc E xx
	out.setf(ios::fixed);
	for (int m = 0; m < exponent-1; m++) sev /= 10.0;   
	out << sev << " " << name << " ";
	*(out.str()+out.pcount()) = 0;
	return out.str();
      } else if (max_sev < 0.1) {
	out.precision(2); // a.bc E xx
	out.setf(ios::fixed); // omit the E xx part 
	exponent = get_exponent2(max_sev);
	for (int m = 0; m < exponent; m++) sev *= 10.0;
	out << sev << " " << p->get_name() << " ";
	*(out.str()+out.pcount()) = 0;
	return out.str();
      }
    }
    if (isInteger && max_sev >= 1000) {
      out.precision(2); // a.bc E xx
      //out.setf(ios::scientific);
      out.setf(ios::fixed);
      for (int m = 0; m < exponent-1; m++) sev /= 10.0;
      out << sev << " " << name << " ";
      *(out.str()+out.pcount()) = 0;
      return out.str();    
    } else if (isInteger) {
      int z = (int) sev;
      out << z << " " << name << " ";
      *(out.str()+out.pcount()) = 0;
      return out.str();  
    }
  }
  out.precision(1);
  out.setf(ios::fixed);
  out << disp_v << " " << name << " ";
  *(out.str()+out.pcount()) = 0;  
  return out.str();

}

int Cube::get_max_thrdnum() {
  int ret = 0;
  for (int i = 0; i < procv.size(); i++) {
    Process* proc = procv[i];
    if (proc->num_children() > ret) ret = proc->num_children();
  }
  return ret;
}


/*
void Cube::copy_struct(Cube& target) {
  //copy metric dimension
  for(int i = 0; i < rmetv.size(); i++) {
    Metric* root = rmetv[i];
    int id = target.def_met
      (root->get_name(), root->get_uom(), root->get_descr(), -1);
    Metric* target_root = (Metric*) target.get_met(id);
    copy_met_tree(root, target_root, target);
  }
}
// firstly, root is already copied successfully.
void copy_met_tree(Metric* source_root, Metric* target_root, Cube& target) {
  if (source_root->num_children() == 0) return;
  for (int i = 0; i < source_root->num_children(); i++) {
    Metric* schild = (Metric*) source_root->get_child(i);
    int id = target.def_met
      (schild->get_name(), schild->get_uom(),
       schild->get_descr(), target_root->get_id());
    Metric* tchild = (Metric*) target.get_met(id);
    copy_met_tree(schild, tchild, target);
  }
}
*/

// performance algebra operators

//copy constructor
Cube::Cube(const Cube& cube) {
  srand((unsigned)time(NULL));
  int n = rand();
  char tmp[256]; 
  sprintf(tmp, "tmp%d.cube", n);
  // char* tmp = "tmp000.cube";
  ofstream out;
  out.open(tmp);
  out << (Cube&) cube;
  out.close();
  ifstream in(tmp);
  if(!in) cout << "Cube copy constructor error" << endl;
  in >> *this;
  in.close();
  remove(tmp);
}

Cube& operator-= (Cube& cube1, Cube& cube2) {
  // compute each metric tree's maximum
  double d1, d2;
  cube1.compute_max();
  cube2.compute_max();
  // set up cube severity matrix  
  map<int, map<int, map<int, double> > >::const_iterator i1, i2;
  for (i1 = cube1.severity.begin(), i2 = cube2.severity.begin(); 
       i1 != cube1.severity.end() && i2 != cube2.severity.end(); 
       ++i1, ++i2) {
    d1 = cube1.get_max(i1->first);
    d2 = cube2.get_max(i2->first);
    CubeMatrix mat1 = i1->second;
    CubeMatrix mat2 = i2->second;
    // convert to percentage matrix
    CubeMatrix perc_mat1 = mat1/d1;
    CubeMatrix perc_mat2 = mat2/d2;
    CubeMatrix diff = perc_mat1 - perc_mat2;
    cube1.severity[i1->first] = (Map2D)diff;
  }  
  return cube1;
}
