/****************************************************************************
**  Copyright (C) 2004                                                     **
**  University of Tennessee, Innovative Computing Laboratory               **
**                                                                         **
**  See the file COPYRIGHT in the package base directory for details       **
****************************************************************************/


/*******************************
				
     PresenterFrame.cpp	
					
********************************/

#include <iostream>
#include <fstream>
#include <strstream>

#include "PresenterFrame.h"
#include "PresenterApp.h"
#include "Metric.h"
#include "Cnode.h"
#include "Callsite.h"
#include "Region.h"
#include "Location.h"
#include "Module.h"
#include "Machine.h"

#include "iclogo.xpm"


using namespace std;

// declaration of const variable (local linkage).
const int PresenterFrame::PADDING;

BEGIN_EVENT_TABLE(PresenterFrame, wxFrame)
  EVT_SIZE               (PresenterFrame::OnSize)
  EVT_SASH_DRAGGED_RANGE (ID_Metric_Panel, ID_Location_Panel, PresenterFrame::OnSashDrag)
  // EVT_PAINT              (PresenterFrame::OnPaint)
  // For menu items
  EVT_MENU               (ID_Open_File, PresenterFrame::OnOpenFile)
  EVT_MENU               (ID_Close_File, PresenterFrame::OnCloseFile)
  EVT_MENU               (ID_Exit, PresenterFrame::OnExit)
  EVT_MENU               (ID_About, PresenterFrame::OnAbout)
  EVT_MENU               (ID_Function_View, PresenterFrame::OnFunctionView)
  EVT_MENU               (ID_Calltree_View, PresenterFrame::OnCalltreeView)
  EVT_MENU               (ID_Absolute_View, PresenterFrame::OnAbsoluteView)
  EVT_MENU               (ID_Percent_View,  PresenterFrame::OnPercentageView)
  EVT_MENU               (ID_Relative_View, PresenterFrame::OnRelativeView)
END_EVENT_TABLE()

/* create and initialize a metric tree */
void  PresenterFrame::build_met_tree() {
  if (cube == NULL || met_tree == NULL) return;
  vector<Metric*> roots = cube->get_met_roots();
  for (int k = 0; k < roots.size(); k++) {
    // several metric roots possibly
    wxTreeItemId rootId = met_tree->AddRoot("NULL", 101); // temporary label
    Metric* p = roots[k];
    met_tree->SetItemData(rootId, new NodeData(p->get_id()));
    ostrstream out;
    if (k == 0) {
      met_tree->SelectItem(rootId);
    }
    int mode[] = {CUBE_INCL, -1, -1};
    double sev = cube->get_vsev(mode, p->get_id());
    string label = cube->gen_met_label(p->get_id(), sev, view_type());
    met_tree->SetItemText(rootId, label, cube->index(sev));
    
    /* build a wxTree from a cube metric tree*/
    vector<Metric*>  fromv[2]; // store ids for only two levels
    vector<wxTreeItemId>        tov[2];
    int depth = 0;
    fromv[depth].push_back(p);       // add root of cube tree
    tov[depth].push_back(rootId);    // add root of wxWindows tree
    while(fromv[depth%2].size() > 0) {
      //building tree from level #depth to level #(depth+1) 
      for (int i = 0; i < fromv[depth%2].size(); i++) {
	p = fromv[depth%2][i];
	wxTreeItemId pid = tov[depth%2][i];
	for (int j = 0; j < p->num_children(); j++) {
	  Metric* child = (Metric*) p->get_child(j);
	  // convert to label
	  ostrstream out2;
	  sev = cube->get_vsev(mode, child->get_id());
	  string label = cube->gen_met_label(child->get_id(), sev, view_type());
	  // store to the alternative vector
	  fromv[(depth+1)%2].push_back(child);
	  tov[(depth+1)%2].
	    push_back(met_tree->AppendItem(pid, label, 
					    cube->index(sev), 
					    new NodeData(child->get_id())));
	}
      }
      fromv[depth%2].clear();
      tov[depth%2].clear();
      depth++;
    } // end of construction of one tree
  } // end of construction of one forest
 
}

/* create and initialize a cnode tree */
void  PresenterFrame::build_cnode_tree() {
  if (cube == NULL || cnode_tree == NULL || !is_calltree()) return;
  wxTreeItemId rootId = cnode_tree->GetVRootItem();
  cnode_tree->DeleteChildren(rootId);
  if (cube->num_cnodes() == 0) return;
  vector<Cnode*> roots = cube->get_cnode_roots();

  for (int k = 0; k < roots.size(); k++) {
    rootId = cnode_tree->AddRoot("NULL", 101); // temporary label
    Cnode* cnode = roots[k]; // get a root node
    cnode_tree->SetItemData(rootId, new NodeData(cnode->get_id()));
    if (k == 0) cnode_tree->SelectItem(rootId);
    int sel_met = met_tree->GetItemData(met_tree->GetSelection())->get_id();
    // ostrstream out;
    int mode[] = {CUBE_INCL, CUBE_INCL, -1};
    double sev = cube->get_vsev(mode, sel_met, cnode->get_id());
    string label = cube->gen_cnode_label(sel_met, cnode->get_id(), sev);
    cnode_tree->SetItemText(rootId, label, cube->index(sev));
    vector<Cnode*>  fromv[2];        // store ids for only two levels
    vector<wxTreeItemId>        tov[2];
    int depth = 0;
    fromv[depth].push_back(cnode);   // add root of cube tree
    tov[depth].push_back(rootId);    // add root of wxWindows tree
    while(fromv[depth%2].size() > 0) {
      //building tree from depth to depth+1
      for (int i = 0; i < fromv[depth%2].size(); i++) {
	cnode = fromv[depth%2][i];
	wxTreeItemId pid = tov[depth%2][i];
	for (int j = 0; j < cnode->num_children(); j++) {
	  Cnode* child = (Cnode*) cnode->get_child(j);
	  // convert to label
	  sev = cube->get_vsev(mode, sel_met, child->get_id());
	  label = cube->gen_cnode_label(sel_met, child->get_id(), sev);
	  // store to the alternative vector
	  fromv[(depth+1)%2].push_back(child);
	  tov[(depth+1)%2].push_back(cnode_tree->
				     AppendItem(pid, label, cube->index(sev),
						new NodeData(child->get_id())));
	}
      }
      fromv[depth%2].clear();
      tov[depth%2].clear();
      depth++;
    }
  }

  // if creation is not the first time, refresh it based on metric node info.
  wxTreeItemId id = met_tree->GetSelection(); // newly selected node.
  int met_id = met_tree->GetItemData(id)->get_id();
  int met_mode;
  if (met_tree->IsExpanded(id)) 
    met_mode = CUBE_EXCL;
  else met_mode = CUBE_INCL;
  cnode_tree->refresh(met_id, met_mode);
}

/* If choosing "function profile tree", the cnode_tree will be destroyed,
   then function tree is recreated.
*/
void PresenterFrame::build_func_tree() {
  wxTreeItemId sel_func; // default one in the beginning
  if (cube == NULL || cnode_tree == NULL || is_calltree()) return;
  wxTreeItemId rootId = cnode_tree->GetVRootItem();
  cnode_tree->DeleteChildren(rootId);
  vector<Module*> modv = get_cube()->get_mod_roots();
  if (modv.size() == 0) return; // no modules loaded ever
  // append every module to the "root"
  int sel_met = met_tree->GetItemData(met_tree->GetSelection())->get_id();

  for (int i = 0; i < modv.size(); i++) {
    Module* m = modv[i];
    int modid = m->get_id();
    wxTreeItemId pid = cnode_tree->AddRoot(m->get_name(), 0);
    cnode_tree->SetItemData(pid, new NodeData(modid));
    // for each module, append its functions as child nodes
    for (int j = 0; j < m->num_region(); j++) {
      Region* r = (Region*) m->get_child(j);
      if (!r->get_name().compare("virtual caller")) continue;
      int mode[] = {CUBE_INCL, CUBE_INCL, -1};
      double sev = cube->get_vsev(mode, sel_met, r->get_id(), -1, false);
      string label=cube->gen_function_label(sel_met, r->get_id(), sev);
      wxTreeItemId funcid = 
	cnode_tree->AppendItem(pid, label, cube->index(sev), 
			       new NodeData(r->get_id()));
      if (i == 0 && j == 0) sel_func = funcid;
      // append a virtual "children" node
      if (r->num_children() > 0)
	cnode_tree->AppendItem(funcid, "subregions", 101, NULL);
    } // end of inner for loop
  } // end of outer for loop
  
  cnode_tree->SelectItem(sel_func);
  cnode_tree->Collapse(cnode_tree->GetParent(sel_func));
  cnode_tree->refresh(sel_met, CUBE_INCL);
}

/* create and initialize a location tree */
void  PresenterFrame::build_loc_tree() {
  if (cube == NULL || loc_tree == NULL) return;
  wxTreeItemId rootId = loc_tree->AddRoot("NULL", 101);
  loc_tree->SelectItem(rootId);
  Location* p = (Location*) cube->get_loc(0);
  int process_level; // >= 0
  // if grid has only one machine, don't show the grid item
  if (p->num_children() > 1) {
    loc_tree->SetItemData(rootId, new NodeData(0));
    process_level = 3;
  } else {
    Machine* m = (Machine*) p->get_child(0);
    loc_tree->SetItemData(rootId, new NodeData(m->get_locid()));
    p = m; // beginning from machine, instead of grid
    process_level = 2;
  }
  //ostrstream out;
  int mode[] = {CUBE_INCL, CUBE_INCL, CUBE_INCL};
  double sev = cube->get_vsev(mode, 
			     0, 0, p->get_locid());
  string label = cube->
    gen_loc_label(0, p->get_locid(), sev);
  loc_tree->SetItemText(rootId, label, cube->index(sev));
  vector<Location*>  fromv[2]; // store ids for only two levels
  vector<wxTreeItemId>        tov[2];
  int depth = 0;
  fromv[depth].push_back(p);       // add root of cube tree
  tov[depth].push_back(rootId);    // add root of wxWindows tree
  while(fromv[depth%2].size() > 0) {
    // building tree from depth to depth+1
    for (int i = 0; i < fromv[depth%2].size(); i++) {
      if (depth == process_level && cube->get_max_thrdnum() == 1) 
	continue; // no threads displayed
      p = fromv[depth%2][i];
      wxTreeItemId pid = tov[depth%2][i];
      for (int j = 0; j < p->num_children(); j++) {
	Location* child = (Location*) p->get_child(j);
	sev = cube->get_vsev(mode, 0, 0, child->get_locid());
	string label = cube->gen_loc_label(0, child->get_locid(), sev);	
	// store to the alternative vector
	fromv[(depth+1)%2].push_back(child);
	tov[(depth+1)%2].push_back
	  (loc_tree->AppendItem(pid, label, cube->index(sev),
				new NodeData(child->get_locid())));
      }
    }
    fromv[depth%2].clear();
    tov[depth%2].clear();
    depth++; // move down to next level
  }
}

PresenterFrame::PresenterFrame(wxWindow *parent, 
			       const wxWindowID id, 
			       const wxString& title, 
			       const wxPoint& pos, 
			       const wxSize& size,
			       const long style):
  wxMDIParentFrame(parent, id, title, pos, size, style) {
  PresenterApp* app = (PresenterApp*) wxTheApp;
  SetBackgroundColour(app->bg_color);
  logo = new wxBitmap(iclogo);

  /* create "File" menu */
  wxMenu* file_menu = new wxMenu;
  file_menu->Append(ID_Open_File, "&Open...");
  file_menu->Append(ID_Close_File, "&Close"); 
  file_menu->AppendSeparator();
  file_menu->Append(ID_Exit, "E&xit" );
  /* create "View" menu */
  view_menu = new wxMenu;
  // "function profile tree menu"
  view_menu->AppendCheckItem(ID_Calltree_View, "Call tree");
  view_menu->Check(ID_Calltree_View, true); // default check
  view_menu->AppendCheckItem(ID_Function_View, "&Region profile");
  view_menu->AppendSeparator();

  view_menu->AppendCheckItem(ID_Absolute_View, "&Absolute");
  view_menu->Check(ID_Absolute_View, true); // default check
  view_menu->AppendCheckItem(ID_Percent_View, "&Percentage");
  view_menu->AppendCheckItem(ID_Relative_View, "&Relative percentage");
  /* create "Help" menu */
  wxMenu* help_menu = new wxMenu;
  help_menu->Append(ID_About, "&About");
  /* create Menu Bar on wxFrame */
  wxMenuBar* menu_bar = new wxMenuBar;
  menu_bar->Append(file_menu, "&File");
  menu_bar->Append(view_menu, "&View");
  menu_bar->Append(help_menu, "&Help");
  SetMenuBar(menu_bar);
  CreateStatusBar(2);
  status_bar = GetStatusBar();
  /* initialization of some data members */
  show_callpath = true;
  view_option = 0; // absolute view
  // SetStatusText("Welcome to CUBE Presenter!");
  set_statusbar(); // display n x m for mpi/openMP
  /* create color spectrum pane */
  spectrum_pane = new SpectrumPane(this, 1000);
  int default_w = 800/3; // default window width
  met_panel    = create_metric_pane(default_w, ID_Metric_Panel);
  cnode_panel   = create_callnode_pane(default_w, ID_Cnode_Panel);
  loc_panel     = create_location_pane(default_w, ID_Location_Panel);
}

PresenterFrame::~PresenterFrame() {
  if (cube != NULL) delete cube;
}

string get_base(const string name) {
  int pos = name.find_last_of('/');
  if (pos != string::npos)
    return name.substr(pos+1, name.length()); 
  else 
    return name;
}
string get_path(const string name) {
  int pos = name.find_last_of('/');
  if (pos != string::npos) {
    return name.substr(0, pos);
  } 
  else 
    return "";
}


/* Open file dialog */
void PresenterFrame::OnOpenFile(wxCommandEvent& event) {
  wxCommandEvent tmp;
  PresenterApp* app = (PresenterApp*) wxTheApp;
  wxString path=::wxFileSelector(_T("Select the cube file to load"),
				 _T(""), _T(""), _T(""),
				 _T("cube files (*.cube)|*.cube|All files (*.*)|*.*"),
				 wxOPEN|wxCHANGE_DIR,this);
  if ( !path )  return;
  std::ifstream cubefile((const char*)path);
  if (!cubefile) 
    wxLogMessage(_T("Open file error: '%s'"), (const wxChar*) path);

  // close the previous file if necessary
  OnCloseFile(tmp);
  string title = "CUBE:  ";
  // only display a file's base name
  title += get_base(path.c_str());
  SetTitle(title.c_str());
  cube = new Cube();
  cubefile >> *cube;
  app->set_proc_num(cube->num_procs());
  app->set_thrd_num(cube->get_max_thrdnum());
  refresh_label_mn();
  // cube.update_met_cache(0);
  build_met_tree();
  if (is_calltree())
    build_cnode_tree();
  else 
    build_func_tree();
  build_loc_tree();
  spectrum_pane->Refresh();
}
/* Close current file */
void PresenterFrame::OnCloseFile(wxCommandEvent& event) {
  wxTreeItemId rootId;
  rootId = met_tree->GetVRootItem();
  met_tree->DeleteChildren(rootId);
  rootId = cnode_tree->GetVRootItem();
  cnode_tree->DeleteChildren(rootId);
  rootId = loc_tree->GetVRootItem();
  loc_tree->DeleteChildren(rootId);
  if (cube != NULL) {
    delete cube;
    cube = NULL;
  }
  spectrum_pane->Refresh();
}

void PresenterFrame::OnExit(wxCommandEvent& event) {
  Close(true);
}

void PresenterFrame::OnAbout(wxCommandEvent& WXUNUSED(event))
{
  wxDialog* about = new wxDialog(this, -1, "About CUBE", 
				 wxPoint(0,0), wxSize(400, 150));
  wxStaticBitmap *icon = new wxStaticBitmap(about, -1, *logo);
  wxBoxSizer* topsizer = new wxBoxSizer(wxVERTICAL);
  wxBoxSizer* sizer = new wxBoxSizer(wxHORIZONTAL);
  sizer->Add(icon, 0, wxCENTER|wxLEFT, 30);
  sizer->Add(new wxStaticText
	     (about, -1, 
	      "This is the CUBE Version 1.0.5.\n(C) ICL at UTK 2004", 
	      wxDefaultPosition, wxSize(280, -1)),
	     0,
	     wxCENTER|wxLEFT, 10);
  topsizer->Add(sizer, 1,  wxCENTER | wxLEFT|wxRIGHT|wxTOP, 10);
  topsizer->Add(new wxButton(about, wxID_OK, _("&OK")), 0, 
		wxCENTER|wxALL, 10); 
  about->SetSizer(topsizer);
  about->Centre(wxBOTH);
  about->ShowModal();
  delete about;

}

/*   Menu: Funcition Profile Tree &  Callpath Tree     */
void PresenterFrame::OnFunctionView  (wxCommandEvent& event) {
  view_menu->Check(ID_Function_View, true);
  view_menu->Check(ID_Calltree_View, false);
  show_callpath = false;
  cnodelabel->SetLabel(_T("Region Profile"));

  build_func_tree();
  // met_tree->refresh(); // let met_tree display for function tree
}
void PresenterFrame::OnCalltreeView  (wxCommandEvent& event) {
  view_menu->Check(ID_Calltree_View, true);
  view_menu->Check(ID_Function_View, false);
  show_callpath = true;

  cnodelabel->SetLabel(_T("Call Tree"));
  build_cnode_tree();
  met_tree->refresh(); // let met_tree display for callpath tree
}
// absolute severity value view
void PresenterFrame::OnAbsoluteView  (wxCommandEvent& event) {
  view_menu->Check(ID_Absolute_View, true);
  view_menu->Check(ID_Percent_View, false);
  view_menu->Check(ID_Relative_View, false);
  if (view_option != 0) {
    view_option = 0;
    spectrum_pane->Refresh();
    met_tree->refresh();
  }
}
// percentage view
void PresenterFrame::OnPercentageView  (wxCommandEvent& event) {
  view_menu->Check(ID_Percent_View,  true);
  view_menu->Check(ID_Relative_View, false);
  view_menu->Check(ID_Absolute_View, false);
  if (view_option != 1) {
    view_option = 1;
    spectrum_pane->Refresh();
    met_tree->refresh();
  }
}
// relative percentage view
void PresenterFrame::OnRelativeView  (wxCommandEvent& event) {
  view_menu->Check(ID_Relative_View, true);
  view_menu->Check(ID_Absolute_View, false);
  view_menu->Check(ID_Percent_View,  false);
  if (view_option != 2) {
    view_option = 2;
    spectrum_pane->Refresh();
    met_tree->refresh();
  }
}

void PresenterFrame::OnSize(wxSizeEvent& WXUNUSED(event)) {
  int width;
  GetClientSize(&width, NULL);
  width /= 3;
  met_panel->SetDefaultSize(wxSize(width, 1000));
  cnode_panel->SetDefaultSize(wxSize(width, 1000));
  loc_panel->SetDefaultSize(wxSize(width, 1000));

  wxLayoutAlgorithm layout;
  layout.LayoutWindow(this, loc_panel);
  /* adjust statusbar's field size */
  int width2[2];
  int w;
  GetClientSize(&w, NULL);
  width2[0] = w/12;
  width2[1] = w - width2[0];
  status_bar->SetFieldsCount(2, width2);  
}

void PresenterFrame::OnSashDrag(wxSashEvent& event) {
  if (event.GetDragStatus() == wxSASH_STATUS_OUT_OF_RANGE)
    return;
  
  switch (event.GetId()) {
  case ID_Metric_Panel:
    met_panel->SetDefaultSize(wxSize(event.GetDragRect().width, 1000));
    break;
  case ID_Cnode_Panel:
    cnode_panel->SetDefaultSize(wxSize(event.GetDragRect().width, 1000));
    break;
  }

  wxLayoutAlgorithm layout;
  // layout.LayoutMDIFrame(this);
  layout.LayoutWindow(this, loc_panel);
  // GetClientWindow()->Refresh();
}

/* create a label with 3D style */
wxPanel* PresenterFrame::create_3Dlabel(wxWindow* parent, wxString label) {
  wxBoxSizer* sizer = new wxBoxSizer(wxVERTICAL);
  wxPanel* label_pane = new wxPanel(parent, -1, wxDefaultPosition,
				    wxDefaultSize, wxRAISED_BORDER);
  // a data member
  wxStaticText* text = new wxStaticText(label_pane,
					-1, label, wxPoint(0,0),
					wxDefaultSize, wxALIGN_LEFT);
  if (label.CompareTo("Call Tree") == 0) cnodelabel = text;
  label_pane->SetSize(100, 25); // default height
  label_pane->SetSizer(sizer);
  sizer->Add(text, 1, wxGROW|wxALL, 2);
  return label_pane;
}

wxSashLayoutWindow* 
PresenterFrame::create_metric_pane(int width, int ID) {
  wxSashLayoutWindow* win;
  win = new wxSashLayoutWindow(this, ID,
                               wxDefaultPosition, wxSize(200, 30),
                               wxSW_3D);
  win->SetDefaultSize(wxSize(width, 1000));
  win->SetOrientation(wxLAYOUT_VERTICAL);
  win->SetAlignment(wxLAYOUT_LEFT);
  win->SetSashVisible(wxSASH_RIGHT, TRUE);
  win->SetExtraBorderSize(PADDING);  
  // setup a wxPanel covering it.
  wxBoxSizer* topsizer = new wxBoxSizer(wxVERTICAL);
  wxPanel* pane = new wxPanel(win, -1);
  win->SetSizer(topsizer);
  wxBoxSizer* pane_sizer = new wxBoxSizer(wxVERTICAL);
  pane->SetSizer(pane_sizer);
  wxPanel* label_pane = create_3Dlabel(pane, _T("Performance Metrics"));
  pane_sizer->Add(label_pane, 0, wxGROW|wxALL, 0);
  // add metric tree
  met_tree = new MetricTree(pane);
  met_tree->SetBackgroundColour(GetBackgroundColour());
  //met_tree->AddRoot("NULL", 101); // 101 indicates "white color"
  wxSize size = pane->GetClientSize();
  pane_sizer->Add(met_tree,
		  1,
		  wxGROW|wxALL,
		  0);  
  topsizer->Add(pane, 1, wxGROW|wxALL, 0);
  return win;
}

wxSashLayoutWindow* 
PresenterFrame::create_callnode_pane(int width, int ID) {
  wxSashLayoutWindow* win;
  win = new wxSashLayoutWindow(this, ID,
                               wxDefaultPosition, wxSize(200, 30),
                               wxSW_3D);
  win->SetDefaultSize(wxSize(width, 1000));
  win->SetOrientation(wxLAYOUT_VERTICAL);
  win->SetAlignment(wxLAYOUT_LEFT);
  win->SetSashVisible(wxSASH_RIGHT, TRUE);
  win->SetExtraBorderSize(PADDING);  
  // setup a wxPanel covering it.
  wxBoxSizer* topsizer = new wxBoxSizer(wxVERTICAL);
  wxPanel* pane = new wxPanel(win, -1);
  win->SetSizer(topsizer);
  wxBoxSizer* pane_sizer = new wxBoxSizer(wxVERTICAL);
  pane->SetSizer(pane_sizer);
  wxPanel* label_pane = create_3Dlabel(pane, _T("Call Tree"));
  pane_sizer->Add(label_pane, 0, wxGROW|wxALL, 0);
  // add call node tree
  cnode_tree = new CnodeTree(pane);
  cnode_tree->SetBackgroundColour(GetBackgroundColour());
  //cnode_tree->AddRoot("NULL", 101); // white color
  wxSize size = pane->GetClientSize();
  pane_sizer->Add(cnode_tree,
		  1,
		  wxGROW|wxALL,
		  0);  
  topsizer->Add(pane, 1, wxGROW|wxALL, 0);
  return win;
}

wxSashLayoutWindow* 
PresenterFrame::create_location_pane(int width, int ID) {
  wxSashLayoutWindow* win;
  win = new wxSashLayoutWindow(this, ID,
                               wxDefaultPosition, wxSize(200, 30),
                               wxSW_3D);
  win->SetDefaultSize(wxSize(width, 1000));
  win->SetOrientation(wxLAYOUT_VERTICAL);
  win->SetAlignment(wxLAYOUT_LEFT);
  win->SetExtraBorderSize(PADDING);  
  // setup a wxPanel covering it.
  wxBoxSizer* topsizer = new wxBoxSizer(wxVERTICAL);
  wxPanel* pane = new wxPanel(win, -1);
  win->SetSizer(topsizer);
  wxBoxSizer* pane_sizer = new wxBoxSizer(wxVERTICAL);
  pane->SetSizer(pane_sizer);
  wxPanel* label_pane = create_3Dlabel(pane, _T("Locations"));
  pane_sizer->Add(label_pane, 0, wxGROW|wxALL, 0);
  // gap between label and tree
  // wxPanel* gap = new wxPanel(pane, -1);
  // gap->SetSize(100, 10);
  // pane_sizer->Add(gap, 0, wxGROW|wxALL, 0);
  // add location tree
  loc_tree = new LocationTree(pane);
  loc_tree->SetBackgroundColour(GetBackgroundColour());
  //loc_tree->AddRoot("NULL", 101);
  wxSize size = pane->GetClientSize();
  pane_sizer->Add(loc_tree,
		  1,
		  wxGROW|wxALL,
		  0);  
  topsizer->Add(pane, 1, wxGROW|wxALL, 0);
  return win;
}

void PresenterFrame::OnPaint(wxPaintEvent& event) {

}

void PresenterFrame::set_statusbar() {
  PresenterApp* app = (PresenterApp*) wxTheApp;
  wxString text;
  text.Printf("%d x %d", app->get_proc_num(), app->get_thrd_num());
  label_mn = new wxStaticText(status_bar, -1, text,
			      wxPoint(5,5), wxDefaultSize, wxALIGN_CENTRE);
  /* adjust statusbar's field size */
  int width[2];
  int w;
  GetClientSize(&w, NULL);
  width[0] = w/12;
  width[1] = w - width[0];
  status_bar->SetFieldsCount(2, width);  
}

void PresenterFrame::refresh_label_mn() {
  PresenterApp* app = (PresenterApp*) wxTheApp;
  wxString text;
  text.Printf("%d x %d", app->get_proc_num(), app->get_thrd_num());
  label_mn->SetLabel(text);
}
