/****************************************************************************
**  Copyright (C) 2004                                                     **
**  University of Tennessee, Innovative Computing Laboratory               **
**                                                                         **
**  See the file COPYRIGHT in the package base directory for details       **
****************************************************************************/

/*******************************
				
     CnodeTree.cpp	
					
********************************/

#include <iostream>
#include <strstream>
#include <fstream>
#include <unistd.h>

#include "Tree.h"
#include "CnodeTree.h"
#include "LocationTree.h"

#include "PresenterApp.h"
#include "PresenterFrame.h"
#include "SourceCodeDlg.h"

#include "Cnode.h"
#include "Callsite.h"
#include "Region.h"
#include "Module.h"

using namespace std;

BEGIN_EVENT_TABLE(CnodeTree, Tree)
  // For menu items
  EVT_MENU               (Cnode_POP_ID_Des,   CnodeTree::OnDes)
  EVT_MENU               (Cnode_POP_ID_Code,  CnodeTree::OnSourceCode)
END_EVENT_TABLE()

void CnodeTree::cnode_descr_dlg(int id) {
  PresenterFrame* frame = ((PresenterApp*)wxTheApp)->get_frame();
  Cube* cube = frame->get_cube();
  wxString text;
  Callsite* csite;
  if (id >= 0) {
    Cnode* cnode = (Cnode*) cube->get_cnode(id);
    csite = cnode->get_csite();
  }
  wxDialog* dlg = new wxDialog(this, -1, "Call site description", 
			       wxPoint(0,0), wxSize(300, 100));
  wxBoxSizer* sizer = new wxBoxSizer(wxVERTICAL);
  Module* m = (Module*) csite->get_mod();
  // module name
  string fname = m->get_name();
  text = "Module name: ";
  text.Append(fname.c_str());
  sizer->Add(new wxStaticText(dlg, -1, text, wxDefaultPosition, wxSize(300, -1)),
	    1,
	    wxALL, 3);
  // line number
  int ln = csite->get_line();
  text.Printf("Line number: %d", ln);
  sizer->Add(new wxStaticText(dlg, -1, text, wxDefaultPosition, wxSize(300, -1)),
	    1,wxALL, 3);
  // "OK" button
  sizer->Add(new wxButton(dlg, wxID_OK, _("&Ok")), 1, 
	    wxALIGN_CENTRE_HORIZONTAL, 3);
  
  dlg->SetSizer(sizer);
  dlg->Centre(wxBOTH);
  dlg->ShowModal();
  delete dlg;
}

void CnodeTree::region_descr_dlg(int id) {
  PresenterFrame* frame = ((PresenterApp*)wxTheApp)->get_frame();
  Cube* cube = frame->get_cube();
  wxString text;
  Region* r;
  if (id >= 0) {
    r = (Region*) cube->get_region(id);
  }
  wxDialog* dlg = new wxDialog(this, -1, "Function description", 
			       wxPoint(0,0), wxSize(300, 100));
  wxBoxSizer* sizer = new wxBoxSizer(wxVERTICAL);
  Module* m = (Module*) r->get_mod();
  // module name
  string fname = m->get_name();
  text = "Module name: ";
  text.Append(fname.c_str());
  sizer->Add(new wxStaticText(dlg, -1, text, wxDefaultPosition, wxSize(300, -1)),
	    1,
	    wxALL, 3);
  // begin line number
  int ln1 = r->get_begn_ln();
  text.Printf("Begin line number: %d", ln1);
  sizer->Add(new wxStaticText(dlg, -1, text, wxDefaultPosition, wxSize(300, -1)),
	    1,wxALL, 3);
  // end line number
  int ln2 = r->get_end_ln();
  text.Printf("End line number: %d", ln2);
  sizer->Add(new wxStaticText(dlg, -1, text, wxDefaultPosition, wxSize(300, -1)),
	     1,wxALL, 3); 
  // "OK" button
  sizer->Add(new wxButton(dlg, wxID_OK, _("&Ok")), 1, 
	    wxALIGN_CENTRE_HORIZONTAL, 3);
  
  dlg->SetSizer(sizer);
  dlg->Centre(wxBOTH);
  dlg->ShowModal();
  delete dlg;
}

void CnodeTree::OnDes(wxCommandEvent& event) {
  PresenterFrame* frame = ((PresenterApp*)wxTheApp)->get_frame();
  wxTreeItemId sel =  right_click_item; //GetSelection();
  if (GetItemData(sel) == NULL) {
    // "subrountines" of function profile tree
    wxMessageBox(_T("Corresponding subregions of its parent."), 
		 _T("Description"), wxOK, this);
    return;
  }
  int id = GetItemData(sel)->get_id();
  if (frame->is_calltree()) {
    cnode_descr_dlg(id);
  } else {
    if (!frame->is_calltree() && GetItemParent(sel) == GetVRootItem()) {
      // "module" of function profile tree
      wxMessageBox(_T("It is a module."), 
		   _T("Description"), wxOK, this);
      return;
    }
    region_descr_dlg(id);
  }
}

/* Check whether it's possible to read out that line.
   
   if module is not existing, return empty string;
   if the line is empty, return "empty line";
   if the line is not existing, return "N/A".
*/
string CnodeTree::get_line(string path, string fname, int line) {
  char code[256];
  if (path.empty()) {
    // if empty, set it as last searching path.
    path = default_path;
  }
  // if path is not empty, we assume the path is correct.
  // Users have to modify their cube files to make it work.

  path += "/";
  path += fname;
  cout << "module name: " << path << endl;
  ifstream in (path.c_str());
  string ret;
  if (!in) {
    wxMessageDialog 
      msg(this, "Module not found!\nClick Ok to specify another directory", 
	  "", wxOK | wxCANCEL | wxICON_ERROR);
    if (msg.ShowModal() == wxID_OK){
      wxString dir=::wxDirSelector(_T("Select search directory"),
				    _T(default_path.c_str()));
      if ( !dir )  return "";
      default_path = dir.c_str(); // set it as default path
      string code = get_line(dir.c_str(), fname, line);
      return code;
    }
    return ""; // no such file existing, return empty string.
  }

  successpath = path; // for display_region_code() to use

  int counter = 0;
  while (in.getline(code, 256)) {
    counter++;
    if (counter == line) break;
  }
  if (counter == line) {
    in.close();
    if (string(code).empty()) return "  empty Line";
    else return string(code);
  }
  in.close();
  return "N/A";
}

/* All the file's souce code will be shown in dialog.
   if highlighting one line, just give begn_ln==end_ln */
void CnodeTree::display_region_code(int begn_ln, int end_ln) {
  PresenterFrame* frame = ((PresenterApp*)wxTheApp)->get_frame();
  /*if (path.empty()) {
    path = default_path;
  }
  path += "/";
  path += file;
  */
  ifstream in (successpath.c_str());
  if (!in) {
    cout << "File open error: " << successpath << endl;
    return; // do nothing about it.
  }
  in.close();
  
  frame->get_source_dlg()->show_module(successpath);
  frame->get_source_dlg()->highlight(begn_ln, end_ln);
  frame->get_source_dlg()->show_line(begn_ln);
  frame->get_source_dlg()->Show(true);
}

/******************************************************************
   display source code accordingly 
   note:
   - source-code display
   * specified path (CUBE XML file)
   * user-selected directory and subdirectory (if there is one)
   * pop up directory-selection dialog and let user pic directory
     (remember): default current directory
   + the line of interest should be marked
       - if you have the line: mark line
       - else mark caller region

*******************************************************************/
void CnodeTree::OnSourceCode(wxCommandEvent& event) {
  PresenterFrame* frame = ((PresenterApp*)wxTheApp)->get_frame();
  Cube* cube = frame->get_cube();
  wxTreeItemId sel = right_click_item; // GetSelection();
  int line;       // the callsite line number
  Callsite* csite;
  Module* m;
  string mname;
  
  // if function profile tree
  if (!frame->is_calltree()) {
    // 1) subroutines
    if (GetItemData(sel) == NULL) return;

    SourceCodeDlg* dlg = frame->get_source_dlg();
    if (dlg == NULL) {
      dlg = new SourceCodeDlg(this);
      dlg->Move(frame->GetPosition()-dlg->GetPosition()-wxPoint(400,0));
      frame->set_source_dlg(dlg);
    }
    int id = GetItemData(sel)->get_id();
    // 2) module
    if (GetItemParent(sel) == GetVRootItem()) { 
      m = (Module*) cube->get_mod(id);
      mname = m->get_name();
      frame->get_source_dlg()->set_module(mname);
      frame->get_source_dlg()->set_line(1);
      string ln = get_line(get_path(mname), get_base(mname), 1);
      if (ln.empty()) return; // user click "Cancel"
      display_region_code(0, 0);     
      return;
    }
    // 3) regular function
    Region* r = (Region*) cube->get_region(id);
    m = (Module*) r->get_mod();
    mname = m->get_name();
    frame->get_source_dlg()->set_module(mname);
    frame->get_source_dlg()->set_line(r->get_begn_ln());
    string ln = get_line(get_path(mname), get_base(mname), 1);
    if (ln.empty()) return; // user click "Cancel"
    display_region_code(r->get_begn_ln(), r->get_end_ln());  
    return;
  }

  // if call path tree
  int id = GetItemData(sel)->get_id();
  if (id >= 0) {
    Cnode* cnode = (Cnode*) cube->get_cnode(id);
    csite = cnode->get_csite();
    m = (Module*) csite->get_mod();
    mname = m->get_name();
  } else return;
  line = csite->get_line(); // callsite line number
  SourceCodeDlg* dlg = frame->get_source_dlg();
  if (dlg == NULL) {
    dlg = new SourceCodeDlg(this);
    dlg->Move(frame->GetPosition()-dlg->GetPosition()-wxPoint(400,0));
    frame->set_source_dlg(dlg);
  }
  frame->get_source_dlg()->set_module(mname);
  frame->get_source_dlg()->set_line(line);
  if (line <= 0) {
    // user didn't define the line number, show the caller region
    Region* caller = (Region*) csite->get_caller();
    string ln = get_line
      (get_path(mname), get_base(mname), caller->get_begn_ln());
    if (ln.empty()) return; // user click "Cancel"
    display_region_code(caller->get_begn_ln(), caller->get_end_ln());
    return;
  }
  // if having a line number, always assume the line is correct.
  string code = get_line(get_path(mname), 
			 get_base(mname), line); // code, "empty line", or "N/A"
  if (!code.empty()) {
    // Assume module name is valid
    display_region_code(line, line); // one line is highlighted
  }
}
  

CnodeTree::CnodeTree(wxWindow* parent) : Tree(parent) {
  target_met_id = 0;
  target_met_mode = CUBE_INCL;
  default_path = ".";
  char dirname[128];
  if (getcwd(dirname, 128) != NULL) {
    default_path = dirname;
  }

  popup_menu->Append(Cnode_POP_ID_Des,   _T("Description"));
  popup_menu->Append(Cnode_POP_ID_Code,  _T("Source code"));
}

void CnodeTree::OnExpanded(wxTreeEvent& event) {
  int id = (int) event.GetItem();
  PresenterFrame* frame = ((PresenterApp*)wxTheApp)->get_frame();
  if (frame->is_calltree()) {
    // refresh the callpath tree
    int cnode_id = GetItemData(id)->get_id();
    int mode[3];
    mode[CUBE_MET] = target_met_mode, mode[CUBE_PROG] = CUBE_EXCL, mode[CUBE_LOC] = -1;
    double sev = frame->get_cube()->get_vsev(mode,
					     target_met_id, cnode_id, -1,
					     frame->is_calltree());
    mode[CUBE_PROG] = -1;
    double refsev = frame->get_cube()->get_vsev(mode, 
					       target_met_id, -1, -1,
					       frame->is_calltree());
    string label = frame->get_cube()->
      gen_cnode_label(target_met_id, cnode_id, sev, frame->view_type(), refsev);
    SetItemText(id, label, 
		frame->get_cube()->index(sev,frame->view_type(),refsev));
    //if it's also the selected cnode. 
    if (IsSelected(id)) 
      frame->get_loc_tree()->refresh(target_met_id, target_met_mode,
				     cnode_id, CUBE_EXCL);
  } else {
    // refresh the function tree
    set_funcnode_label(id, frame->get_cube());
    if (!IsSelected(id)) return;
    refresh_loc_tree4func(id);
  }

}

void CnodeTree::OnCollapsed(wxTreeEvent& event) {
  int id = (int) event.GetItem();
  PresenterFrame* frame = ((PresenterApp*)wxTheApp)->get_frame();
  if (frame->is_calltree()) {
    // refresh the callpath tree
    int cnode_id = GetItemData(id)->get_id();
    int mode[3];
    mode[CUBE_MET] = target_met_mode, mode[CUBE_PROG] = CUBE_INCL, mode[CUBE_LOC] = -1;
    double sev = frame->get_cube()->get_vsev(mode, 
					    target_met_id, cnode_id, -1,
					    frame->is_calltree());
    mode[CUBE_PROG] = -1;
    double refsev = frame->get_cube()->get_vsev(mode, 
					       target_met_id, -1, -1,
					       frame->is_calltree());
    string label = frame->get_cube()->
      gen_cnode_label(target_met_id, cnode_id, sev, frame->view_type(), refsev);
    SetItemText(id, label, 
		frame->get_cube()->index(sev,frame->view_type(),refsev));
    //if it's also the selected metric. 
    if (IsSelected(id)) 
      frame->get_loc_tree()->refresh(target_met_id, target_met_mode,
				     cnode_id, CUBE_INCL);
  } else {
    // refresh the function profile tree
    set_funcnode_label(id, frame->get_cube());
    if (!IsSelected(id)) return;
    refresh_loc_tree4func(id);
  }
  
}


void CnodeTree::refresh_loc_tree4func(int id) {
  PresenterFrame* frame = ((PresenterApp*)wxTheApp)->get_frame();
  int rid, rmode;
  // refresh the function tree
  NodeData* itemdata = GetItemData(id);
  if (itemdata != NULL) {
    rid = itemdata->get_id();
    if (IsExpanded(id)) rmode = CUBE_EXCL;
    else rmode = CUBE_INCL;
    bool ismod = GetItemParent(id) == GetVRootItem();
    if (!ismod)
      frame->get_loc_tree()->refresh(target_met_id, target_met_mode, rid, rmode);
    else 
      frame->get_loc_tree()->refresh4mod(target_met_id, target_met_mode, rid, rmode);
  } else { // "subroutine" vertex found
    wxTreeItemId pid = GetItemParent(id); // get parent function
    NodeData* itemdata = GetItemData(pid);
    int rid = itemdata->get_id();
    frame->get_loc_tree()->refresh4sub(target_met_id, target_met_mode, rid);
  }  
}

void CnodeTree::OnSelChanged(wxTreeEvent& event) {
  //int id = (int) event.GetItem();
  int id = GetSelection(); // newly selected node.
  int rid, rmode;
  PresenterFrame* frame = ((PresenterApp*)wxTheApp)->get_frame();
  if (id == old_sel) return;

  if (frame->is_calltree()) {
    // refresh the callpath tree
    int cnode_id = GetItemData(id)->get_id();
    int cnode_mode;
    if (IsExpanded(id)) cnode_mode = CUBE_EXCL;
    else cnode_mode = CUBE_INCL;
    frame->get_loc_tree()->refresh(target_met_id, target_met_mode,
				   cnode_id, cnode_mode);
  } else {
    refresh_loc_tree4func(id);
  }
  old_sel = id;
}

void CnodeTree::OnRightClick(wxTreeEvent& event) {
  right_click_item = HitTest(event.GetPoint());
  PopupMenu(popup_menu, event.GetPoint());
}

void CnodeTree::refresh (int met_id, int met_mode) {
  cout << "call tree is updated:" << met_id << " " << met_mode << endl;
  target_met_id   = met_id;
  target_met_mode = met_mode;
  PresenterFrame* frame = ((PresenterApp*)wxTheApp)->get_frame();
  Cube* cube = frame->get_cube();
  if (GetChildrenCount(wxTreeCtrl::GetRootItem(), false) == 0) return;
  if (frame->is_calltree()) {
    // Refresh the callpath tree
    traverse_calltree(wxTreeCtrl::GetRootItem(), cube);
    // what's the currently selected cnode
    int selection = GetSelection();
    int cnode_id = GetItemData(selection)->get_id();
    int cnode_mode = IsExpanded(selection) ? CUBE_EXCL : CUBE_INCL;
    frame->get_loc_tree()->refresh(met_id, met_mode, cnode_id, cnode_mode);
  } else {
    // Refresh the function tree
    traverse_functree(GetVRootItem(), cube);
    int selection = GetSelection();
    refresh_loc_tree4func(selection);
  }
}
/* traverse callpath tree */
void CnodeTree::traverse_calltree(wxTreeItemId root, Cube* cube) {
  long cookie; // wxWindows's variable
  double sev, refsev;
  ostrstream out;
  
  PresenterFrame* frame = ((PresenterApp*)wxTheApp)->get_frame();

  // update the root's label.
  int cnode_mode = IsExpanded(root) ? CUBE_EXCL : CUBE_INCL;
  int cnode_id = GetItemData(root)->get_id();
  if (cnode_id >= 0) {
    int mode[3];
    mode[CUBE_MET] = target_met_mode, mode[CUBE_PROG] = cnode_mode, mode[CUBE_LOC] = -1;
    sev = cube->get_vsev(mode, 
			target_met_id, cnode_id, -1);
    mode[CUBE_PROG] = -1;
    refsev = cube->get_vsev(mode,
			   target_met_id, -1, -1);
    string label = cube->gen_cnode_label(target_met_id, cnode_id, sev, frame->view_type(),
					refsev);
    SetItemText(root, label, 
		cube->index(sev, frame->view_type(),refsev));
  }
  if (GetChildrenCount(root) == 0) {
     return;
  }
  wxTreeItemId child = GetFirstChild(root, cookie); 
  traverse_calltree(child, cube);
  while (child = GetNextChild(root, cookie)) {
    traverse_calltree(child, cube);
  }
  return;
}

/* traverse function profile tree */
void CnodeTree::traverse_functree(wxTreeItemId root, Cube* cube) {
  long cookie; 
  PresenterFrame* frame = ((PresenterApp*)wxTheApp)->get_frame();

  // update the root's label if it's a function node
  set_funcnode_label(root, cube);
  if (GetChildrenCount(root) > 0) {
    wxTreeItemId child = GetFirstChild(root, cookie); 
    traverse_functree(child, cube);
    while (child = GetNextChild(root, cookie)) {
      traverse_functree(child, cube);
    }
  }

}

void CnodeTree::set_funcnode_label(wxTreeItemId cur, Cube* cube) {
  double sev = 0;
  double refsev = 0;
  ostrstream out;
  int rmode, rid;
  string label;
  int mode[3];

  NodeData* itemdata = GetItemData(cur);
  PresenterFrame* frame = ((PresenterApp*)wxTheApp)->get_frame();
  if (IsExpanded(cur))  rmode = CUBE_EXCL;
  else rmode = CUBE_INCL;

  if (itemdata == NULL) {
    // for subroutines
    wxTreeItemId pid = GetItemParent(cur);
    NodeData* itemdata = GetItemData(pid);
    rid = itemdata->get_id();
    mode[CUBE_MET] = target_met_mode;
    mode[CUBE_PROG] = CUBE_EXCL; // not useful
    mode[CUBE_LOC] = -1;
    sev = cube->get_vsev(mode, target_met_id, rid, -1, false, false ,true);
    mode[CUBE_PROG] = -1;
    refsev = cube->get_vsev(mode, target_met_id, -1, -1, false, false, true);
    label = cube->gen_func_children_label(target_met_id, sev, frame->view_type(), refsev);
    SetItemText(cur, label, cube->index(sev, frame->view_type(), refsev));
    return;
  }
  rid = itemdata->get_id(); // region id
  if (rid < 0) return; // virtual root
  if (GetItemParent(cur) == GetVRootItem()) {
    mode[CUBE_MET] = target_met_mode;
    mode[CUBE_PROG] = rmode;
    mode[CUBE_LOC] = -1;
    sev = cube->get_vsev(mode, target_met_id, rid, -1, false, true);
    mode[CUBE_PROG] = -1;
    refsev = cube->get_vsev(mode, target_met_id, -1, -1, false, true);
    // cout << "module: " << sev << "," << refsev << endl;
    label = cube->gen_mod_label(target_met_id, rid, sev, frame->view_type(), refsev);
    SetItemText(cur, label, cube->index(sev, frame->view_type(), refsev));     
    return; // module vertex
  } 
  // region vertex
  mode[CUBE_MET] = target_met_mode, mode[CUBE_PROG] = rmode, mode[CUBE_LOC] = -1;
  sev = cube->get_vsev(mode, target_met_id, rid, -1, false);
  mode[CUBE_PROG] = -1;
  refsev = cube->get_vsev(mode, target_met_id, -1, -1, false);
  label = cube->gen_function_label(target_met_id, rid, sev,
				  frame->view_type(), refsev);
  SetItemText(cur, label, cube->index(sev, frame->view_type(), refsev)); 
}
