/****************************************************************************
**  Copyright (C) 2004                                                     **
**  University of Tennessee, Innovative Computing Laboratory               **
**                                                                         **
**  See the file COPYRIGHT in the package base directory for details       **
****************************************************************************/


/************************************************
             CubeMatrix.cpp


 ************************************************/

#include "CubeMatrix.h"

using namespace std;

const int CubeMatrix::MAX_COLS;
const int CubeMatrix::MAX_ROWS;

double CubeMatrix::sum() {
  double val = 0.0;
  CubeMatrix::const_iterator i;
  map<int, double>::const_iterator j;
  for (i = begin(); i != end(); i++) {
    for (j = i->second.begin(); j != i->second.end(); j++) {
      val += j->second;
    }
  }  
  
  return val;
}


/* helper functions */


/* matrix - matrix */
CubeMatrix operator-(CubeMatrix& mat1, CubeMatrix& mat2) {
  int r1, r2; // row index
  CubeMatrix ret;
  CubeMatrix::const_iterator i1, i2;
  
  for (i1 = mat1.begin(), i2 = mat2.begin(); 
       i1 != mat1.end() || i2 != mat2.end(); /* empty */) {
    if (i1 == mat1.end()) r1 = CubeMatrix::MAX_ROWS;
    else r1 = i1->first;
    if (i2 == mat2.end()) r2 = CubeMatrix::MAX_ROWS;
    else r2 = i2->first;
    if (r1 == r2) {
      // compute one row
      ret[r1] = mat1[r1] - mat2[r2];
      i1++;
      i2++;
    } else if (r1 < r2) {
      ret[r1] = mat1[r1];
      i1++;
    } else {
      ret[r2] = -mat2[r2];
      i2++;
    }
  } // end of for loop
  
  return ret;
}

/* vector - vector */
map<int,double> operator-(map<int, double>& map1,
			  map<int, double>& map2) {
  map<int, double> ret ;
  int c1, c2; // column index
  map<int, double>::const_iterator i1, i2;
  // input should not be empty map
  for (i1 = map1.begin(), i2 = map2.begin();
       i1 != map1.end() || i2 != map2.end(); ) {
    if (i1 == map1.end()) c1 = CubeMatrix::MAX_COLS;
    else c1 = i1->first;
    if (i2 == map2.end()) c2 = CubeMatrix::MAX_COLS;
    else c2 = i2->first;
    if (c1 == c2) {
      // compute one row
      ret[c1] = map1[c1] - map2[c2];
      if (ret[c1] < 0 && -ret[c1] < 1.0e-6) ret[c1] = 0;
      if (ret[c1] < 0) {
       ret[c1] = 0.0;
      }
      i1++;
      i2++;
    } else if (c1 < c2) {
      ret[c1] = map1[c1];
      i1++;
    } else {
      ret[c2] = -map2[c2];
      i2++;
    }    
  } // end of for loop
  
  return ret;
}

/* -vector, argument value not changed. */
map<int,double> operator-(map<int, double>& map1) {
  map<int, double> ret ;
  int c; // column index
  map<int, double>::const_iterator i;
  
  for (i = map1.begin(); i != map1.end(); i++) {
    c = i->first;
    ret[c] = -map1[c];
  } // end of for loop
  
  return ret;
}

// vector + operator
map<int, double> operator+(map<int, double>& map1, map<int, double>& map2) {
  map<int, double> ret ;
  int c1, c2; // column index
  map<int, double>::const_iterator i1, i2;
  // input should not be empty map
  for (i1 = map1.begin(), i2 = map2.begin();
       i1 != map1.end() || i2 != map2.end(); ) {
    if (i1 == map1.end()) c1 = CubeMatrix::MAX_COLS;
    else c1 = i1->first;
    if (i2 == map2.end()) c2 = CubeMatrix::MAX_COLS;
    else c2 = i2->first;
    if (c1 == c2) {
      // compute one row
      ret[c1] = map1[c1] + map2[c2];
      i1++;
      i2++;
    } else if (c1 < c2) {
      ret[c1] = map1[c1];
      i1++;
    } else {
      ret[c2] = map2[c2];
      i2++;
    }    
  } // end of for loop
  
  return ret;
}

// matrix + operator
CubeMatrix operator+(CubeMatrix& mat1, CubeMatrix& mat2) {
  int r1, r2; // row index
  CubeMatrix ret;
  CubeMatrix::const_iterator i1, i2;
  
  for (i1 = mat1.begin(), i2 = mat2.begin(); 
       i1 != mat1.end() || i2 != mat2.end(); /* empty */) {
    if (i1 == mat1.end()) r1 = CubeMatrix::MAX_ROWS;
    else r1 = i1->first;
    if (i2 == mat2.end()) r2 = CubeMatrix::MAX_ROWS;
    else r2 = i2->first;
    if (r1 == r2) {
      // compute one row
      ret[r1] = mat1[r1] + mat2[r2];
      i1++;
      i2++;
    } else if (r1 < r2) {
      ret[r1] = mat1[r1];
      i1++;
    } else {
      ret[r2] = mat2[r2];
      i2++;
    }
  } // end of for loop
  
  return ret;

}

// matrix / operator
CubeMatrix operator/(CubeMatrix& mat, double d) {
  CubeMatrix ret;

  CubeMatrix::const_iterator i;
  map<int, double>::const_iterator j;
  for (i = mat.begin(); i != mat.end(); i++) {
    for (j = i->second.begin(); j != i->second.end(); j++) {
      ret[i->first][j->first] = j->second/d;
    }
  }  
  return ret;
}

/* print out the matrix */
ostream& operator<<(ostream& out, CubeMatrix& matrix) {
  CubeMatrix::const_iterator i;
  map<int, double>::const_iterator j;
  out << "matrix: " << endl;
  for (i = matrix.begin(); i != matrix.end(); i++) {
    for (j = i->second.begin(); j != i->second.end(); j++) {
      out << j->second << "  ";
    }
    out << endl;
  }  
  
  return out;
}
