#ifndef _CUBEXMLPARSER_H
#define _CUBEXMLPARSER_H

/*
 *----------------------------------------------------------------------------
 *
 * class CUBEXMLParser
 *
 *----------------------------------------------------------------------------
 */

#include <iostream>
#include <string>
#include <stack>
#include <vector>
#include <utility>
  
#include "libxml/parser.h"
#include "Vertex.h"

#define BUFFSIZE       1024

/* parser's states */
#define PARSE_METRIC    0

#define PARSE_MODULE      1
#define PARSE_REGION      2
#define PARSE_CSITE       3
#define PARSE_CNODE       4

#define PARSE_GRID        5
#define PARSE_MACHINE     6
#define PARSE_NODE        7
#define PARSE_PROCESS     8
#define PARSE_THREAD      9

#define PARSE_SEVERITY    10

class Cube;
typedef std::pair<std::string, std::vector<std::string> > ItemType;

class CUBEXMLParser {
 public:

    CUBEXMLParser      (Cube& cube);
    ~CUBEXMLParser();
    void do_parse      (std::istream& in); 

    
    void start_element (const xmlChar* name, const xmlChar ** atts);
    void end_element   (const xmlChar* name);
    void data_content  (std::string content);

 private:
    /* private methods */
    void parse_met    (ItemType& parent, ItemType& current);
    void parse_module  (ItemType& parent, ItemType& current); 
    void parse_region  (ItemType& parent, ItemType& current); 
    void parse_csite   (ItemType& parent, ItemType& current); 
    void parse_cnode   (ItemType& current); 
    void leave_cnode   ();
    void parse_grid    (ItemType& parent, ItemType& current); 
    void parse_machine (ItemType& parent, ItemType& current); 
    void parse_node    (ItemType& parent, ItemType& current); 
    void parse_process (ItemType& parent, ItemType& current); 
    void parse_thread  (ItemType& parent, ItemType& current);  
    void parse_row     (ItemType& parent, ItemType& current);

    void enter_state_change     (ItemType& cur);
    void exit_state_change      (ItemType& cur);
    void dispatch_state_handler (ItemType& parent, ItemType& current);

    int get_attr_id(ItemType item);

 private:
    
    Cube*              cube;
    Vertex*            last_vertex; //parent node kept for next new object.

    /* used only by libxml2 */
    xmlSAXHandler      handler;
    xmlParserCtxtPtr   ctxt;
    char*              buffer;

    /* for walking xml documents */
    std::stack<ItemType, std::vector<ItemType> > track_stack;
    int state;
    std::string             data; // data within elements
    std::vector<std::string>     builder_memo;
    /* builder_memo is used to record a CUBE object's information located
     at previously walked elements. It contains those elements' contents.*/

    int thrd_counter; // thread's label is only local to its process.
};


/* dispatch specific parser's event-handler methods */
void startElement (void* parser, const xmlChar *name, const xmlChar **atts);
void endElement   (void* parser, const xmlChar *name);
void characters   (void* parser, const xmlChar* text, int length);

#endif


