/****************************************************************************
**  Copyright (C) 2004                                                     **
**  University of Tennessee, Innovative Computing Laboratory               **
**                                                                         **
**  See the file COPYRIGHT in the package base directory for details       **
****************************************************************************/

/*******************************************


	CUBEXMLParser.cpp


********************************************/

#include <strstream>
#include <iostream>
//#include <cstdio>

#include "CUBEXMLParser.h"
#include "CUBException.h"
#include "Metric.h"
#include "Module.h"
#include "Region.h"
#include "Callsite.h"
#include "Cnode.h"
#include "Grid.h"
#include "Machine.h"
#include "Node.h"
#include "Process.h"
#include "Thread.h"
#include "Cube.h"

using namespace std;

/* Constructor */
CUBEXMLParser::CUBEXMLParser(Cube& cube) {
  this->cube = &cube;
  last_vertex  = NULL;
  state = -1;
  thrd_counter = 0;
  memset(&handler, 0, sizeof(handler));

  /* assign event handlers */
  handler.startElement = startElement;
  handler.endElement   = endElement;
  handler.characters   = characters;

  ctxt = ::xmlCreatePushParserCtxt(&handler,this, 0, 0, 0);
  if (ctxt == NULL) throw CUBException("ParserCtxt created error");
  /* allocate parsing buffer */
  buffer = (char*) malloc(BUFFSIZE);
  if (buffer == NULL) throw CUBException("malloc error");

}

/* Destructor */
CUBEXMLParser::~CUBEXMLParser() { 
  xmlFreeParserCtxt(ctxt);
  free(buffer);
  builder_memo.clear();
}

/* Parse user's given istream and build a Cube object in memory*/
void CUBEXMLParser::do_parse(istream& in) {
  bool success;
  int count;
  bool ret; // parser's return valule.
  
  while (in.good() && in.read(buffer, BUFFSIZE) && (count = in.gcount())){
    ret = ::xmlParseChunk(ctxt, buffer, count, 0); // 0 indicates success!   
    if (ret != false) {
      throw CUBException("Parser reading XML error!"); 
    }   
  }  
  ::xmlParseChunk(ctxt, buffer, count, 1); //'count' can't be 0 here.  

  return;
}

/* Parser changes its state when entering an element */
void CUBEXMLParser::enter_state_change(ItemType& item) {
  if(item.first == "metric") {
    state = PARSE_METRIC;      
  } 
  else if (item.first == "module") {
    state = PARSE_MODULE;
  }
  else if (item.first == "region") {
    state = PARSE_REGION;
  }
  else if (item.first == "csite") {
    state = PARSE_CSITE;
  }
  else if (item.first == "cnode") {
    state = PARSE_CNODE;
  }
  else if (item.first == "grid") {
    state = PARSE_GRID;
  }
  else if (item.first == "machine") {
    state = PARSE_MACHINE;   
  }
  else if (item.first == "node") {
    state = PARSE_NODE;
  }
  else if (item.first == "process") {
    state = PARSE_PROCESS;
  }
  else if (item.first == "thread") {
    state = PARSE_THREAD;
  }
  else if (item.first == "severity") {
    state = PARSE_SEVERITY;
  }

  return;
}

/* Parser changes its state when leaving an element */
void CUBEXMLParser::exit_state_change(ItemType& current) {
  if (current.first      == "region")  state = PARSE_MODULE;
  else if (current.first == "csite")   state = PARSE_REGION;
  else if (current.first == "cnode")   leave_cnode();
  else if (current.first == "machine") state = PARSE_GRID;
  else if (current.first == "node")    state = PARSE_MACHINE;
  else if (current.first == "process") state = PARSE_NODE;
  else if (current.first == "thread")  state = PARSE_PROCESS;

  return;
}

/* Based on current state, parser calls its corresponding handler. 
   This method is called in element_end_event handler.
*/
void CUBEXMLParser::dispatch_state_handler (ItemType& parent, 
					    ItemType& current )
{
  try {
    switch(state) {
    case PARSE_METRIC:
      parse_met(parent, current);
      break;
    case PARSE_MODULE:
      parse_module(parent, current);
      break;
    case PARSE_REGION:
      parse_region(parent, current);
    break;
    case PARSE_CSITE:
      parse_csite(parent, current);
      break;
    case PARSE_GRID:
      parse_grid(parent, current);
    break;
    case PARSE_MACHINE:
      parse_machine(parent, current);
      break;
    case PARSE_NODE:
      parse_node(parent, current);
      break;
    case PARSE_PROCESS:
      parse_process(parent, current);
      break;
    case PARSE_THREAD:
      parse_thread(parent, current);
      break;
    case PARSE_SEVERITY:
      parse_row(parent, current);
    }
  } catch(CUBException e) {
    cerr << e;
  }
}

/* be called whenever entering an element */
void CUBEXMLParser::start_element(const xmlChar* name, const xmlChar ** atts) {
  char** attributes = (char**) atts;
  ItemType item;
  /* back up the element */
  item.first = (char*) name;
  for(int i = 0; attributes != NULL && attributes[i] != NULL; i++) {
    item.second.push_back(string(attributes[i]));
  }
  /* push stack */
  track_stack.push(item);
  /* state changes */
  enter_state_change(item);
  /* parse the cnode tree at the entry point*/
  if (state == PARSE_CNODE) parse_cnode(item); 
  
  return;
}

/* being called whenever leaving an element */
void CUBEXMLParser::end_element(const xmlChar* name) {
  ItemType parent;
  ItemType current;
  
  current = track_stack.top();
  track_stack.pop();
  if (track_stack.empty()) return; /* parser is done */
  parent = track_stack.top();
  
  /* call different handlers */
  dispatch_state_handler(parent, current);
  
  /* exit the element */
  data = "";
  exit_state_change(current);

}

/* SAX events order: start_element->characters->end_element */
void CUBEXMLParser::data_content(string content) {
  int i;
  char ch;
  /* skip the string consisting of only spaces, tabs and newlines. */
  for (i = 0; i < content.size(); i++) {
    ch = content.at(i);
    if (ch != ' ' && ch != '\n' && ch != '\t') break;
  }
  if (i >= content.size()) {
    return; // no content in string
  }
  data = content;
  return;
}

int CUBEXMLParser::get_attr_id(ItemType item) {
  int i;
  if (item.second.size() < 2) throw CUBException("attribute id not found");
  i = atoi(item.second[1].c_str());
  return i;
}

void CUBEXMLParser::parse_met(ItemType& parent, ItemType& current) {
  /* collecting a metric's info. */
  if (parent.first == "metric" && 
      (current.first == "name"   || 
       current.first == "uom"    ||
       current.first == "descr") ) {
    builder_memo.push_back(data);
    /* when at the end of <descr>, new a metric object */
    if (current.first == "descr") {
      int id = get_attr_id(parent);
      string descr = builder_memo[2];// builder_memo size == 3 always.   
      if (last_vertex == NULL) {
	last_vertex = cube->create_met(id, builder_memo[0], 
					builder_memo[1], descr, -1);
      } 
      else {
	  Metric* p = (Metric*) last_vertex;
	  last_vertex = cube->create_met(id, builder_memo[0], builder_memo[1],
					  descr, p->get_id());
      }
      builder_memo.clear(); // after creation of a new metric
    }
  }
  /* after finishing a metric element, last_vertex
     moves one level up pointing to its parent metric */
  if (current.first == "metric") {
    if (last_vertex != NULL) {
      Metric* p = (Metric*) ((Metric*)last_vertex)->get_parent();
      last_vertex = (Vertex*) p;
    }    
  }
}

void CUBEXMLParser::parse_module(ItemType& parent, ItemType& current) {
  /* collecting a module's info. */
  if (parent.first == "module" && 
      (/*current.first == "path" ||*/ current.first == "name" ) ) {
    builder_memo.push_back(data);
    /* when at the end of <name>, new a Module object */
    if (current.first == "name") {
      int id = get_attr_id(parent);
      if (last_vertex == NULL) {
	last_vertex = cube->create_module(id,/*builder_memo[1],*/builder_memo[0]);
      } 
      else throw CUBException("Parser: module has a non-zero parent");
      builder_memo.clear(); // after creation of a new Module
    }
  }
  if (current.first == "module") {
    last_vertex = NULL;
  }
}

void CUBEXMLParser::parse_region(ItemType& parent, ItemType& current) {
  if (parent.first == "region" && 
      (current.first == "name" || current.first == "begin" ||
       current.first == "end"  || current.first == "descr") ) {
    builder_memo.push_back(data);
    /* when at the end of <descr>, new a Region object */
    if (current.first == "descr") {
      int id, modid, begin, end;
      if (builder_memo[1].empty() || builder_memo[2].empty())
	throw CUBException("Parser: region line number empty");
      if (last_vertex != NULL) { // should has a parent
	Module* m = (Module*) last_vertex;
	modid = m->get_id();
	begin = atoi(builder_memo[1].c_str());
	end = atoi(builder_memo[2].c_str());
	id = get_attr_id(parent);
	last_vertex = cube->create_region(id, builder_memo[0], begin,
					  end, builder_memo[3], modid);
      } 
      else throw CUBException("Parser: region's parent = NULL");
      builder_memo.clear(); // after creation of a new Module
    }
  }
  if (current.first == "region") {
    Region* r = (Region*) last_vertex;
    last_vertex = (Module*) r->get_mod();
  }
}

void CUBEXMLParser::parse_csite(ItemType& parent, ItemType& current) {
  if (parent.first == "csite" && 
      (current.first == "line" || current.first == "callee")) {
    builder_memo.push_back(data);
    // when at the end of <callee>, new a csite object
    if (current.first == "callee") {
      int id, mid, line, callee, caller;
      if (builder_memo[0].empty()) {
	cout << "Warning: callsite line number empty" << endl;
	// throw CUBException("Parser: callsite line number empty");
      }
      if (builder_memo[1].empty()) {
	throw CUBException("Parser: callee empty");
      }	
      // try to look for its caller region's id from the stack
      ItemType item1 = track_stack.top(); // pop "csite"
      track_stack.pop();
      ItemType callerElem = track_stack.top();
      track_stack.push(item1);
      if (last_vertex != NULL) { // should has a parent
	Region* r = (Region*) last_vertex;
	mid = ((Module*)r->get_mod())->get_id();
	line = atoi(builder_memo[0].c_str());
	callee = atoi(builder_memo[1].c_str());
	caller = atoi(callerElem.second[1].c_str());
	// cout << "caller " << callerElem.first << " id is: " << callerElem.second[1] << endl;
	id = get_attr_id(parent);
	last_vertex = cube->create_csite(id, mid, line, callee, caller);
      } 
      else throw CUBException("Parser: csite's parent = NULL");
      builder_memo.clear(); // after creation of a new Module
    }
  }
  if (current.first == "csite") {
    Callsite* csite = (Callsite*) last_vertex;
    last_vertex = (Region*) csite->get_caller();
  }
}

/* being called from element_start event handler */
void CUBEXMLParser::parse_cnode(ItemType& current) {
  int id;
  string csite = current.second[3];

  if (csite.empty()) throw CUBException("Parser: invalid csiteId");
  int csid = atoi(csite.c_str());
  id = get_attr_id(current);
  if (last_vertex == NULL) {
    last_vertex = cube->create_cnode(id, csid, -1);
  } 
  else {
    Cnode* p = (Cnode*) last_vertex;
    last_vertex = cube->create_cnode(id, csid,  p->get_id());
  }

}
void CUBEXMLParser::leave_cnode() {
  if (last_vertex != NULL) {
    Cnode* cnode = ((Cnode*) last_vertex);
    last_vertex = (Vertex*) cnode->get_parent();
  }
  if (last_vertex == NULL) state = -1; // not in PARSE_CNODE any more
}

void CUBEXMLParser::parse_grid(ItemType& parent, ItemType& current) {
  if (parent.first == "grid" && current.first == "name" ) {
    builder_memo.push_back(data);
    int id = get_attr_id(parent);
    if (parent.second.size() < 4) throw CUBException("grid attribute locid not found");
    int locid = atoi(parent.second[3].c_str());
    string name = builder_memo[0];
    if (last_vertex == NULL) {
      last_vertex = cube->create_grid(locid, id, name);
    } else throw CUBException("grid's last_vertex != NULL");
    builder_memo.clear(); // after creation of a new Grid
  }
  if (current.first == "grid") {
    last_vertex = NULL;
  }
}

void CUBEXMLParser::parse_machine(ItemType& parent, ItemType& current) {
  if (parent.first == "machine" && current.first == "name" ) {
    builder_memo.push_back(data);
    int id = get_attr_id(parent);
    if (parent.second.size() < 4) 
      throw CUBException("machine attribute locid not found");
    int locid = atoi(parent.second[3].c_str());
    string name = builder_memo[0];
    if (last_vertex != NULL) { // should has a parent
      Grid* g = (Grid*) last_vertex;
      int gid = g->get_id();
      last_vertex = cube->create_mach(locid, id, name,gid);
    }
    else throw CUBException("Parser: machine's parent = NULL");
    builder_memo.clear(); // after creation of a new Module
  }
  if (current.first == "machine") {
    Machine* m = (Machine*) last_vertex;
    last_vertex = (Vertex*) m->get_grid();
  }
}

void CUBEXMLParser::parse_node(ItemType& parent, ItemType& current) {
  if (parent.first == "node" && current.first == "name") {
    builder_memo.push_back(data);
    /*if (current.first == "cpuNum") {
      if (builder_memo[1].empty()) 
      throw CUBException("Parser: empty cpu number");*/      
      int id;
      // num = atoi(builder_memo[1].c_str());
      if (last_vertex != NULL) { // should has a parent
	Machine* m = (Machine*) last_vertex;
	id = m->get_id();
	if (parent.second.size() < 4) 
	  throw CUBException("node attribute locid not found");
	int locid = atoi(parent.second[3].c_str());
	last_vertex = cube->create_node(locid, get_attr_id(parent),
					builder_memo[0], id);
      } 
      else throw CUBException("Parser: node's parent = NULL");
      builder_memo.clear(); // after creation of a new Module
      //}
  }
  if (current.first == "node") {
    Node* m = (Node*) last_vertex;
    last_vertex = (Vertex*) m->get_mach();
  }
}

void CUBEXMLParser::parse_process(ItemType& parent, ItemType& current) {
  if (parent.first == "process" && current.first == "name" ) {
    thrd_counter = 0;
    builder_memo.push_back(data);
    int id;
    string name = builder_memo[0];
    if (last_vertex != NULL) { // should has a parent
      Node* node = (Node*) last_vertex;
      id = node->get_id();
      if (parent.second.size() < 4) 
	throw CUBException("process attribute locid not found");
      int locid = atoi(parent.second[3].c_str());
      if (name.empty()) {
	name = "Process ";
	name += parent.second[1];
      }
      last_vertex = cube->create_proc(locid, get_attr_id(parent), name,id);
    } 
    else throw CUBException("Parser: process's parent = NULL");
    builder_memo.clear(); // after creation of a new Module
  }
  
  if (current.first == "process") {
    Process* p = (Process*) last_vertex;
    last_vertex = (Vertex*) p->get_node();
  }
}

void CUBEXMLParser::parse_thread(ItemType& parent, ItemType& current) {
  if (parent.first == "thread" && current.first == "name" ) {
    builder_memo.push_back(data);
    int id;
    string name = builder_memo[0];
    if (last_vertex != NULL) { // should has a parent
      Process* p = (Process*) last_vertex;
      id = p->get_id();
      if (parent.second.size() < 4) 
	throw CUBException("thread attribute locid not found");
      int locid = atoi(parent.second[3].c_str());
      if (name.empty()) {
	name = "Thread ";
	//name += parent.second[1];
	char tid[10];
	sprintf(tid, "%d", thrd_counter);
	name += tid;
	thrd_counter++;
      }
      last_vertex = cube->create_thrd(locid, get_attr_id(parent), name, id);
    }
    else throw CUBException("Parser: thread's parent = NULL");
    builder_memo.clear(); // after creation of a new Module
  }
  if (current.first == "thread") {
    Thread* t = (Thread*) last_vertex;
    last_vertex = (Vertex*) t->get_proc();
  }
}

/* parse severity values */
void CUBEXMLParser::parse_row(ItemType& parent, ItemType& current) {
  double val = 0.0;
  int thrd_index = 0;

  if (current.first != "row") return;
  int met_id  = get_attr_id(parent);
  int cnode_id = get_attr_id(current);
  istrstream in(data.c_str());
  in.setf(ios::fixed, ios::floatfield);
  while(in >> val) {
    cube->set_sev(met_id, cnode_id, thrd_index, val);
    thrd_index++;
  }
  return;
}

// switch to the parser's event-handlers.
void startElement(void* parser, const xmlChar *name, const xmlChar **atts) {
  ((CUBEXMLParser*)(parser))->start_element(name, atts);
}

void endElement(void* parser, const xmlChar *name) {
  ((CUBEXMLParser*)(parser))->end_element(name);
}

void characters(void* parser, const xmlChar* text, int length) {
  string content((char*)text, length);
  ((CUBEXMLParser*)(parser))->data_content(content);
  
}
