/*
	HARNESS G_HCORE
	HARNESS FT_MPI

	Innovative Computer Laboratory,
	University of Tennessee,
	Knoxville, TN, USA.

	harness@cs.utk.edu

 --------------------------------------------------------------------------

 Authors:	
 			Antonin Bukovsky <tone@cs.utk.edu>
			Graham E Fagg <fagg@cs.utk.edu>

 --------------------------------------------------------------------------

                              NOTICE

 Permission to use, copy, modify, and distribute this software and
 its documentation for any purpose and without fee is hereby granted
 provided that the above copyright notice appear in all copies and
 that both the copyright notice and this permission notice appear in
 supporting documentation.

 Neither the University of Tennessee nor the Authors make any
 representations about the suitability of this software for any
 purpose.  This software is provided ``as is'' without express or
 implied warranty.

 HARNESS, HARNESS G_HCORE and  FT_MPI was funded in part by the 
 U.S. Department of Energy.

*/

#include "mpi.h"
#include "ft-mpi-ddt-sys.h"
#include "ft-mpi-attr.h"
#include <rpc/xdr.h>
#include "debug.h"


/* Subroutines implemented in this file: 

   Management routines:
   - ftmpi_ddt_create_list:          allocates memory for the ddt hash-list
   - ftmpi_ddt_add_root:             allocate the add-root struct if required
   - ftmpi_ddt_add_element:          allocate an element-struct for the linked list
   - ftmpi_ddt_get_root:             get the root struct for a ddt
   - ftmpi_ddt_get_element:          get next element-struct in the linked list
   - ftmpi_ddt_init:                 initializes the ddt stuff 
                                     (bdt list, ddt hash table etc.,
                                     defines the basic datatypes).
   - ftmpi_ddt_finalize:             free all structures
   - ftmpi_ddt_remove_all_but_basic: guess what it does :-)
   - ftmpi_ddt_reset_ref_counts:     guess again :-)
   - ftmpi_ddt_increment_dt_uses     increase the ref-count of the dt
   - ftmpi_ddt_decrement_dt_uses     decrease the ref-count of the dt
   - ftmpi_dt_set_size               set's the length for each basic dt
   - ftmpi_dt_set_xdr_size           set's the length of each basic dt in xdr-format
   - ftmpi_ddt_set_xdr_extent

   Routines moving ddt's to contiguous buffers:
   - ftmpi_ddt_write_dt_to_buffer:        compactes a ddt from user mem into a buffer
   - ftmpi_ddt_read_dt_from_buffer:       reads a (compacted) ddt from a buffer to user 
                                          memory
   - ftmpi_ddt_write_dt_to_buffer_block:  writes a ddt to a compact buffer, but
                                          the basic datatypes are sorted (for xdr).
   - ftmpi_ddt_read_dt_from_buffer_block: reads a ddt from a buffer, which is
                                          sorted according to its basic datatypes


   - ftmpi_ddt_get_first_offset           Get the real starting point of the ddt
   - ftmpi_ddt_copy_ddt_to_ddt            copy data described in one ddt into another 
                                          the buffer described by another ddt
   - ftmpi_ddt_get_element_count          used in MPI_Get_elements
   - ftmpi_ddt_check_ddt

   Data conversion routines
   - ftmpi_ddt_xdr_encode_dt_to_buffer:   general data encoding function
   - ftmpi_ddt_xdr_decode_dt_to_buffer:   general data decoding function
   - ftmpi_ddt_xdr_vector_code
   - ftmpi_ddt_xdr_code
   - ftmpi_ddt_decode_size_det
   - ftmpi_ddt_encode_size_det
   - ftmpi_ddt_bs_xdr_det

   - ftmpi_ddt_check_array
   - ftmpi_ddt_write_read_size_det        find out exact size to be sent/recv
   - ftmpi_ddt_get_remote_size


   Attribute caching functions for dt's (not yet in use)
   - ftmpi_type_set_attr                  Attach an attribute to a dt 
   - ftmpi_type_del_attr                  Detach an attribute from a dt

   - ftmpi_ddt_dt_mode
   - ftmpi_ddt_get_bdt_sizes
   - ftmpi_ddt_set_dt_mode
   - ftmpi_mpi_type_xdr_size

   MPI-2 type-decoding functions
   - ftmpi_ddt_set_type
   - ftmpi_ddt_set_args
   - ftmpi_ddt_get_args

   Data conversion for long-double 
   - ftmpi_ddt_convert_long_double
   - ftmpi_ddt_copy_bits
   - dump_memory_order
   - swap_bytes
   - swap_words

*/

/* #define DEBUG_DDT_INIT 1 */

int FTMPI_DDT_B_DT_ADD = 0;
int FTMPI_DDT_MPI_BUFFER_SIZE = 0;
char * FTMPI_DDT_MPI_BUFFER;

FTMPI_DDT_ITEM * FTMPI_DDT_BASIC_DATATYPE[FTMPI_DDT_B_DT_MAX];

FTMPI_DDT_R_ITEM ** FTMPI_DDT_ROOT = NULL;
int FTMPI_DDT_INIT = 0;
int FTMPI_DDT_HASH_SIZE= -1;
int FTMPI_DDT_ROOT_VAL = -1;
int FTMPI_DDT_NEXT_DATATYPE = 500,FTMPI_DDT_INIT_CREATE = 0,
  FTMPI_DDT_DATATYPE_HANDLE,FTMPI_DDT_DATATYPE_BASIC_INFO,
  FTMPI_DDT_B_DATATYPE_TOTAL=0,FTMPI_DDT_WILL_DO_BS_XDR=0;

static FTMPI_DDT_CODE_INFO ftmpi_ddt_my_longdouble;

/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
int ftmpi_ddt_create_list(int hash_size)
{
  int i;

  if(FTMPI_DDT_ROOT ==  NULL){
    if(hash_size <= 0){
      printf("ERROR:  Invalid Hash Size\n");
      return(-1);
    }

#ifdef DEBUG_DDT
    printf("CREATING HASH LIST ...");
#endif
    FTMPI_DDT_ROOT = (FTMPI_DDT_R_ITEM **)_MALLOC(sizeof(FTMPI_DDT_R_ITEM *)*hash_size);
    if ( FTMPI_DDT_ROOT == NULL ) {
      fprintf(stderr, "FTMPI_ddt_create_list: could not allocate memory\n");
      return ( MPI_ERR_INTERN );
    }
#ifdef DEBUG_DDT
    printf("DONE\n");
#endif
    for (i=0;i<hash_size;i++){
      FTMPI_DDT_ROOT[i] = NULL;
    }

    FTMPI_DDT_ROOT_VAL = 460917;
    FTMPI_DDT_HASH_SIZE = hash_size;

    return(FTMPI_DDT_ROOT_VAL);
  }
  else {
    printf("ERROR:  LIST ALREADY EXISTS\n");
    return(-1);
  }
}
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
FTMPI_DDT_R_ITEM * ftmpi_ddt_add_root(int ddt_id,int add)
{
  int i,j;
  FTMPI_DDT_R_ITEM * temp = NULL;

  /* first check if the ddt already exists */

  if(ddt_id%FTMPI_DDT_HASH_SIZE < 0){
    printf("ERROR:  INVALID DDT ID\n");
    return(NULL);
  }

  /* element may exist --- need to check */
  temp = FTMPI_DDT_ROOT[ddt_id%FTMPI_DDT_HASH_SIZE];
  if(temp != NULL){
    if(temp->dt == ddt_id){
      printf("ERROR: DDT_ID %d already exists\n",ddt_id);
      return(NULL);
    }
    else {
      while(temp->next != NULL){
        temp = temp->next;
        if(temp->dt == ddt_id){
          printf("ERROR: DDT_ID %d already exists\n",ddt_id);
          return(NULL);
        }
      } 
    }
    

    /* at the last root */
    temp->next = (FTMPI_DDT_R_ITEM*)_MALLOC(sizeof(FTMPI_DDT_R_ITEM));
    temp = temp->next;
    temp->next = NULL;
    if(temp == NULL){
      printf("----- !!! NO MORE MEMORY !!! -----\n");
      exit(-1);
    }
  }
  else {
    /* inserting first root */
    temp = (FTMPI_DDT_R_ITEM*)_MALLOC(sizeof(FTMPI_DDT_R_ITEM));          
    FTMPI_DDT_ROOT[ddt_id%FTMPI_DDT_HASH_SIZE] =  temp;
    temp->next = NULL;
  }

#ifdef DEBUG_DDT
  printf("ADDING A ROOT %d @ %x\n",ddt_id,temp);
#endif


  temp->dt = ddt_id;
  temp->size = -999;
  temp->extent = -999;
  temp->last = -999;
  temp->hash = -999;
  temp->commited = 0;
  temp->bs_xdr_count = -1;
  temp->bs_xdr_type = -1;


  if(add == 1){
    temp->root_add =_MALLOC(sizeof(FTMPI_DDT_RA_ITEM));          
    temp->root_add->first_e = NULL;
    temp->root_add->last_e = NULL;
    for(i=0;i<FTMPI_DDT_B_DT_MAX;i++){
      temp->root_add->type_count[i] = 0;
    }
    temp->root_add->t_comp = 0;
    for(i=0;i<FTMPI_DDT_COMP_MAX;i++){
      for(j=0;j<6;j++){
        temp->root_add->comp[i][j] = 0;
      }
    }
    temp->root_add->i = NULL;
    temp->root_add->a = NULL;
    temp->root_add->d = NULL;
    temp->root_add->create_type = -1;
    temp->root_add->ref_cnt = 1;
  }
  else 
    temp->root_add = NULL;

  return(temp);
}
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
FTMPI_DDT_E_ITEM * ftmpi_ddt_add_element(FTMPI_DDT_R_ITEM * root,int ddt)
{
  if(root == NULL){
    printf("ERROR: Don't know ROOT\n");
    return(NULL);
  }
  else {
    if(root->root_add->last_e == NULL){
      root->root_add->last_e = (FTMPI_DDT_E_ITEM *)_MALLOC(sizeof(FTMPI_DDT_E_ITEM));
      root->root_add->first_e = root->root_add->last_e;
      root->root_add->last_e->next = NULL;
    }
    else {
      root->root_add->last_e->next = (FTMPI_DDT_E_ITEM *)_MALLOC(sizeof(FTMPI_DDT_E_ITEM));
      root->root_add->last_e = root->root_add->last_e->next;
      root->root_add->last_e->next = NULL;
    }
  }

#ifdef DEBUG_DDT
  printf("ADDING A ELEMENT %d @ %x\n",ddt,root->root_add->last_e);
#endif
  root->root_add->last_e->dt = ddt;
  root->root_add->last_e->size = -999;
  root->root_add->last_e->extent = -999;
  root->root_add->last_e->count = -999;
  root->root_add->last_e->padding = -999;
  return(root->root_add->last_e);
}
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
FTMPI_DDT_R_ITEM * ftmpi_ddt_get_root(int ddt_id)
{
  FTMPI_DDT_R_ITEM * temp;

  if(ddt_id%FTMPI_DDT_HASH_SIZE < 0){
    return(NULL);
  }

  if(FTMPI_DDT_ROOT == NULL){
    return(NULL);
  }

  temp = FTMPI_DDT_ROOT[ddt_id%FTMPI_DDT_HASH_SIZE];
  while(temp != NULL){
    if(temp->dt == ddt_id){
#ifdef DEBUG_DDT
      printf("FOUND ROOT %d @ %x\n",temp->dt,temp);
#endif
      return(temp);
    }
    temp = temp->next;
  }
  return(NULL);
}
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
FTMPI_DDT_E_ITEM * ftmpi_ddt_get_element(FTMPI_DDT_R_ITEM * root,
					 FTMPI_DDT_E_ITEM * last_element)
{

  if(root == NULL){
    printf("ERROR: improper ROOT ptr\n");
    return(NULL);
  }

  if(last_element == NULL){
#ifdef DEBUG_DDT
    printf("FOUND FIRST element %d @ %x\n",root->root_add->first_e->dt,
	   root->root_add->first_e);
#endif
    return(root->root_add->first_e);
  }
  else {
#ifdef DEBUG_DDT
    if(last_element->next != NULL){
      printf("FOUND NEXT element %d @ %x\n",last_element->next->dt,
	     last_element->next);
    }
#endif
    return(last_element->next);
  }
}
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
void ftmpi_ddt_init()
{   
  int i;


  if (!FTMPI_DDT_INIT) {
    int block[3],type[3],ret,reth;
    MPI_Aint disp[3];

      ftmpi_convinfo_mysettings(&ftmpi_ddt_my_longdouble.ld_flag);
    ftmpi_ddt_my_longdouble.bs = 0;
    ftmpi_ddt_my_longdouble.mode_l = -1;

    FTMPI_DDT_MPI_BUFFER = (char*)_MALLOC(1024*1);
    FTMPI_DDT_MPI_BUFFER_SIZE = 1024;

    for(i=0;i<FTMPI_DDT_B_DT_MAX;i++){
      FTMPI_DDT_BASIC_DATATYPE[i] = (FTMPI_DDT_ITEM *)_MALLOC(sizeof(FTMPI_DDT_ITEM));
      FTMPI_DDT_BASIC_DATATYPE[i]->size = -1;
      FTMPI_DDT_BASIC_DATATYPE[i]->extent = -1;
      FTMPI_DDT_BASIC_DATATYPE[i]->hash = -1;
    } 
  
    FTMPI_DDT_B_DATATYPE_TOTAL=1;
    for(i=1;i<FTMPI_DDT_B_DT_MAX;i++){
      ftmpi_ddt_set_size(i,&FTMPI_DDT_BASIC_DATATYPE[i]->size);
      ftmpi_ddt_set_xdr_size(i,&FTMPI_DDT_BASIC_DATATYPE[i]->xdr_size);
      ftmpi_mpi_type_extent(i,&FTMPI_DDT_BASIC_DATATYPE[i]->extent);
      ftmpi_ddt_set_xdr_extent(i,&FTMPI_DDT_BASIC_DATATYPE[i]->xdr_extent);
      FTMPI_DDT_BASIC_DATATYPE[i]->hash = (i+1);
      FTMPI_DDT_B_DATATYPE_TOTAL++;
    }

    /* Set up basic derived datatypes*/

   FTMPI_DDT_INIT = 1;

   /***********************************************************************************/
   block[0] = 1;
   block[1] = 1;
   block[2] = 1;

   /* MPI_FLOAT_INT - 500 */
   {
     struct{float a;int b;} str;
     disp[0] = (MPI_Aint)((char*)(&str.a) - (char*)(&str));
     disp[1] = (MPI_Aint)((char*)&str.b - (char*)&str);
     disp[2] = sizeof(struct{float a;int b;});

     type[0] = MPI_FLOAT;
     type[1] = MPI_INT;
     type[2] = MPI_UB;

     ret = ftmpi_mpi_type_struct(3,block,disp,type,&reth);
     ftmpi_mpi_type_commit(&reth);
#ifdef DEBUG_DDT_INIT
     printf("BLOCK %2d %2d  -  ",block[0],block[1]); 
     printf("DISP  %2d %2d  -  ",disp[0],disp[1]); 
     printf("TYPE  %2d %2d  ****  ",type[0],type[1]);
     printf("%-22s 500 - %2d\n","MPI_FLOAT_INT",MPI_FLOAT_INT,reth);
#endif
     FTMPI_DDT_B_DT_ADD++;
   }

   /* MPI_DOUBLE_INT - 501 */
   {
     struct{double a;int b;} str;
     disp[0] = (MPI_Aint)((char*)&str.a - (char*)&str);
     disp[1] = (MPI_Aint)((char*)&str.b - (char*)&str);
     disp[2] = sizeof(struct{double a;int b;});

     type[0] = MPI_DOUBLE;
     type[1] = MPI_INT;
     type[2] = MPI_UB;

     ret = ftmpi_mpi_type_struct(3,block,disp,type,&reth);
     ftmpi_mpi_type_commit(&reth);
#ifdef DEBUG_DDT_INIT
     printf("BLOCK %2d %2d  -  ",block[0],block[1]); 
     printf("DISP  %2d %2d  -  ",disp[0],disp[1]); 
     printf("TYPE  %2d %2d  ****  ",type[0],type[1]);
     printf("%-22s %d - %2d\n","MPI_DOUBLE_INT",MPI_DOUBLE_INT,reth);
#endif
     FTMPI_DDT_B_DT_ADD++;
   }

   /* MPI_LONG_INT - 502 */
   {
     struct{long a;int b;} str;
     disp[0] = (MPI_Aint)((char*)&str.a - (char*)&str);
     disp[1] = (MPI_Aint)((char*)&str.b - (char*)&str);
     disp[2] = sizeof(struct{long a;int b;});
     type[0] = MPI_LONG;
     type[1] = MPI_INT;
     type[2] = MPI_UB;

     ret = ftmpi_mpi_type_struct(3,block,disp,type,&reth);
     ftmpi_mpi_type_commit(&reth);
#ifdef DEBUG_DDT_INIT
     printf("BLOCK %2d %2d  -  ",block[0],block[1]); 
     printf("DISP  %2d %2d  -  ",disp[0],disp[1]); 
     printf("TYPE  %2d %2d  ****  ",type[0],type[1]);
     printf("%-22s %d - %2d\n","MPI_LONG_INT",MPI_LONG_INT,reth);
#endif
     FTMPI_DDT_B_DT_ADD++;
   }

   /* MPI_2INT - 503 */
   {
     struct{int a;int b;} str;
     disp[0] = (MPI_Aint)((char*)&str.a - (char*)&str);
     disp[1] = (MPI_Aint)((char*)&str.b - (char*)&str);
     disp[2] = sizeof(struct{int a;int b;});
     type[0] = MPI_INT;
     type[1] = MPI_INT;
     type[2] = MPI_UB;

     ret = ftmpi_mpi_type_struct(3,block,disp,type,&reth);
     ftmpi_mpi_type_commit(&reth);
#ifdef DEBUG_DDT_INIT
     printf("BLOCK %2d %2d  -  ",block[0],block[1]); 
     printf("DISP  %2d %2d  -  ",disp[0],disp[1]); 
     printf("TYPE  %2d %2d  ****  ",type[0],type[1]);
     printf("%-22s %d - %2d\n","MPI_2INT",MPI_2INT,reth);
#endif
     FTMPI_DDT_B_DT_ADD++;
   }

   /* MPI_SHORT_INT - 504 */
   {
     struct{short a;int b;} str;
     disp[0] = (MPI_Aint)((char*)&str.a - (char*)&str);
     disp[1] = (MPI_Aint)((char*)&str.b - (char*)&str);
     disp[2] = sizeof(struct{short a;int b;});
     type[0] = MPI_SHORT;
     type[1] = MPI_INT;
     type[2] = MPI_UB;

     ret = ftmpi_mpi_type_struct(3,block,disp,type,&reth);
     ftmpi_mpi_type_commit(&reth);
#ifdef DEBUG_DDT_INIT
     printf("BLOCK %2d %2d  -  ",block[0],block[1]); 
     printf("DISP  %2d %2d  -  ",disp[0],disp[1]); 
     printf("TYPE  %2d %2d  ****  ",type[0],type[1]);
     printf("%-22s %d - %2d\n","MPI_SHORT_INT",MPI_SHORT_INT,reth);
#endif
     FTMPI_DDT_B_DT_ADD++;
   }

   /* MPI_LONG_DOUBLE_INT - 505 */
   {
     struct{long double a;int b;} str;
     disp[0] = (MPI_Aint)((char*)&str.a - (char*)&str);
     disp[1] = (MPI_Aint)((char*)&str.b - (char*)&str);
     disp[2] = sizeof(struct{long double a;int b;});
     type[0] = MPI_LONG_DOUBLE;
     type[1] = MPI_INT;
     type[2] = MPI_UB;

     ret = ftmpi_mpi_type_struct(3,block,disp,type,&reth);
     ftmpi_mpi_type_commit(&reth);
#ifdef DEBUG_DDT_INIT
     printf("BLOCK %2d %2d  -  ",block[0],block[1]); 
     printf("DISP  %2d %2d  -  ",disp[0],disp[1]); 
     printf("TYPE  %2d %2d  ****  ",type[0],type[1]);
     printf("%-22s %d - %2d\n","MPI_LONG_DOUBLE_INT",MPI_LONG_DOUBLE_INT,reth);
#endif
     FTMPI_DDT_B_DT_ADD++;
   }

   /***********************************************************************************/
   /* MPI_2REAL - 506 */
   block[0] = 2;
   {
     disp[0] = 0;
     type[0] = MPI_REAL;

     ret = ftmpi_mpi_type_struct(1,block,disp,type,&reth);
     ftmpi_mpi_type_commit(&reth);
#ifdef DEBUG_DDT_INIT
     printf("BLOCK %2d     -  ",block[0]); 
     printf("DISP  %2d     -  ",disp[0]); 
     printf("TYPE  %2d     ****  ",type[0]);
     printf("%-22s %d - %2d\n","MPI_2REAL",MPI_2REAL,reth);
#endif
     FTMPI_DDT_B_DT_ADD++;
   }


   /* MPI_2DOUBLE_PRECISION - 507 */
   {
     disp[0] = 0;
     type[0] = MPI_DOUBLE_PRECISION;

     ret = ftmpi_mpi_type_struct(1,block,disp,type,&reth);
     ftmpi_mpi_type_commit(&reth);
#ifdef DEBUG_DDT_INIT
     printf("BLOCK %2d     -  ",block[0]); 
     printf("DISP  %2d     -  ",disp[0]); 
     printf("TYPE  %2d     ****  ",type[0]);
     printf("%-22s %d - %2d\n","MPI_2DOUBLE_PRECISION",
	    MPI_2DOUBLE_PRECISION,reth);
#endif
     FTMPI_DDT_B_DT_ADD++;
   }

   /* MPI_2INTEGER - 508 */
   {
     disp[0] = 0;
     type[0] = MPI_INTEGER;

     ret = ftmpi_mpi_type_struct(1,block,disp,type,&reth);
     ftmpi_mpi_type_commit(&reth);
#ifdef DEBUG_DDT_INIT
     printf("BLOCK %2d     -  ",block[0]); 
     printf("DISP  %2d     -  ",disp[0]); 
     printf("TYPE  %2d     ****  ",type[0]);
     printf("%-22s %d - %2d\n","MPI_2INTEGER",MPI_2INTEGER,reth);
#endif
     FTMPI_DDT_B_DT_ADD++;
   }
   /***********************************************************************************/
   /* MPI_COMPLEX - 509 */
   {
     disp[0] = 0;
     type[0] = MPI_REAL;

     ret = ftmpi_mpi_type_struct(1,block,disp,type,&reth);
     ftmpi_mpi_type_commit(&reth);
#ifdef DEBUG_DDT_INIT
     printf("BLOCK %2d     -  ",block[0]); 
     printf("DISP  %2d     -  ",disp[0]); 
     printf("TYPE  %2d     ****  ",type[0]);
     printf("%-22s %d - %2d\n","MPI_COMPLEX",MPI_COMPLEX,reth);
#endif
     FTMPI_DDT_B_DT_ADD++;
   }

   /* MPI_DOUBLE_COMPLEX - 510 */
   {
     disp[0] = 0;
     type[0] = MPI_DOUBLE_PRECISION;

     ret = ftmpi_mpi_type_struct(1,block,disp,type,&reth);
     ftmpi_mpi_type_commit(&reth);
#ifdef DEBUG_DDT_INIT
     printf("BLOCK %2d     -  ",block[0]); 
     printf("DISP  %2d     -  ",disp[0]); 
     printf("TYPE  %2d     ****  ",type[0]);
     printf("%-22s %d - %2d\n","MPI_DOUBLE_COMPLEX",
	    MPI_DOUBLE_COMPLEX,reth);
#endif
     FTMPI_DDT_B_DT_ADD++;
   }
   
   /***********************************************************************************/
   block[0] = 4;
   /* MPI_2COMPLEX - 511 */
   {
     disp[0] = 0;
     type[0] = MPI_REAL;

     ret = ftmpi_mpi_type_struct(1,block,disp,type,&reth);
     ftmpi_mpi_type_commit(&reth);
#ifdef DEBUG_DDT_INIT
     printf("BLOCK %2d     -  ",block[0]); 
     printf("DISP  %2d     -  ",disp[0]); 
     printf("TYPE  %2d     ****  ",type[0]);
     printf("%-22s %d - %2d\n","MPI_2COMPLEX",MPI_2COMPLEX,reth);
#endif
     FTMPI_DDT_B_DT_ADD++;
   }

   /* MPI_2DOUBLE_COMPLEX - 512 */
   {
     disp[0] = 0;
     type[0] = MPI_DOUBLE_PRECISION;

     ret = ftmpi_mpi_type_struct(1,block,disp,type,&reth);
     ftmpi_mpi_type_commit(&reth);
#ifdef DEBUG_DDT_INIT
     printf("BLOCK %2d     -  ",block[0]); 
     printf("DISP  %2d     -  ",disp[0]); 
     printf("TYPE  %2d     ****  ",type[0]);
     printf("%-22s %d - %2d\n","MPI_2DOUBLE_COMPLEX",
	    MPI_2DOUBLE_COMPLEX,reth);
#endif
     FTMPI_DDT_B_DT_ADD++;
   }
   /***********************************************************************************/
#ifdef HAVE_LONG_LONG

   block[0] = 1;
   block[1] = 1;
   block[2] = 1;

   /* MPI_LONG_LONG_INT - 513 */
   {
     struct{long long a;int b;} str;
     disp[0] = (MPI_Aint)(&str.a - str);
     disp[1] = (MPI_Aint)(&str.b - &str);
     disp[2] = sizeof(struct{long long a;int b;});
     type[0] = MPI_LONG LONG;
     type[1] = MPI_INT;
     type[2] = MPI_UB;

     ret = ftmpi_mpi_type_struct(3,block,disp,type,&reth);
     ftmpi_mpi_type_commit(&reth);
#ifdef DEBUG_DDT_INIT
     printf("BLOCK %2d %2d  -  ",block[0],block[1]); 
     printf("DISP  %2d %2d  -  ",disp[0],disp[1]); 
     printf("TYPE  %2d %2d  ****  ",type[0],type[1]);
     printf("%-22s %d - %2d\n","MPI_LONG_LONG_INT",MPI_LONG_LONG_INT,reth);
#endif
     FTMPI_DDT_B_DT_ADD++;
   }
#endif
   /***********************************************************************************/


#ifdef DEBUG_DDT_INIT
   printf("FTMPI_DDT_B_DT_ADD = %d\n",FTMPI_DDT_B_DT_ADD);
#endif
#ifdef DEBUG_DDT
    for(i=1;i<FTMPI_DDT_B_DATATYPE_TOTAL;i++){
      printf("BASIC DATATYPE %2d T_EXT: %2d  --  T_SIZE: %2d HASH: %4x\n",
	     i,FTMPI_DDT_BASIC_DATATYPE[i]->extent,
	     FTMPI_DDT_BASIC_DATATYPE[i]->size,
	     FTMPI_DDT_BASIC_DATATYPE[i]->hash);
    } 
#endif
  } /* if FTMPI_DDT_INIT not set */
}   
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
void ftmpi_ddt_remove_all_but_basic(void)
{
  int i;
  FTMPI_DDT_R_ITEM * rc = NULL,* rt = NULL;
  FTMPI_DDT_E_ITEM * ec = NULL,* et = NULL;


  for(i=0;i<FTMPI_DDT_HASH_SIZE;i++){
    rc = FTMPI_DDT_ROOT[i];
    while(rc != NULL){
      ec = rc->root_add->first_e;
      while(ec != NULL){
        et = ec->next;
        _FREE(ec);
        ec = et;
      }
      rt = rc->next;
      if(rc->root_add->i != NULL)_FREE(rc->root_add->i);
      if(rc->root_add->a != NULL)_FREE(rc->root_add->a);
      if(rc->root_add->d != NULL)_FREE(rc->root_add->d);
      _FREE(rc->root_add);
      _FREE(rc);
      rc = rt;
    }
    FTMPI_DDT_ROOT[i] = NULL;
  }
  FTMPI_DDT_NEXT_DATATYPE = ((FTMPI_DDT_NEXT_DATATYPE/100)+1)*100;
}
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
void ftmpi_ddt_finalize()
{
  int i;
  _FREE(FTMPI_DDT_MPI_BUFFER);
  
  ftmpi_ddt_remove_all_but_basic();
  _FREE(FTMPI_DDT_ROOT);

  for(i=0;i<FTMPI_DDT_B_DT_MAX;i++){
    _FREE(FTMPI_DDT_BASIC_DATATYPE[i]);
  }
}

/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
void ftmpi_ddt_reset_ref_counts()
{
  int i;
  FTMPI_DDT_R_ITEM * rc = NULL;

  for(i=0;i<FTMPI_DDT_HASH_SIZE;i++){
    rc = FTMPI_DDT_ROOT[i];
    while(rc != NULL){
      rc->uses = 1;
      rc = rc->next;
    }
  }
}
/***************************************************************************/
/***************************************************************************
****************************************************************************
**  writes a datatype to a buffer:
**  buffer - pointer to where data is copied into
**  buffer_size - size of an available buffer
**  datatype - datatype to be copied
**  count - number of datatypes to be copied
**  data - pointer from where data is being copied
**  spaces - 0 for now spaces , 1 for spaces
**  last_dt - used internaly, needs to be NULL to begin
**  NOT ABLE TO CONTINUE IF NOT ENOUGH BUFFER SUPPLIED TO COPY EVERYTHING
**  BUT VERY EFFICIENT.  Definitely done.
****************************************************************************
***************************************************************************/
int ftmpi_ddt_write_dt_to_buffer(char *buffer, int buffer_size, int dt,
				 int count, char *data, int spaces, int *last_dt,
				 int bswap_var)
{
  int i,ret,last_dt_a[2],offset=0,j,k,l;
  FTMPI_DDT_R_ITEM * xtemp = NULL;
  FTMPI_DDT_E_ITEM * temp = NULL;
  FTMPI_DDT_mem_CP_BS_vars

  if(FTMPI_DDT_INIT == 0){
    ftmpi_ddt_init();
  }

  if (count == 0 ) {
#ifdef DEBUG_DDT
    printf("write_dt_to_buf: called with count=0\n");
#endif
    return ( 0 );
  }

  if(last_dt == NULL){
    last_dt = last_dt_a;
    last_dt[0] = 0;
    last_dt[1] = -1;
  }
   
  offset = last_dt[0];

  if(dt < FTMPI_DDT_B_DATATYPE_TOTAL){
    if(FTMPI_DDT_BASIC_DATATYPE[dt]->size * count <= buffer_size){
      dt_size = ret = FTMPI_DDT_BASIC_DATATYPE[dt]->size * count;
    }
    else {
      dt_size = (buffer_size/FTMPI_DDT_BASIC_DATATYPE[dt]->size)*
	FTMPI_DDT_BASIC_DATATYPE[dt]->size;
      ret = -2;
    }

    dt_dest = buffer;
    dt_src = data;

    if(bswap_var == 0){
      dt_type = 1;
    }
    else {
      dt_type = FTMPI_DDT_BASIC_DATATYPE[dt]->size;
    }
    FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
    return(ret);
  }

 
  xtemp = ftmpi_ddt_get_root(dt);

  if(xtemp != NULL){
#ifdef DEBUG_DDT
    printf("%d EXTENT == %d %d\n",count,xtemp->extent,xtemp->last);
#endif
    for(i=0;i<count;i++){
      if(xtemp->root_add->t_comp == -1){
#ifdef DEBUG_DDT
        printf("NOT DOING THE FAST THING\n");
#endif
        temp = ftmpi_ddt_get_element(xtemp,temp);
        while(temp != NULL){
          if(temp->dt < FTMPI_DDT_B_DATATYPE_TOTAL){
            if(spaces){
              if(last_dt[1] != temp->dt && last_dt[1] != -1){
                last_dt[0] = ((last_dt[0]-1+sizeof(int))/sizeof(int))*(sizeof(int));
              } 
            }  
            if((last_dt[0] + (temp->size*temp->count)) <= buffer_size){
	      /*********************************************************************/
              dt_dest = &buffer[last_dt[0]];
              dt_src = &data[temp->padding + (i*xtemp->extent)];
              dt_size = (temp->size * temp->count);
              if(bswap_var == 0){
                dt_type = 1;
              }
              else {
                dt_type = FTMPI_DDT_BASIC_DATATYPE[temp->dt]->size;
              }

              FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
		/*************************************************************/
              last_dt[0] = last_dt[0] + temp->size * temp->count;
            }
            else {
	      /***************************************************************/
              dt_dest = &buffer[last_dt[0]];
              dt_src = &data[temp->padding+(i*xtemp->extent)];
              dt_size = (temp->size - ((last_dt[0] + temp->size) - buffer_size));
              if(bswap_var == 0){
                dt_type = 1;
              }
              else {
                dt_type = FTMPI_DDT_BASIC_DATATYPE[temp->dt]->size;
              }

              FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
		/************************************************************/
              printf("Buffer is full!!!\n");
              return(-2);
            }
            last_dt[1] = temp->dt;
          }
          else {
            ret = ftmpi_ddt_write_dt_to_buffer(buffer,buffer_size,temp->dt,
					       temp->count,
					       &data[xtemp->extent*i+temp->padding],
					       spaces,last_dt,bswap_var);
            if(ret < 0){
              return (ret);
            }
          }
          temp = ftmpi_ddt_get_element(xtemp,temp);
        }
      } 
      else {
#ifdef DEBUG_DDT
        printf("PERFORMING FAST COPY\n");
#endif
        /*
          0 - datatype
          1 - count
          2 - displacement
          3 - extent
          4 - number of items
          5 - number of repeats
        */



#ifdef NO_DT_CARE
        for(j=0;j<xtemp->t_b_comp;j += xtemp->b_comp[j][4]){
          for(k=0;k<xtemp->b_comp[j][5];k++){
            for(l=0;l<xtemp->b_comp[j][4];l++){
              if(spaces){  /* if there should be spaces */
                if(xtemp->b_comp[j+l-1][0] != xtemp->b_comp[j+l][0] && (j-1) != -1 ){
                  offset = ((offset-1+sizeof(int))/sizeof(int))*(sizeof(int));
                }
              }
              if(offset + xtemp->b_comp[j+l][0]*xtemp->b_comp[j+l][1] <= buffer_size){
		/********************************************************************/
                dt_dest = &buffer[offset];
                dt_src = &data[xtemp->b_comp[j+l][2]+(i*xtemp->extent)+
			       (k*xtemp->b_comp[j+xtemp->b_comp[j][4]-1][3])];
                dt_size = xtemp->b_comp[j+l][1]*xtemp->b_comp[j+l][0];
                if(bswap_var == 0){
                  dt_type = 1;
                }
                else {
                  dt_type = xtemp->b_comp[j+l][0];
                }

                FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
		  /******************************************************************/
		  offset += xtemp->b_comp[j+l][1]*(xtemp->b_comp[j+l][0]);
              }
              else {
		/*******************************************************************/
                dt_dest = &buffer[offset];
                dt_src = &data[xtemp->b_comp[j+l][2]+(i*xtemp->extent)+
			       (k*xtemp->b_comp[j+xtemp->b_comp[j][4]-1][3])];
                dt_size = buffer_size - offset;
                if(bswap_var == 0){
                  dt_type = 1;
                }
                else {
                  dt_type = xtemp->b_comp[j+l][0];
                }

                FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
		  /*****************************************************************/
                printf("W - Buffer is full!!!\n");
                return(-2);
              }
            }
          }
        }
#else
/* XXXXX */
        for(j=0;j<xtemp->root_add->t_comp;j += xtemp->root_add->comp[j][4]){
          for(k=0;k<xtemp->root_add->comp[j][5];k++){
            for(l=0;l<xtemp->root_add->comp[j][4];l++){
              if(spaces){  /* if there should be spaces */
                if(xtemp->root_add->comp[j+l-1][0] != xtemp->root_add->comp[j+l][0] && 
		   (j-1) != -1 ){
                  offset = ((offset-1+sizeof(int))/sizeof(int))*(sizeof(int));
                }
              }
              if(offset + FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size*
		 xtemp->root_add->comp[j+l][1] <= buffer_size){
		        /*
			 * 0 - datatype
			 * 1 - count
			 * 2 - displacement
			 * 3 - extent
			 * 4 - number of items
			 * 5 - number of repeats
			 **/
		/*******************************************************************/
                dt_dest = &buffer[offset];
                dt_src = &data[xtemp->root_add->comp[j+l][2]+(i*xtemp->extent)+
			       (k*(xtemp->root_add->comp[j+xtemp->root_add->comp[j][4]-1][3]))];
                /* 
		** dt_src = &data[xtemp->root_add->comp[j+l][2]+(i*xtemp->extent)+
		**   (k*(xtemp->root_add->comp[j+
		**   xtemp->root_add->comp[j][4]-1][2]-xtemp->root_add->comp[j][2]+
		**   xtemp->root_add->comp[j+xtemp->root_add->comp[j][4]-1][3]))]; 
		*/

                dt_size = xtemp->root_add->comp[j+l][1]*
		  (FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size);
                if(bswap_var == 0){
                  dt_type = 1;
                }
                else {
                  dt_type = FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size;
                }


                FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
		  /**************************************************************/
                offset += xtemp->root_add->comp[j+l][1]*
		  (FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size);
              }
              else {
		/*****************************************************************/
                dt_dest = &buffer[offset];
                dt_src = &data[xtemp->root_add->comp[j+l][2]+(i*xtemp->extent)+(k*xtemp->root_add->comp[j+xtemp->root_add->comp[j][4]-1][3])];
                dt_size = buffer_size - offset;
                if(bswap_var == 0){
                  dt_type = 1;
                }
                else {
                  dt_type = FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size;
                }


                FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
		  /****************************************************************/
                printf("Wx - Buffer is full!!!\n");
                return(-2);
              }
            }
          }
        }
#endif
        last_dt[0] = offset;
      } 
    }
#ifdef DEBUG_DDT
    printf("RETURNING %d\n",last_dt[0]);
#endif
    return(last_dt[0]);
  }
  printf("WR DATATYPE %d does NOT exist\n",dt); 
  return(-1);
}
/***************************************************************************
****************************************************************************
**  reads a datatype from a buffer into a data:
**  buffer - pointer from where datatype is copied
**  buffer_size - size of an available buffer
**  datatype - datatype to be copied
**  count - number of datatypes to be copied
**  data - pointer to where data is being copied
**  spaces - 0 for now spaces , 1 for spaces
**  last_dt - used internaly, needs to be NULL to begin
**  NOT ABLE TO CONTINUE IF NOT ENOUGH BUFFER SUPPLIED TO COPY EVERYTHING
**  BUT VERY EFFICIENT. Definitely done.
****************************************************************************
***************************************************************************/
int ftmpi_ddt_read_dt_from_buffer(char *from_buffer, int buffer_size,
				  int dt, int count, char *to_buffer,
				  int spaces, int *last_dt,
				  FTMPI_DDT_CODE_INFO *code_info)
{
  int i,ret,last_dt_a[2],offset=0,j,k,l;
  int bdt_size;
  FTMPI_DDT_R_ITEM * xtemp = NULL;
  FTMPI_DDT_E_ITEM * temp = NULL;
  FTMPI_DDT_mem_CP_BS_vars

  if(FTMPI_DDT_INIT == 0){
    ftmpi_ddt_init();
  }


  if (count == 0 ) {
#ifdef DEBUG_DDT
    printf("read_dt_to_buf: called with count=0\n");
#endif
    return ( 0 );
  }

  if(last_dt == NULL){
    last_dt =  last_dt_a;
    last_dt[0] = 0;
    last_dt[1]=-1;
  }

  offset = last_dt[0];


  /******************************************************************/
  if(dt < FTMPI_DDT_B_DATATYPE_TOTAL){
    dt_mode = ftmpi_ddt_dt_mode(dt,code_info);
    bdt_size = ftmpi_ddt_get_bdt_sizes(code_info,dt);

    if(bdt_size * count <= buffer_size){
      dt_size = ret = bdt_size * count;
    }
    else {
      dt_size = (buffer_size/bdt_size)*bdt_size;
      ret = -2;
    }

    dt_dest = to_buffer;
    dt_src = from_buffer;

    if(code_info->bs == 0) 
      dt_type = 1;
    else   
      dt_type = ftmpi_ddt_get_bdt_sizes(code_info,dt);

    FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)

    return(ret);
  }

  /***************************************************************/
  xtemp = ftmpi_ddt_get_root(dt);

  if(xtemp != NULL){
    for(i=0;i<count;i++){
      if(xtemp->root_add->t_comp == -1){
        temp = ftmpi_ddt_get_element(xtemp,temp);
        while(temp != NULL){
          if(temp->dt < FTMPI_DDT_B_DATATYPE_TOTAL){
            if(spaces){
              if(last_dt[1] != temp->dt && last_dt[1] != -1){
                last_dt[0] = ((last_dt[0]-1+sizeof(int))/sizeof(int))*(sizeof(int));
              } 
            }  

            dt_mode = ftmpi_ddt_dt_mode(dt,code_info);
            bdt_size = ftmpi_ddt_get_bdt_sizes(code_info,temp->dt);

            if((last_dt[0] + (bdt_size * temp->count )) <= buffer_size){
	      /********************************************************/
              dt_dest = &to_buffer[temp->padding + (i*xtemp->extent)];
              dt_src = &from_buffer[last_dt[0]];
              dt_size = (bdt_size * temp->count);
              if(code_info->bs == 0){
                dt_type = 1;
              }
              else {
                dt_type = bdt_size;
              }

              FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
              last_dt[0] = last_dt[0] + (bdt_size * temp->count);
	      /*****************************************************************/
            }
            else {
	      /*****************************************************************/
              dt_dest = &to_buffer[temp->padding];
              dt_src = &from_buffer[last_dt[0]];
              dt_size = bdt_size - ((last_dt[0] + bdt_size) - buffer_size);
              if(code_info->bs == 0){
                dt_type = 1;
              }
              else {
                dt_type = bdt_size;
              }

              FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
              printf("Buffer is full!!!\n");
	      /*****************************************************************/
              return(-2);
            }
            last_dt[1] = temp->dt;
          }
          else {
            ret = ftmpi_ddt_read_dt_from_buffer(from_buffer,buffer_size,
						temp->dt,temp->count,
						&to_buffer[xtemp->extent*i+temp->padding],
						spaces,last_dt,code_info);
            if(ret < 0){
              return (ret);
            }
            offset = ret;
          }
          temp = ftmpi_ddt_get_element(xtemp,temp);
        }
      } 
      else {
        /*
          0 - datatype
          1 - count
          2 - displacement
          3 - extent
          4 - number of items
          5 - number of repeats
        */
#ifdef NO_DT_CARE
        for(j=0;j<xtemp->t_b_comp;j += xtemp->b_comp[j][4]){
          for(k=0;k<xtemp->b_comp[j][5];k++){
            for(l=0;l<xtemp->b_comp[j][4];l++){
              if(spaces){
                if(xtemp->b_comp[j+l-1][0] != xtemp->b_comp[j+l][0] && (j-1) != -1 ){
                  offset = ((offset-1+sizeof(int))/sizeof(int))*(sizeof(int));
                }
              }
              if(offset + xtemp->b_comp[j+l][1]*xtemp->b_comp[j+l][0] <= buffer_size){
		/********************************************************************/
                dt_dest = &to_buffer[xtemp->b_comp[j+l][2]+(i*xtemp->extent)+
				     (k*xtemp->b_comp[j+xtemp->b_comp[j][4]-1][3])];
                dt_src = &from_buffer[offset];
                dt_size = xtemp->b_comp[j+l][1]*(xtemp->b_comp[j+l][0]);
                if(code_info->bs == 0){
                  dt_type = 1;
                }
                else {
                  dt_type = xtemp->b_comp[j+l][0];
                }

                /* NOTE: canot do the ftmpi_ddt_dt_mode ... because INT != LONG */

		/* dt_mode = ftmpi_ddt_dt_mode(dt,code_info); */
                FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
                offset += xtemp->b_comp[j+l][1]*(xtemp->b_comp[j+l][0]);
		/***************************************************************/
              }
              else {
		/****************************************************************/
                dt_dest = &to_buffer[xtemp->b_comp[j+l][2]+(i*xtemp->extent)+
				     (k*xtemp->b_comp[j+xtemp->b_comp[j][4]-1][3])];
                dt_src = &from_buffer[offset];
                dt_size = buffer_size - offset;
                if(code_info->bs == 0){
                  dt_type = 1;
                }
                else {
                  dt_type = xtemp->b_comp[j+l][0];
                }

                /* NOTE: canot do the ftmpi_ddt_dt_mode ... because INT != LONG */

		/* dt_mode = ftmpi_ddt_dt_mode(dt,code_info); */
                FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
                printf("R - Buffer is full -- AAA!!!\n");
                return(-2);
		/**************************************************************/
              }
            }
          }
        }
#else

        for(j=0;j<xtemp->root_add->t_comp;j += xtemp->root_add->comp[j][4]){
          for(k=0;k<xtemp->root_add->comp[j][5];k++){
            for(l=0;l<xtemp->root_add->comp[j][4];l++){
              if(spaces){  /* if there should be spaces */
                if(xtemp->root_add->comp[j+l-1][0] != xtemp->root_add->comp[j+l][0] && (j-1) != -1 ){
                  offset = ((offset-1+sizeof(int))/sizeof(int))*(sizeof(int));
                }
              }
              dt_mode = ftmpi_ddt_dt_mode(xtemp->root_add->comp[j+l][0],code_info);
              bdt_size = ftmpi_ddt_get_bdt_sizes(code_info,xtemp->root_add->comp[j+l][0]);

              if(offset + xtemp->root_add->comp[j+l][1]*bdt_size <= buffer_size){
                /*****************************************************************/
                dt_dest = &to_buffer[xtemp->root_add->comp[j+l][2]+
				     (i*xtemp->extent)+
				     (k*(xtemp->root_add->comp[j+xtemp->root_add->comp[j][4]-1][3]))];
                /* 
		** dt_dest = &to_buffer[xtemp->root_add->comp[j+l][2]+
		** (i*xtemp->extent)+
		** (k*(xtemp->root_add->comp[j+xtemp->root_add->comp[j][4]-1][2]
		** -xtemp->root_add->comp[j][2]+
		** xtemp->root_add->comp[j+xtemp->root_add->comp[j][4]-1][3]))]; 
		*/
                dt_src = &from_buffer[offset];
                dt_size = xtemp->root_add->comp[j+l][1]*bdt_size;

                if(code_info->bs == 0) 
		  dt_type = 1;
                else
		  dt_type = bdt_size;
	
		  /*
		   * 0 - datatype
		   * 1 - count
		   * 2 - displacement
		   * 3 - extent
		   * 4 - number of items
		   * 5 - number of repeats
		   * */

                FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
                offset += xtemp->root_add->comp[j+l][1]*(bdt_size);
                /**************************************************************/
              }
              else {
                /**************************************************************/
                dt_dest = &to_buffer[xtemp->root_add->comp[j+l][2]+
				     (i*xtemp->extent)+
				     (k*xtemp->root_add->comp[j+xtemp->root_add->comp[j][4]-1][3])];
                dt_src = &from_buffer[offset];

                dt_size = buffer_size - offset;

                if(code_info->bs == 0) 
		  dt_type = 1;
                else 
		  dt_type = bdt_size;

                FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)

                printf("R - Buffer is full ABC!!!\n");
                return(-2);
                /***************************************************************/
              }
            }
          }
        }




#endif
        last_dt[0] = offset;
      } 
    }
    return(last_dt[0]);
  }
  printf("RD DATATYPE %d does NOT exist\n",dt); 
  return(-1);
}
/***************************************************************************
****************************************************************************
** Same as write_dt_to_buffer but puts all basic_datatypes of the same kind together
**
**  NOT ABLE TO CONTINUE IF NOT ENOUGH BUFFER SUPPLIED TO COPY EVERYTHING
**  BUT VERY EFFICIENT.
**
**
**
** Definitely done.
****************************************************************************
***************************************************************************/
int ftmpi_ddt_write_dt_to_buffer_block(char *buffer,int buffer_size,
				       int dt, int count,char *data,
				       int spaces, int *dt_write,int bswap_var)
{
  int i,j,k,l,ret,retvar=0;
  int dt_write_new[FTMPI_DDT_B_DT_MAX],last_dt=-1,extent;
  FTMPI_DDT_R_ITEM * xtemp = NULL;
  FTMPI_DDT_E_ITEM * temp = NULL;
  FTMPI_DDT_mem_CP_BS_vars

  if(FTMPI_DDT_INIT == 0){
    ftmpi_ddt_init();
  }

  if(dt < FTMPI_DDT_B_DATATYPE_TOTAL){
    if(FTMPI_DDT_BASIC_DATATYPE[dt]->size * count <= buffer_size){
      dt_size = ret = FTMPI_DDT_BASIC_DATATYPE[dt]->size * count;
    }
    else {
      dt_size = (buffer_size/FTMPI_DDT_BASIC_DATATYPE[dt]->size)*
	FTMPI_DDT_BASIC_DATATYPE[dt]->size;
      ret = -2;
    }

    dt_dest = buffer;
    dt_src = data;

    if(bswap_var == 0){
      dt_type = 1;
    }
    else {
      dt_type = FTMPI_DDT_BASIC_DATATYPE[dt]->size;
    }

    FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)

    return(ret);
  }

  xtemp = ftmpi_ddt_get_root(dt);


  if(xtemp != NULL){
    extent = xtemp->extent;
    if(dt_write == NULL){
      dt_write = dt_write_new;
      for(i=1;i<FTMPI_DDT_B_DATATYPE_TOTAL;i++){
        if(xtemp->root_add->type_count[i] != 0){

          if(last_dt == -1){
            dt_write[i] = 0;
            last_dt = i;
          }
          else {
            if(spaces) {
              dt_write[i] = ((dt_write[last_dt]+
			      (FTMPI_DDT_BASIC_DATATYPE[last_dt]->size * 
			       xtemp->root_add->type_count[last_dt]*count)-1+sizeof
			      (int))/sizeof(int))*(sizeof(int));
            }
            else {
              dt_write[i] = dt_write[last_dt]+
		(FTMPI_DDT_BASIC_DATATYPE[last_dt]->size * 
		 xtemp->root_add->type_count[last_dt]*count);
            }
            last_dt = i;
          }
        }
      } 
    }
    for(i=0;i<count;i++){
      if(xtemp->root_add->t_comp == -1){

        temp = ftmpi_ddt_get_element(xtemp,temp);

        while(temp != NULL){
          if(temp->dt < FTMPI_DDT_B_DATATYPE_TOTAL){
            if((dt_write[temp->dt] + (temp->size * temp->count)) <= buffer_size){
	      /*****************************************************************/
              dt_dest = &buffer[dt_write[temp->dt]];
              dt_src = &data[temp->padding+(extent*i)];
              dt_size = temp->size * temp->count;
              if(bswap_var == 0){
                dt_type = 1;
              }
              else {
                dt_type = FTMPI_DDT_BASIC_DATATYPE[temp->dt]->size;
              }
 
              FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
              dt_write[temp->dt] +=(temp->size * temp->count);
              if(dt_write[temp->dt] > retvar){
                retvar = dt_write[temp->dt];
              }
            }
            else {
	      /****************************************************************/
              dt_dest = &buffer[dt_write[temp->dt]];
              dt_src = &data[temp->padding+(extent*i)];
              dt_size = (temp->size*temp->count)-((dt_write[temp->dt] + 
						   temp->size*temp->count)-buffer_size);
              if(bswap_var == 0){
                dt_type = 1;
              }
              else {
                dt_type = FTMPI_DDT_BASIC_DATATYPE[temp->dt]->size;
              }
 
              FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
              printf("Buffer is full!!!\n");
              return(-2);
            }
          }
          else {
	    /***************************************************************/
            ret = ftmpi_ddt_write_dt_to_buffer_block(buffer,buffer_size,
						     temp->dt,temp->count,
						     &data[temp->padding],
						     spaces,dt_write,bswap_var);
            if(ret < 0){
              return (ret);
            }
            if(ret > retvar){
              retvar = ret;
            }
          }
          temp = ftmpi_ddt_get_element(xtemp,temp);
        }
      } 
      else {
        /*
          0 - datatype
          1 - count
          2 - displacement
          3 - extent
          4 - number of items
          5 - number of repeats
        */
        for(j=0;j<xtemp->root_add->t_comp;j += xtemp->root_add->comp[j][4]){
          for(k=0;k<xtemp->root_add->comp[j][5];k++){
            for(l=0;l<xtemp->root_add->comp[j][4];l++){
              if((dt_write[xtemp->root_add->comp[j+l][0]] + 
		  (FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size * 
		   xtemp->root_add->comp[j+l][1])) <= buffer_size){
#ifdef DEBUG_DDT
		printf("%d FROM %d TO %d %d\n",xtemp->root_add->comp[j+l][0],
		       xtemp->root_add->comp[j+l][2]+
		       (xtemp->root_add->comp[j+xtemp->root_add->comp[j][4]-1][3]*k)+
		       (i*extent),dt_write[xtemp->root_add->comp[j+l][0]],
		       FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size * 
		       xtemp->root_add->comp[j+l][1]);
#endif
		/*******************************************************************/
                dt_dest = &buffer[dt_write[xtemp->root_add->comp[j+l][0]]];
                dt_src = &data[xtemp->root_add->comp[j+l][2]+
			       (xtemp->root_add->comp[j+xtemp->root_add->comp[j][4]-1][3]*k)
			       +(i*extent)];
                dt_size = FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size * 
		  xtemp->root_add->comp[j+l][1];
                if(bswap_var == 0){
                  dt_type = 1;
                }
                else {
                  dt_type = FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size;
                }
 
                FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
		  /*****************************************************************/
		  dt_write[xtemp->root_add->comp[j+l][0]] += 
		  (FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size * 
		   xtemp->root_add->comp[j+l][1]);
                if(dt_write[xtemp->root_add->comp[j+l][0]] > retvar){
                  retvar = dt_write[xtemp->root_add->comp[j+l][0]];
                }
              }
              else {
		/********************************************************************/
                dt_dest = &buffer[dt_write[xtemp->root_add->comp[j+l][0]]];
                dt_src = &data[xtemp->root_add->comp[j+l][2]+
			       (xtemp->root_add->comp[j+xtemp->root_add->comp[j][4]-1][3]*k)
			       +(i*extent)];
                dt_size = (FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size*
			   xtemp->root_add->comp[j+l][1])-
		  ((dt_write[xtemp->root_add->comp[j+l][0]] + 
		    FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size*
		    xtemp->root_add->comp[j+l][1])-buffer_size);
                if(bswap_var == 0){
                  dt_type = 1;
                }
                else {
                  dt_type = FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size;
                }
 
                FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
                printf("W - Buffer is full!!!\n");
                return(-2);
		/*****************************************************************/
              }
            }
          }
        }
      } 
    }
    return(retvar);
  }
  printf("DATATYPE %d does NOT exist\n",dt);
  return(-1);
}
/***************************************************************************
****************************************************************************
** Same as ftmpi_ddt_read_dt_from_buffer() but puts all basic_datatypes 
** of the same kind together
**
**  NOT ABLE TO CONTINUE IF NOT ENOUGH BUFFER SUPPLIED TO COPY EVERYTHING
**  BUT VERY EFFICIENT.
**
**
**
** Definitely done.
****************************************************************************
***************************************************************************/
int ftmpi_ddt_read_dt_from_buffer_block(char *buffer, int buffer_size, int dt,
					int count, char *data, int spaces, 
					int *dt_read, int bswap_var)
{
  int i,j,k,l,ret,retvar=0;
  int dt_read_new[FTMPI_DDT_B_DT_MAX],last_dt=-1,extent;
  FTMPI_DDT_R_ITEM * xtemp = NULL;
  FTMPI_DDT_E_ITEM * temp = NULL;
  FTMPI_DDT_mem_CP_BS_vars

  if(FTMPI_DDT_INIT == 0){
    ftmpi_ddt_init();
  }




  if(dt < FTMPI_DDT_B_DATATYPE_TOTAL){
    if(FTMPI_DDT_BASIC_DATATYPE[dt]->size * count <= buffer_size){
      dt_size = ret = FTMPI_DDT_BASIC_DATATYPE[dt]->size * count;
    }
    else {
      dt_size = (buffer_size/FTMPI_DDT_BASIC_DATATYPE[dt]->size)*
	FTMPI_DDT_BASIC_DATATYPE[dt]->size;
      ret = -2;
    }

    dt_dest = data;
    dt_src = buffer;

    if(bswap_var == 0){
      dt_type = 1;
    }
    else {
      dt_type = FTMPI_DDT_BASIC_DATATYPE[dt]->size;
    }

    FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)

    return(ret);
  }


  xtemp = ftmpi_ddt_get_root(dt);


  if(xtemp != NULL){
    extent = xtemp->extent;
    if(dt_read == NULL){
      dt_read = dt_read_new;
      for(i = 1;i <FTMPI_DDT_B_DATATYPE_TOTAL;i++){
        if(xtemp->root_add->type_count[i] != 0){
          if(last_dt == -1){
            dt_read[i] = 0;
            last_dt = i;
          }
          else {
            if(spaces){
              dt_read[i] = ((dt_read[last_dt]+
			     (FTMPI_DDT_BASIC_DATATYPE[last_dt]->size * 
			      xtemp->root_add->type_count[last_dt]*count)-
			     1+sizeof(int))/sizeof(int))*(sizeof(int));
            }   
            else {
              dt_read[i] = dt_read[last_dt]+
		           (FTMPI_DDT_BASIC_DATATYPE[last_dt]->size * 
			    xtemp->root_add->type_count[last_dt]*count);
            }   
            last_dt = i;
          }     
        }       
      }   
    }   
    for(i=0;i<count;i++){
      if(xtemp->root_add->t_comp == -1){ /* if DT composition does not exist */
        temp = ftmpi_ddt_get_element(xtemp,temp);
        while(temp != NULL){
          if(temp->dt < FTMPI_DDT_B_DATATYPE_TOTAL){
            if(dt_read[temp->dt] + (temp->size * temp->count) <= buffer_size){
	      /**************************************************************/
              dt_dest = &data[temp->padding+(extent*i)];
              dt_src = &buffer[dt_read[temp->dt]];
              dt_size = temp->size * temp->count;
              if(bswap_var == 0){
                dt_type = 1;
              } 
              else {
                dt_type = FTMPI_DDT_BASIC_DATATYPE[temp->dt]->size;
              } 
                
              FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
              dt_read[temp->dt] = dt_read[temp->dt] + (temp->size * temp->count);
              if(dt_read[temp->dt] > retvar){
                retvar = dt_read[temp->dt];
              } 
            }   
            else {
	      /**************************************************************/
              dt_dest = &data[temp->padding+(extent*i)];
              dt_src = &buffer[dt_read[temp->dt]];
              dt_size = ((int)buffer + buffer_size)-(int)dt_read[temp->dt];
              if(bswap_var == 0){
                dt_type = 1;
              } 
              else {
                dt_type = FTMPI_DDT_BASIC_DATATYPE[temp->dt]->size;
              } 
                
              FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
              printf("Buffer is full!!!\n");
              return(-2);
            }   
          }     
          else {
	    /*****************************************************************/
            ret = ftmpi_ddt_read_dt_from_buffer_block(buffer,buffer_size,
						      temp->dt,temp->count,
						      &data[temp->padding],
						      spaces,dt_read,bswap_var);  
            if(ret < 0){
              return (ret);
            }   
            if(ret > retvar){
              retvar = ret;
            }   
          }     
          temp = ftmpi_ddt_get_element(xtemp,temp);
        }       
      }         
      else {
        /* 
          0 - datatype
          1 - count
          2 - displacement
          3 - extent
          4 - number of items
          5 - number of repeats
        */      
        for(j=0;j<xtemp->root_add->t_comp;j += xtemp->root_add->comp[j][4]){
          for(k=0;k<xtemp->root_add->comp[j][5];k++){
            for(l=0;l<xtemp->root_add->comp[j][4];l++){
              if(dt_read[xtemp->root_add->comp[j+l][0]] + 
		 (FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size * 
		  xtemp->root_add->comp[j+l][1]) <= buffer_size){         
#ifdef DEBUG_DDT
                printf("%d FROM %d TO %d %d\n",xtemp->root_add->comp[j+l][0],
		       dt_read[xtemp->root_add->comp[j+l][0]],
		       xtemp->root_add->comp[j+l][2]+
		       (xtemp->root_add->comp[j+xtemp->root_add->comp[j][4]-1][3]*k)+
		       (i*extent),
		       FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size * 
		       xtemp->root_add->comp[j+l][1]);
#endif
		/********************************************************************/
                dt_dest = &buffer[xtemp->root_add->comp[j+l][2]+
				  (xtemp->root_add->comp[j+xtemp->root_add->comp[j][4]-1][3]*
				   k)+(i*extent)];
                dt_src = &data[dt_read[xtemp->root_add->comp[j+l][0]]];
                dt_size = FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size * 
		          xtemp->root_add->comp[j+l][1];
                if(bswap_var == 0){
                  dt_type = 1;
                }
                else {
                  dt_type = FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size;
                }
                
                FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
                dt_read[xtemp->root_add->comp[j+l][0]] +=
		  (FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size * 
		   xtemp->root_add->comp[j+l][1]);
                if(dt_read[xtemp->root_add->comp[j+l][0]] > retvar){
                  retvar = dt_read[xtemp->root_add->comp[j+l][0]];
                }
              } 
              else {
		/****************************************************************/
                dt_dest = &buffer[xtemp->root_add->comp[j+l][2]+
				  (xtemp->root_add->comp[j+xtemp->root_add->comp[j][4]-1][3]*
				   k)+(i*extent)];
                dt_src = &data[xtemp->root_add->comp[j+l][2]+
			       ((extent/xtemp->root_add->t_comp)*k)];
                dt_size = ((int)buffer + buffer_size)-
		           (int)dt_read[xtemp->root_add->comp[j+l][0]];
                if(bswap_var == 0){
                  dt_type = 1;
                }
                else {
                  dt_type = FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[j+l][0]]->size;
                }
                
                FTMPI_DDT_mem_COPY_BSWAP(dt_dest,dt_src,dt_size,dt_type,dt_mode)
                printf("R - Buffer is full!!!\n");
                return(-2);
              } 
            }   
          }   
        }   
      }   
    }   
    return(retvar);
  }             
  printf("DATATYPE %d does NOT exist\n",dt);
  return(-1);   
}               
/***************************************************************************/
/***************************************************************************
****************************************************************************
**
**
**
**
**
**
**
** Definitely done
****************************************************************************
***************************************************************************/
int ftmpi_ddt_xdr_encode_dt_to_buffer(char * from,char * to,char * tmp,
				      int dest_size,int cnt,int dt,
				      FTMPI_DDT_CODE_INFO * code_info)
{
  int offset=0,i;
  FTMPI_DDT_R_ITEM * xtemp=NULL;

  if(FTMPI_DDT_INIT == 0){
    ftmpi_ddt_init();
  }


  if(dt < FTMPI_DDT_B_DATATYPE_TOTAL){
#ifdef DEBUG_DDT
    printf("ENCODE BASIC DDT %d\n",code_info->code_type); 
#endif

    if(code_info->code_type == FTMPI_XDR){
      ftmpi_ddt_xdr_code(to,from,dt,cnt,1);
      return(1);
    }
    else if(code_info->code_type == FTMPI_RSC){
      ftmpi_ddt_write_dt_to_buffer(to,dest_size,dt,cnt,from,0,NULL,0);
      return(1);
    }
    else {
      memcpy(to,from,dest_size);
      return(0);
    }
  }

  xtemp = ftmpi_ddt_get_root(dt);

  if(xtemp != NULL){
#ifdef DEBUG_DDT
    printf("ENCODE DDT\n"); 
#endif
    if(code_info->code_type == FTMPI_XDR){
      ftmpi_ddt_write_dt_to_buffer_block(tmp,dest_size,dt,cnt,from,1,NULL,0);
      for(i=1;i<FTMPI_DDT_B_DATATYPE_TOTAL;i++){
        if(xtemp->root_add->type_count[i] >0){
          ftmpi_ddt_xdr_code(&to[offset],&tmp[offset],i,
			     xtemp->root_add->type_count[i]*cnt,1);
          offset += (((xtemp->root_add->type_count[i]*
		       FTMPI_DDT_BASIC_DATATYPE[i]->size*cnt)-1+
		      sizeof(int))/sizeof(int))*(sizeof(int));
        }
      }
      return(0);   /* it should now be in src */
    }
    else if(code_info->code_type == FTMPI_RSC){
      ftmpi_ddt_write_dt_to_buffer(to,dest_size,dt,cnt,from,0,NULL,0);
      return(1);
    }
    else {
      ftmpi_ddt_write_dt_to_buffer(to,dest_size,dt,cnt,from,0,NULL,0);
      return(1);
    }
  } 
  printf("DATATYPE %d does NOT exist\n",dt);
  return(-1);
}
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
int ftmpi_ddt_xdr_decode_dt_to_buffer(char *from, char *to, char *tmp, 
				      int dest_size, int cnt, int dt,
				      FTMPI_DDT_CODE_INFO * code_info)
{
  int offset=0,i;
  FTMPI_DDT_R_ITEM *xtemp = NULL;
  int size;

  if(FTMPI_DDT_INIT == 0){
    ftmpi_ddt_init();
  }

  ftmpi_mpi_type_size(dt,&size);


  if(dt < FTMPI_DDT_B_DATATYPE_TOTAL){
#ifdef DEBUG_DDT
    printf("DECODE BASIC DDT\n"); 
#endif
    if(code_info->code_type == FTMPI_XDR){
      ftmpi_ddt_xdr_code(to,from,dt,cnt,2);
      return(1);
    }
    else if(code_info->code_type == FTMPI_RSC){
      ftmpi_ddt_read_dt_from_buffer(from,dest_size,dt,cnt,to,0,NULL,code_info);
      return(1);
    }
    memcpy(to,from,dest_size);
    return(0);
  }

  xtemp = ftmpi_ddt_get_root(dt);

  if(xtemp != NULL){
#ifdef DEBUG_DDT
     printf("DECODE DDT\n"); 
#endif
     if(code_info->code_type == FTMPI_XDR){
      for(i=1;i<FTMPI_DDT_B_DATATYPE_TOTAL;i++){
        if(xtemp->root_add->type_count[i] >0){
          ftmpi_ddt_xdr_code(&tmp[offset],&from[offset],i,
			     xtemp->root_add->type_count[i]*cnt,2);
          offset += (((xtemp->root_add->type_count[i]*
		       FTMPI_DDT_BASIC_DATATYPE[i]->size*cnt)-1+
		      sizeof(int))/sizeof(int))*(sizeof(int));
        }
      }  
      ftmpi_ddt_read_dt_from_buffer_block(to,dest_size,dt,cnt,tmp,1,NULL,0);
      return(0);
    }
    else{
      ftmpi_ddt_read_dt_from_buffer(from,dest_size,dt,cnt,to,0,NULL,code_info);
      return(1);
    }
  }  
  printf("DATATYPE %d does NOT exist\n",dt);
  return(-1);
}

/***************************************************************************/
/***************************************************************************
****************************************************************************
**  performs XDR encode or decode depending on XDR setup. At the moment will
**  not XDR CHARs as I have no idea why
**
**
**
**
**
**  Definitely done
****************************************************************************
***************************************************************************/
int ftmpi_ddt_xdr_vector_code(XDR * xdr,char * buffer,int count,int size,int datatype)
{
  if(datatype == MPI_SHORT || datatype == MPI_UNSIGNED_SHORT ){
    if(!xdr_vector(xdr,buffer,(u_int)count,(u_int)size,(xdrproc_t)xdr_short)){
      printf("%d %d - XDR ENCODE ERROR - MPI_SHORT\n",datatype,
	     FTMPI_DDT_BASIC_DATATYPE[datatype]->size);
      return(-1);
    }
  }  
  else if(datatype == MPI_INT ){
    if(!xdr_vector(xdr,buffer,(u_int)count,(u_int)size,(xdrproc_t)xdr_int)){
      printf("%d - XDR ENCODE ERROR - MPI_INT\n",datatype);
      return(-1);
    }
  }  
  else if(datatype == MPI_LONG || datatype == MPI_UNSIGNED_LONG){
    if(!xdr_vector(xdr,buffer,(u_int)count,(u_int)size,(xdrproc_t)xdr_long)){
      printf("%d - XDR ENCODE ERROR - MPI_LONG\n",datatype);
      return(-1);
    }
  }  
  else if(datatype == MPI_UNSIGNED ){
    if(!xdr_vector(xdr,buffer,(u_int)count,(u_int)size,(xdrproc_t)xdr_int)){
      printf("%d - XDR ENCODE ERROR -- MPI_UNSIGNED\n",datatype);
      return(-1);
    }
  } 
  else if(datatype == MPI_FLOAT ){
    if(!xdr_vector(xdr,buffer,(u_int)count,(u_int)size,(xdrproc_t)xdr_float)){
      printf("%d - XDR ENCODE ERROR - MPI_FLOAT\n",datatype);
      return(-1);
    }
  } 
  else {
    if(!xdr_vector(xdr,buffer,(u_int)count,(u_int)size,(xdrproc_t)xdr_double)){
      printf("%d %d - XDR ENCODE ERROR - DO NOT KNOW HOW TO ENOCODE THIS\n",
	     datatype,FTMPI_DDT_BASIC_DATATYPE[datatype]->size);
      return(-1);
    }
  } 
  return(1);
}   
/***************************************************************************
****************************************************************************
**  This will decide on its own whether to do xdr or byteswap
**
**  code - 1 for encode and 2 for decode
**
**
**
**
**  Definitely done
****************************************************************************
***************************************************************************/
int ftmpi_ddt_xdr_code(char *out, char *in, int dt, int count, int code)
{
  XDR xdr;
  int ret;

  /* code 1 = ENCODE;  code 2 = DECODE */

#ifdef DEBUG_DDT
      printf("Performing XDR %d %d %d\n",code,dt,count);
#endif

  if(code == 1){
    if(dt == MPI_CHAR || dt == MPI_UNSIGNED_CHAR || dt == MPI_BYTE){
      memcpy(out,in,FTMPI_DDT_BASIC_DATATYPE[dt]->size*count);
      return 0;
    }
    else {
      xdrmem_create(&xdr,out,(u_int)(FTMPI_DDT_BASIC_DATATYPE[dt]->size 
				     * count),XDR_ENCODE);
      ret = ftmpi_ddt_xdr_vector_code(&xdr,in,count,
				      FTMPI_DDT_BASIC_DATATYPE[dt]->size,dt);
    }
  }
  else{
    if(dt == MPI_CHAR || dt == MPI_UNSIGNED_CHAR || dt == MPI_BYTE){
      memcpy(out,in,FTMPI_DDT_BASIC_DATATYPE[dt]->size*count);
      return 0;
    }
    else {
      xdrmem_create(&xdr,in,(u_int)(FTMPI_DDT_BASIC_DATATYPE[dt]->size 
				    * count),XDR_DECODE);
      ret = ftmpi_ddt_xdr_vector_code(&xdr,out,count,
				      FTMPI_DDT_BASIC_DATATYPE[dt]->size,dt);
    }
  }
  return(ret);
}
/***************************************************************************
****************************************************************************
**    
**    
**
**
**
**
**
**   Definitely done   - MAYBE as far as I am aware
****************************************************************************
***************************************************************************/
int ftmpi_ddt_check_array(int * old,int * size)
{     
  int j;
  int ni=0,i,k;
  int t_cnt = 0;
  int new_a[(FTMPI_DDT_COMP_MAX+4)*6];
  int last_ext = 0;
  int l_ext = 0;
  int p_ext = 0;
/*   int last_end; */
/*   int cur_cnt; */
#ifdef DEBUG_DDT
  int prt_ln = 0;
  int tmp;
  int dt_cnt = 0;
#endif
      
  if(*size < 1){
    return(0);
  }

     
      
#ifdef DEBUG_DDT
  printf("--- ORG --\n");
  printf("--- ORG --\n");
  printf("--- ORG --\n");
  tmp = old[4]-1;
  printf("----------------------------------- %d\n",*size);
  for(i=0;i<*size;i++){
    printf("%-3d ",i);
    for(j=0;j<6;j++){
      printf("%4d",old[i*6+j]);
    } 
    printf("\n");
    if(i==tmp){
       if(prt_ln == 1){
         printf("  -----------------------\n");
       }
       tmp += old[i*6+j+4];
       if(old[i*6+j+4] >1){
         prt_ln = 1;
       }
       else {
         prt_ln = 0;
       }
    } 
  }   
#endif
     
     
  /*
    0 - datatype
    1 - count
    2 - displacement
    3 - extent
    4 - number of items
    5 - number of repeats
  */  
      
  for(j=0;j<(*size*6);j += (old[j+4])*6){
    if(old[j+5]>1){
#ifdef DEBUG_DDT
      printf("REPEAT YES ");
#endif
      if(old[j+4] == 1){
#ifdef DEBUG_DDT
        printf("- ONLY ONE");
#endif
        if(t_cnt+1 == FTMPI_DDT_COMP_MAX){
          return(0);
        }
        memcpy(&new_a[ni],&old[j],6*sizeof(int));
        t_cnt++;
        ni+=6;
      }
      else {
#ifdef DEBUG_DDT
        printf("- AT LEAST TWO %d",t_cnt);
#endif
        if(old[j] == old[j+(old[j+4]-1)*6]){
#ifdef DEBUG_DDT
          printf(" - SAME");
#endif
          if(0 == old[j+2]-last_ext){
#ifdef DEBUG_DDT
            printf(" - START OK");
#endif
            if(old[j+(old[j+4]-1)*6+3] == old[j+(old[j+4]-1)*6+2]+
	       FTMPI_DDT_BASIC_DATATYPE[old[j+(old[j+4]-1)*6]]->size * 
	       old[j+(old[j+4]-1)*6+1]){
#ifdef DEBUG_DDT
              printf(" - END OK");
#endif
      
              /* make this into a three part DT */
      
              /*
                0 - datatype            - stays the same
                1 - count               - stays the same
                2 - displacement        - disp - size
                3 - extent              - extent - size;
                4 - number of items     - number - 1;
                5 - number of repeats   - number -1;
              */
      
              l_ext = (old[j+((old[j+4]-1)*6)+3]-p_ext)*(old[j+5]-1);
#ifdef DEBUG_DDT
              printf(" %3d %3d ",l_ext,p_ext);
#endif
      
              if(t_cnt+1 == FTMPI_DDT_COMP_MAX){
                return(0);
              }
              memcpy(&new_a[ni],&old[j],6*sizeof(int));
      
              new_a[ni+4] = 1;
              new_a[ni+5] = 1;
              t_cnt++;
              ni+=6;
      
              i = 0;
      
              if(old[j+4]-2 > 0){
#ifdef DEBUG_DDT
                printf(" - THERE IS MIDDLE");
#endif
                for(i=0;i<old[j+4]-2;i++){
                  if(t_cnt == FTMPI_DDT_COMP_MAX){
                    return(0);
                  }
                  memcpy(&new_a[ni],&old[j+(i+1)*6],6*sizeof(int));
      
                  if(new_a[ni+5] > 2){
                    new_a[ni+4] -= 1;
                  }
                  else {
                    new_a[ni+4] = 1;
                  }
                  new_a[ni+5] -= 1;
      
                  t_cnt++;
                  ni+=6;
                }
              }
      
              if(t_cnt == FTMPI_DDT_COMP_MAX){
                return(0);
              }
              memcpy(&new_a[ni],&old[j+(i+1)*6],6*sizeof(int));
      
              new_a[ni+1] += old[j+1];
              new_a[ni+3] = new_a[ni+3]+
		           (old[j+1]*FTMPI_DDT_BASIC_DATATYPE[old[j]]->size);
      
              if(new_a[ni+5]>2){
                new_a[ni+4] -= 1;
              }
              else {
                new_a[ni+4] = 1;
              }
      
              new_a[ni+5] -= 1;
      
              t_cnt++;
              ni+=6;
      
              /*
                0 - datatype            - stays the same
                1 - count               - stays the same
                2 - displacement        - disp - size
                3 - extent              - extent - size;
                4 - number of items     - number - 1;
                5 - number of repeats   - number -1;
              */
/*              printf("JSUT OUT OF CURRIOUSITY --- %d\n",t_cnt); */
      
              if(old[j+4]-2 > 0){
                for(i=0;i<old[j+4]-2;i++){
                  if(t_cnt == FTMPI_DDT_COMP_MAX){
                    return(0);
                  }
                  memcpy(&new_a[ni],&old[j+(i+1)*6],6*sizeof(int));
      
                  new_a[ni+2] += l_ext;
                  new_a[ni+3] += l_ext;
                  new_a[ni+4] = 1;
                  new_a[ni+5] = 1;
      
                  t_cnt++;
                  ni+=6;
                }
              }
      
              if(t_cnt == FTMPI_DDT_COMP_MAX){
                return(0);
              }
              memcpy(&new_a[ni],&old[j+(old[j+4]-1)*6],6*sizeof(int));
      
              new_a[ni+2] += l_ext;
              new_a[ni+3] += l_ext;
              new_a[ni+4] = 1;
              new_a[ni+5] = 1;
      
      
              p_ext = old[j+(old[j+4]-1)*6+3]*(old[j+5]);
      
              t_cnt++;
              ni+=6;
            }
            else {
#ifdef DEBUG_DDT
              printf(" - END NOK");
#endif
              for(i=0;i<old[j+4];i++){
                if(t_cnt == FTMPI_DDT_COMP_MAX){
                  return(0);
                }
                memcpy(&new_a[ni],&old[j+(i*6)],6*sizeof(int)*2);
      
                t_cnt++;
                ni+=6;
              }
            }
          }
          else {
#ifdef DEBUG_DDT
            printf(" - START NOK %d",t_cnt);
#endif
            for(i=0;i<old[j+4];i++){
              if(t_cnt == FTMPI_DDT_COMP_MAX){
                return(0);
              }
              memcpy(&new_a[ni],&old[j+(i*6)],6*sizeof(int)*2);
      
              t_cnt++;
              ni+=6;
            }
      
          }
        }
        else {
#ifdef DEBUG_DDT
          printf(" - DIFF %d %d",t_cnt,old[j+4]);
#endif
          for(i=0;i<old[j+4];i++){
            if(t_cnt == FTMPI_DDT_COMP_MAX){
              return(0);
            }
            memcpy(&new_a[ni],&old[j+(i*6)],6*sizeof(int)*2);
      
            t_cnt++;
            ni+=6;
          }
        }
      }
    } 
    else {
#ifdef DEBUG_DDT
      printf("REPEAT NO ");
      if(j <0){
        exit(1);
      }
#endif
      for(i=0;i<old[j+4];i++){
        if(t_cnt == FTMPI_DDT_COMP_MAX){
          return(0);
        }
        memcpy(&new_a[ni],&old[j+(i*6)],6*sizeof(int));
        t_cnt++;
        ni+=6;
      }
    } 
      
         /*
           0 - datatype            - stays the same
           1 - count               - stays the same
           2 - displacement        - disp - size
           3 - extent              - extent - size;
           4 - number of items     - number - 1;
           5 - number of repeats   - number -1;
         */
      
#ifdef DEBUG_DDT
    printf("\n");
#endif
    last_ext += old[j+(old[j+4]-1)*6+3]*old[j+5];
  }   
      
  


#ifdef DEBUG_DDT
  printf("--- NEW -- %d\n",t_cnt);
  printf("--- NEW --\n");
  printf("--- NEW --\n");
  tmp = old[4]-1;
  for(i=0;i<t_cnt;i++){
    printf("%-3d ",i);
    for(j=0;j<6;j++){
      printf("%4d",old[i*6+j]);
    } 
    printf("\n");
    if(i==tmp){
       if(prt_ln == 1){
         printf("  -----------------------\n");
       }
       tmp += old[i*6+j+4];
       if(old[i*6+j+4] >1){
         prt_ln = 1;
       }
       else {
         prt_ln = 0;
       }
    } 
  }   
#endif
     
     
  k = 0;
  memcpy(old,new_a,6*sizeof(int));
/*   last_end = 0; */
/*   cur_cnt = old[4]; */
      
  /*  
    0 - datatype            - stays the same
    1 - count               - stays the same
    2 - displacement        - disp - size
    3 - extent              - extent - size;
    4 - number of items     - number - 1;
    5 - number of repeats   - number -1;
  */  
      
  for(i=6;i<t_cnt*6;i+=6){
    if(new_a[i+5] == 1){
#ifdef DEBUG_DDT
      printf("POSSIBLE COMBINATION\n");
#endif
      if(new_a[i] == old[k]){
#ifdef DEBUG_DDT
        printf("SAME DT ||  OLD %d NEW %d\n",k,i);
#endif
        if(old[k+2]+FTMPI_DDT_BASIC_DATATYPE[old[k]]->size*old[k+1] == new_a[i+2]){
#ifdef DEBUG_DDT
          printf("CONTIGUOUS\n");
#endif
          old[k+1] += new_a[i+1];
	  /*          old[k+3] = new_a[i+3]; */
          old[k+3] += new_a[i+3];
        }
        else {
#ifdef DEBUG_DDT
          printf("NOT CONTIGUOUS\n");
#endif
          k+=6;
          memcpy(&old[k],&new_a[i],6*sizeof(int));
        }
      }
      else {
        k+=6;
        memcpy(&old[k],&new_a[i],6*sizeof(int));
      }
    } 
    else {
#ifdef DEBUG_DDT
      printf("REPEATED\n");
#endif
      /* SHOULD JUST MEMCPY STUFF INTO AN APPROPRIATE LOCATION */
      k+=6;
      memcpy(&old[k],&new_a[i],6*4);
    } 
  }   
  k+=6;
      
      


#ifdef DEBUG_DDT
  printf("\n\n*************************************************************\n\n");
  for(i=0;i<k/6;i++){
    printf("  [%d]",i);
    for(j=0;j<6;j++){
      printf("%4d",old[i*6+j]);
    } 
    printf("\n");
  }   
  printf("\n*************************************************************\n\n");
#endif
  *size = k/6;
  return(1);
}     

/***************************************************************************
****************************************************************************
**  NEW ADDITION ... to find out the exact size to be sent and received 
** 
**
**
**
**
**
**   Definitely done
****************************************************************************
***************************************************************************/
int ftmpi_ddt_write_read_size_det(int dt,int count,int * size,int code_type)
{
  int offset=0,i;
  FTMPI_DDT_R_ITEM * xtemp=NULL;



  if(FTMPI_DDT_INIT == 0){
    ftmpi_ddt_init();
  }
  if(dt < FTMPI_DDT_B_DATATYPE_TOTAL){
    *size = count * FTMPI_DDT_BASIC_DATATYPE[dt]->size;
    if(code_type != 0){
      return(1);
    }
    return(0);
  }
  else {
    xtemp = ftmpi_ddt_get_root(dt);
    if(xtemp != NULL){
      if(code_type == 2){
        for(i=1;i<FTMPI_DDT_B_DATATYPE_TOTAL;i++){
          if(xtemp->root_add->type_count[i] >0){
            offset += ((((xtemp->root_add->type_count[i]*
			  FTMPI_DDT_BASIC_DATATYPE[i]->size)*count)-1+
			sizeof(int))/sizeof(int))*(sizeof(int));
          }
        }  
        *size = offset;
        return(2);
      }
      else if(code_type == 1){
        /* if needed we could shave off little bit more time here --- ASK TONE */
        /* will need extra argument */
        *size = xtemp->size*count;
        return(1);
      }
      else {
        if(xtemp->contig == 1){
          *size = xtemp->size*count;
          return(0);
        }
        *size = xtemp->size*count;
        return(1);
      }
    } 
    printf(">wr_rd DATATYPE %d does NOT exist\n",dt);
    return(-1);
  }
}
/**********************************************************************/
/**********************************************************************/
/**********************************************************************/
int ftmpi_ddt_bs_xdr_det(int size)
{
  if(size%4 != 0){
    return(1);     /* doing XDR */
  }
  else{
    if(size  == 4){
      return(4);   /* This will allow 4 byte swapping */
    }
    else if(size == 8){
      return(8);   /* This will allow 8 byte swapping */
    }
    else if(size == 12){
      return(12);   /* This will allow 12 byte swapping */
    }
    else {
      return(1);   /* this will make us use the XDR method */
    }
  }
}

/**********************************************************************/
/**********************************************************************/
/**********************************************************************/
int ftmpi_ddt_get_remote_size(FTMPI_DDT_R_ITEM * xtemp,
			      FTMPI_DDT_CODE_INFO * code_info)
{
  if(ftmpi_ddt_my_longdouble.ld_flag == code_info->ld_flag) return xtemp->size;
  return(xtemp->size + (xtemp->root_add->type_count[MPI_LONG] * 
			code_info->size_l) + 
	 (xtemp->root_add->type_count[MPI_LONG_DOUBLE] * code_info->size_ld) 
	 - (xtemp->root_add->type_count[MPI_LONG] * 
	    FTMPI_DDT_BASIC_DATATYPE[MPI_LONG]->size) + 
	 (xtemp->root_add->type_count[MPI_LONG_DOUBLE] * 
	  FTMPI_DDT_BASIC_DATATYPE[MPI_LONG_DOUBLE]->size));
}
/**********************************************************************/
/**********************************************************************/
/**********************************************************************/
int ftmpi_ddt_decode_size_det(int dt,int count,int * size,
			      FTMPI_DDT_CODE_INFO * code_info)
{
  int offset=0,i;
  FTMPI_DDT_R_ITEM * xtemp=NULL;

  if(FTMPI_DDT_INIT == 0) ftmpi_ddt_init();

  if(dt == MPI_PACKED || dt == MPI_BYTE){
    *size = count;
    return(0);
  }
   
  if(code_info == NULL) {
    *size = -1;
    return(3);
  }


  if(dt < FTMPI_DDT_B_DATATYPE_TOTAL){
    if(code_info->code_type == FTMPI_XDR){
      *size = count * FTMPI_DDT_BASIC_DATATYPE[dt]->xdr_size;
      return 1;
    }
    else if(code_info->code_type == FTMPI_RSC){
      *size = count * ftmpi_ddt_get_bdt_sizes(code_info,dt);
      if(ftmpi_ddt_my_longdouble.ld_flag == code_info->ld_flag)
        return 0;
      return 1;
    }
    else if(code_info->code_type == FTMPI_SSC){
      printf("         !!!   WARNING   !!!\n\nSENDER-SIDE-CONVERSION "
	     "NOT IMPLEMENTED DEBDT\n\n         !!!   WARNING   !!!\n");
      return -1;
    }
    else {
      *size = count * FTMPI_DDT_BASIC_DATATYPE[dt]->size;
      if(ftmpi_ddt_my_longdouble.ld_flag == code_info->ld_flag)
        return 0;
      return 1;
    }
  }



  xtemp = ftmpi_ddt_get_root(dt);
  if(xtemp != NULL){
    if(code_info->code_type == FTMPI_XDR){
      for(i=1;i<FTMPI_DDT_B_DATATYPE_TOTAL;i++){
        if(xtemp->root_add->type_count[i] >0){
          offset += ((((xtemp->root_add->type_count[i]*
			FTMPI_DDT_BASIC_DATATYPE[i]->xdr_size)*count)-1+
		      sizeof(int))/sizeof(int))*(sizeof(int));
        }
      }
      *size = offset;
      return(2);
    }
    else if(code_info->code_type == FTMPI_RSC) {
      *size = ftmpi_ddt_get_remote_size(xtemp,code_info) * count;
      if(xtemp->contig == 1 && ftmpi_ddt_my_longdouble.ld_flag == code_info->ld_flag){
        return 0;
      }
      return 1;
    }
    else if(code_info->code_type == FTMPI_SSC) {
      printf("!!   WARNING \n SENDER-SIDE-CONVERSION NOT IMPLEMENTED !!\n");
      return -1;
    }
    else {
      *size = ftmpi_ddt_get_remote_size(xtemp,code_info) * count;
      if(xtemp->contig == 1 && ftmpi_ddt_my_longdouble.ld_flag == code_info->ld_flag){
        return 0;
      }
      return 1;
    }
  }
  printf(">wr_rd DATATYPE %d does NOT exist\n",dt);
  return(-1);
}


/***********************************************************************************/
/***********************************************************************************/
/***********************************************************************************/
int ftmpi_ddt_encode_size_det(int dt, int count, int *size,
			      FTMPI_DDT_CODE_INFO *code_info)
{
  int offset=0,i;
  FTMPI_DDT_R_ITEM * xtemp=NULL;

  if(FTMPI_DDT_INIT == 0){
    ftmpi_ddt_init();
  }
 
  if(dt == MPI_PACKED || dt == MPI_BYTE){
    *size = count;
    return(0);
  }

  if(code_info == NULL){
    *size = -1;
    return(3);
  }


  if(dt < FTMPI_DDT_B_DATATYPE_TOTAL){
/*     printf("ENCODE_SIZE %d\n",code_info->code_type); */
    if(code_info->code_type == FTMPI_XDR){
      *size = count * FTMPI_DDT_BASIC_DATATYPE[dt]->xdr_size;
      return(1);
    }
    else if(code_info->code_type == FTMPI_SSC){
      printf("WARNING! \n SENDER-SIDE-CONVERSION NOT IMPLEMENTED !\n");
      return(-1);
    }
    else if(code_info->code_type == FTMPI_RSC) {
      *size = count * FTMPI_DDT_BASIC_DATATYPE[dt]->size;
      return(0);
    }
    else{
      *size = count * FTMPI_DDT_BASIC_DATATYPE[dt]->size;
      return(0);
    }
  }
   
  xtemp = ftmpi_ddt_get_root(dt);
  if(xtemp != NULL){
    if(code_info->code_type == FTMPI_XDR){
      for(i=1;i<FTMPI_DDT_B_DATATYPE_TOTAL;i++){
        if(xtemp->root_add->type_count[i] >0){
          offset += ((((xtemp->root_add->type_count[i]*
			FTMPI_DDT_BASIC_DATATYPE[i]->xdr_size)*count)-1+
		      sizeof(int))/sizeof(int))*(sizeof(int));
        }
      }
      *size = offset;
      return(2);
    }

    else if(code_info->code_type == FTMPI_RSC){
      *size = xtemp->size*count;
      if(xtemp->contig != 1) 
        return 1;
      return 0;
    }
    else if(code_info->code_type == FTMPI_SSC){
      printf("WARNING!\n SENDER-SIDE-CONVERSION NOT IMPLEMENTED !\n");
      return(-1);
    }
    else {
      *size = xtemp->size*count;
      if(xtemp->contig != 1) 
        return 1;
      return 0;
    }
  }
  printf("ENCODE_SIZE_DET - DATATYPE %d does NOT exist\n",dt);
  return(-1);
}
/***********************************************************************************/
/***********************************************************************************/
/***********************************************************************************/



/***************************************************************************
****************************************************************************
**  NEW ADDITION ... to find the offset of the very first item ... hehehe
**  no thanx to the MPI_BOTTOM
**
**
**
**
**
**   Definitely done
****************************************************************************
***************************************************************************/
int ftmpi_ddt_get_first_offset(int dt)
{
  FTMPI_DDT_R_ITEM * xtemp=NULL;

  if(dt < FTMPI_DDT_B_DATATYPE_TOTAL){
    return(0);
  }

  /*******************************************************************************/
  /* it is not a basic datatype ... now we can do something about it ... hehehe  */
  /*******************************************************************************/



  xtemp = ftmpi_ddt_get_root(dt);

  if(xtemp != NULL){
    return(xtemp->root_add->first_e->padding);
  }
  else {  
    printf("DDT %d does not exist\n",dt);
    fflush(stdout);
    return(MPI_ERR_TYPE);
  }
}
/***************************************************************************
****************************************************************************
**
**
**
**
**
**
**
**
****************************************************************************
***************************************************************************/
int ftmpi_ddt_copy_ddt_to_ddt(void * fbuf,int fddt,int fcnt,void * tbuf,
			      int tddt,int tcnt)
{
  int fsize,tsize,ret;

  if(FTMPI_DDT_INIT == 0){
    ftmpi_ddt_init();
  }


  if ( ( fcnt == 0 ) || ( tcnt == 0 )) {
#ifdef DEBUG_DDT
    printf("copy_ddt_to_ddt: called with count zero\n");
#endif
    return ( 0 );
  }

  if((ftmpi_mpi_type_size(fddt,&fsize)) != MPI_SUCCESS) return(-3);
  fsize *= fcnt;

  if((ftmpi_mpi_type_size(tddt,&tsize)) != MPI_SUCCESS) return(-4);
  tsize *= tcnt;

  /* OK IF THE SIZE IS NOT SAME .... EXITING */
  if(fsize > tsize){
    printf("FROM ddt %d cnt %d = %d != TO ddt %d cnt %d = %d\n",
	   fddt,fcnt,fsize,tddt,tcnt,tsize);
    return ( -12);
  }
  else if ( fsize < tsize ) {
    if ( fddt == tddt )
      tcnt = fcnt;
    else {
      tsize = tsize/tcnt;
      tcnt = fsize/tsize;
    }
  }

  if(fsize > FTMPI_DDT_MPI_BUFFER_SIZE){
    _FREE(FTMPI_DDT_MPI_BUFFER);
    FTMPI_DDT_MPI_BUFFER = (char*)_MALLOC(((tsize/1024)+1)*1024);
    if(FTMPI_DDT_MPI_BUFFER == NULL){
      printf("ERROR: COULD NOT ALLOCATE MEM FOR FTMPI_DDT_MPI_BUFFER\n");
      exit(1);
    }
    FTMPI_DDT_MPI_BUFFER_SIZE = tsize;
  }

  ret = ftmpi_ddt_write_dt_to_buffer(FTMPI_DDT_MPI_BUFFER,fsize,fddt,fcnt,
				     fbuf, 0,NULL,0);  
  if(ret < 0){
    return(-13);
  }
 
  ret = ftmpi_ddt_read_dt_from_buffer(FTMPI_DDT_MPI_BUFFER,fsize,tddt,tcnt,
				      tbuf,0,NULL,&ftmpi_ddt_my_longdouble);
  if(ret < 0){
    return(-14);
  }
  return(0);
}
/***************************************************************************/
/***************************************************************************/
/***************************************************************************/
int ftmpi_ddt_get_element_count(int ddt, long rsize, int *cnt, 
				FTMPI_DDT_R_ITEM *ddt_ptr, int ddt_cnt)
{
  FTMPI_DDT_R_ITEM *xtemp=NULL; 
  FTMPI_DDT_E_ITEM *etemp=NULL;
  int i,j,k;
  int ret;
  int size;
  int e_cnt = 0;

 

  if(ddt_ptr == NULL){ 
    *cnt = 0;
    if(ftmpi_mpi_type_size(ddt,&size) != 0) return(-1);
    if(size == 0) return 0;
    if( (ddt <= FTMPI_DDT_B_DT_MAX) || ((ddt>500) &&
#ifdef HAVE_LONG_LONG
	(ddt <= 513))) {
#else
        (ddt <= 512))) {
#endif 
      *cnt = rsize/size;
      return(0);
    }

    xtemp = ftmpi_ddt_get_root(ddt);


    /*** COUNT THE ELEMENTS IN THE CURRENT DDT ***/
    for(j=1;j<FTMPI_DDT_B_DATATYPE_TOTAL;j++){
      e_cnt += xtemp->root_add->type_count[j];
    }

    ret = rsize/size;
    *cnt = ret * e_cnt;


    ret = rsize%size;

    if(xtemp->root_add->t_comp != -1){
      for(i=0;i<xtemp->root_add->t_comp;i++){
        for(j=0;j<xtemp->root_add->comp[i][5];j++){
          for(k=i;k<i+xtemp->root_add->comp[i][4];k++){
            if(ret > FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[k][0]]->size * 
	       xtemp->root_add->comp[k][1]){
              *cnt+=xtemp->root_add->comp[k][1];
              ret -= FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[k][0]]->size * 
		xtemp->root_add->comp[k][1];
            }
            else {
              *cnt+=ret/FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[k][0]]->size;
              ret = 0;
              return(ret);
            } 
          }
        }
      }
    }
    else {
      etemp = xtemp->root_add->first_e;
      while(etemp!= NULL || ret > 0){
        if(etemp->dt <= FTMPI_DDT_B_DT_MAX){
          if(etemp->count*etemp->size <= ret){
            *cnt+= etemp->count;
            ret -= etemp->count*etemp->size;
          }
          else {
            *cnt += ret/etemp->size;
            ret = ret%etemp->size;
          }
        }
        else {
          ret -= ftmpi_ddt_get_element_count(ddt,ret,cnt,ftmpi_ddt_get_root(etemp->dt),
					     etemp->count);
        }
        etemp = etemp->next;
      }
    }
  }
  else {
    xtemp = ddt_ptr;
    if(ftmpi_mpi_type_size(xtemp->dt,&size) != 0) return(-1);
    if(size == 0) return 0;

    for(j=1;j<FTMPI_DDT_B_DATATYPE_TOTAL;j++){
      e_cnt += xtemp->root_add->type_count[j];
    }

    /*** FITS IN COMPLETELY ***/
    if(size*ddt_cnt <= rsize){  
      *cnt +=e_cnt*ddt_cnt;
      return size*ddt_cnt;
    }

    ret = rsize;

    if(xtemp->root_add->t_comp != -1){
      for(i=0;i<xtemp->root_add->t_comp;i++){
        for(j=0;j<xtemp->root_add->comp[i][5];j++){
          for(k=i;k<i+xtemp->root_add->comp[i][4];k++){
            if(ret > FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[k][0]]->size * 
	       xtemp->root_add->comp[k][1]){
              *cnt+=xtemp->root_add->comp[k][1];
              ret -= FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[k][0]]->size * 
		xtemp->root_add->comp[k][1];
            }
            else {
              *cnt+=ret/FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[k][0]]->size;
              return(rsize-(ret%FTMPI_DDT_BASIC_DATATYPE[xtemp->root_add->comp[k][0]]->size));
            } 
          }
        }
      }
    }
    else {
      etemp = xtemp->root_add->first_e;
      while(etemp!= NULL || ret > 0){
        if(etemp->dt <= FTMPI_DDT_B_DT_MAX){
          if(etemp->count*etemp->size <= ret){
            *cnt+= etemp->count;
            ret -= etemp->count*etemp->size;
          }
          else {
            *cnt += ret/etemp->size;
            ret = ret%etemp->size;
          }
        }
        else {
          ret -= ftmpi_ddt_get_element_count(ddt,ret,cnt,ftmpi_ddt_get_root(etemp->dt),
					     etemp->count);
        }
        etemp = etemp->next;
      }
    }
  }
  return(rsize-ret);
}


/*************************************************************************************/
/* ftmpi_ddt_check_ddt() checks whether or not a ddt is a valid datatype*/
/*************************************************************************************/
int ftmpi_ddt_check_ddt(int ddt,int cmt)
{
  FTMPI_DDT_R_ITEM * xtemp=NULL;
  if(ddt > 0 && ddt <= FTMPI_DDT_B_DT_MAX){
    return(1);
  }

  xtemp = ftmpi_ddt_get_root(ddt);
  if(xtemp == NULL)
    return 0;

  if(xtemp->commited == 0  && cmt == 1){
    return 0;
  }
  return 1;
}


void ftmpi_ddt_increment_dt_uses(int ddt)
{
  FTMPI_DDT_R_ITEM * xtemp=NULL;
  if(ddt < 500) return;

  
  xtemp = ftmpi_ddt_get_root(ddt);
  if(xtemp != NULL){
    xtemp->uses++;
  }
}

void ftmpi_ddt_decrement_dt_uses(int ddt)
{
  FTMPI_DDT_R_ITEM * xtemp=NULL;
  if(ddt < 500) return;

  xtemp = ftmpi_ddt_get_root(ddt);
  if(xtemp != NULL){
    xtemp->uses--;
    if(xtemp->uses == 0){
      MPI_Type_free(&ddt);
    }
  }
}

/**********************************************************************/
/**********************************************************************/
/**********************************************************************/
/* Attribute caching routines */
#ifdef GIVEITATRY
int ftmpi_type_set_attr ( MPI_Datatype type, int key )
{
  FTMPI_DDT_R_ITEM *typeptr;
  int tmp;

  /* allocate memory if required for the list of
     attributes for the new communicator */

  if(type < FTMPI_DDT_B_DT_MAX)
    typeptr = &FTMPI_DDT_BASIC_DATATYPE[type];
  else {
    typeptr = ftmpi_ddt_get_root(type);
    if(typeptr == NULL ){
      return ( MPI_ERR_TYPE );
    }
  }

  tmp = typeptr->nattr;

  if ( tmp > (typeptr->maxattr-1) ) 
    {
      if ( typeptr->attrs != NULL ) 
	{
	  /* there are already some elements set, we need to enlarge the buffer */
	  int *tmpArray, newSize = typeptr->maxattr;

	  /* allocate a bigger array */
	  newSize = typeptr->maxattr + FTMPI_TYPE_ATTR_BLOCK_SIZE;
	  tmpArray = (int *)_MALLOC( newSize * sizeof(int));
	  if( tmpArray == NULL ) return ( MPI_ERR_INTERN );
	  
	  /* restore the old data into the new array and free the temporary buffer */
	  memcpy ( tmpArray, typeptr->attrs,
		   (size_t)( typeptr->maxattr * sizeof (int)));
	  _FREE( typeptr->attrs );
	  typeptr->attrs = tmpArray;
	  typeptr->maxattr = newSize;
	} 
      else 
	{
	  /* first element to be set, array is not yet allocated */
	  typeptr->maxattr += FTMPI_COMM_ATTR_BLOCK_SIZE;
	  typeptr->attrs = (int *)_MALLOC( typeptr->maxattr * sizeof(int));
	  if ( typeptr->attrs == NULL ) return ( MPI_ERR_INTERN );
	}
    }
  
  /* set now the element */
  typeptr->attrs[tmp] = key;
  typeptr->nattr++;
  
  return ( MPI_SUCCESS );
}
/**********************************************************************/
/**********************************************************************/
/**********************************************************************/
int ftmpi_type_del_attr ( MPI_Datatype type, int key )
{
  int i;
  FTMPI_DDT_R_ITEM *typeptr;
  
  
  if(type < FTMPI_DDT_B_DT_MAX)
    typeptr = &FTMPI_DDT_BASIC_DATATYPE[type];
  else 
    {
      typeptr = ftmpi_ddt_get_root(type);
      if(typeptr == NULL ){
	return ( MPI_ERR_TYPE );
      }
    }
  
  /* find first the according element */
  for ( i = 0; i <typeptr->nattr; i++ )
    if ( typeptr->attrs[i] == key ) break;

  /* the impossible error: at this level we shouldn't have an invalid key */
  if ( i >= typeptr->nattr ) return ( MPI_ERR_INTERN );

  typeptr->nattr--;

  /* now we have to shrink the array. I hope memmove doesn't have a problem to move
     0 bytes, because this might happen....*/
  if ( typeptr->nattr > 0 )
    memmove ( &(typeptr->attrs[i]), &(typeptr->attrs[i+1]),
	      ((typeptr->nattr-i) * sizeof(int)));

  return ( MPI_SUCCESS );
}
#endif

/**********************************/
/*** ADVANCED ENCODING ROUTINES ***/
/**********************************/




/**********************************************************************************/
/**********************************************************************************/
/**********************************************************************************/
/**********************************************************************************/
int ftmpi_ddt_dt_mode(int bdt,FTMPI_DDT_CODE_INFO * code_info)
{
  if(bdt == MPI_LONG){
    return code_info->mode_l;
  }
	if(bdt == MPI_LONG_DOUBLE  && ftmpi_ddt_my_longdouble.ld_flag!= code_info->ld_flag){
		return -2;
	}
  return -1;
}
/**********************************************************************************/
/**********************************************************************************/
/**********************************************************************************/
/**********************************************************************************/
int ftmpi_ddt_get_bdt_sizes(FTMPI_DDT_CODE_INFO * code_info, int dt)
{
  if(dt == MPI_LONG && code_info->ld_flag != ftmpi_ddt_my_longdouble.ld_flag ){
    return code_info->size_l;
  }
  else if(dt == MPI_LONG_DOUBLE && 
	  code_info->ld_flag != ftmpi_ddt_my_longdouble.ld_flag){
		return code_info->size_ld;
  }
  return FTMPI_DDT_BASIC_DATATYPE[dt]->size;

}
/**********************************************************************************/
/**********************************************************************************/
/**********************************************************************************/
/**********************************************************************************/
void ftmpi_ddt_set_dt_mode(unsigned int me,unsigned int other,
			   FTMPI_DDT_CODE_INFO * code_info)
{
  int big1,big2,lng1,lng2;

  code_info->code_type = FTMPI_RSC;

#ifdef DO_FTMPI_NOE
  code_info->code_type = FTMPI_NOE;
#endif
#ifdef DO_FTMPI_SSC
  code_info->code_type = FTMPI_SSC;
#endif
#ifdef DO_FTMPI_XDR
  code_info->code_type = FTMPI_XDR;
#endif


  if(code_info->code_type == FTMPI_XDR) return;


  code_info->bs = 0;
  code_info->mode_l = -1;

  code_info->ld_flag = other;

  if(me == other){
    code_info->size_l = FTMPI_DDT_BASIC_DATATYPE[MPI_LONG]->size;
    code_info->size_ld = FTMPI_DDT_BASIC_DATATYPE[MPI_LONG_DOUBLE]->size;
    return;
  }

  /* this can happen for the blank mode */
  if ( other == 0 ) {
    code_info->size_l = FTMPI_DDT_BASIC_DATATYPE[MPI_LONG]->size;
    code_info->size_ld = FTMPI_DDT_BASIC_DATATYPE[MPI_LONG_DOUBLE]->size;
    return;
  }


  big1 = ftmpi_convinfo_checkmask(&me,FTMPI_ISBIGENDIAN);
  big2 = ftmpi_convinfo_checkmask(&other,FTMPI_ISBIGENDIAN);

  lng1 = ftmpi_convinfo_checkmask(&me,FTMPI_LONGIS64);
  lng2 = ftmpi_convinfo_checkmask(&other,FTMPI_LONGIS64);





  if ( ftmpi_convinfo_checkmask (&other, FTMPI_LDEXPSIZEIS15 ))
					code_info->ld_expl = 15;
  else
					code_info->ld_expl = 11;


  if ( ftmpi_convinfo_checkmask (&other, FTMPI_LDMANTDIGIS64))
	  code_info->ld_sigl = 64;
  else if ( ftmpi_convinfo_checkmask (&other, FTMPI_LDMANTDIGIS105))
	  code_info->ld_sigl = 105;
  else if ( ftmpi_convinfo_checkmask (&other, FTMPI_LDMANTDIGIS106))
	  code_info->ld_sigl = 106;
  else if ( ftmpi_convinfo_checkmask (&other, FTMPI_LDMANTDIGIS107))
	  code_info->ld_sigl = 107;
  else if ( ftmpi_convinfo_checkmask (&other, FTMPI_LDMANTDIGIS113))
	  code_info->ld_sigl = 113;
  else
	  code_info->ld_sigl = 113;


	if(!ftmpi_convinfo_checkmask(&other,FTMPI_LDISINTEL)) code_info->ld_sigl--;


  if ( ftmpi_convinfo_checkmask (&other, FTMPI_LONGDOUBLEIS96))
		code_info->size_ld = 12;
  else if ( ftmpi_convinfo_checkmask (&other, FTMPI_LONGDOUBLEIS128))
		code_info->size_ld = 16;
  else
		code_info->size_ld = 8;


  code_info->size_l = 4;
  if(lng2 != 0) code_info->size_l = 8;



  /**********************************************/
  /******        HANDLING THE LONGS        ******/
  /**********************************************/
  if(lng2 == 0){  /* LONG IS 4 BYTES */
    if(lng1 == 0){  /* LONG IS 4 BYTES */
      if(big1 == big2){
        code_info->bs = 0;
        code_info->mode_l = -1;
      }
      else {
        code_info->bs = 1;
        code_info->mode_l = -1;
      }
    }
    else{                          /* LONG IS 8 BYTES */
      if(big1 == big2){
        code_info->bs = 0;
        if(big1 == 0) code_info->mode_l = 3; /* LITTLE ENDIAN */
        else          code_info->mode_l = 0; /* BIG ENDIAN */
      }
      else {
        code_info->bs = 1;
        if(big1 == 0) code_info->mode_l = 1; /* LITTLE ENDIAN */
        else          code_info->mode_l = 2; /* BIG ENDIAN */
      }
    }
  }
  else {                           /* LONG IS 8 BYTES */
    if(lng1 != 0){                 /* LONG IS 8 BYTES */
      if(big1 == big2){
        code_info->bs = 0;
        code_info->mode_l = -1;
      }
      else {
        code_info->bs = 1;
        code_info->mode_l = -1;
      }
    }
    else{                          /* LONG IS 4 BYTES */
      if(big1 == big2){
        code_info->bs = 0;
        if(big1 == 0) code_info->mode_l = 7; /* LITTLE ENDIAN */
        else          code_info->mode_l = 4; /* BIG ENDIAN */
      }
      else {
        code_info->bs = 1;
        if(big1 == 0) code_info->mode_l = 5; /* LITTLE ENDIAN */
        else          code_info->mode_l = 6; /* BIG ENDIAN */
          
      }
    }
  }

  /*****************************************************/
  /******        HANDLING THE LONG DOUBLES        ******/
  /*****************************************************/

  /* THIS WILL BE DONE LATER WHEN WE KNOW WHAT TO DO WITH THIS */


  return;
}
/**********************************************************************************/
/**********************************************************************************/
/**********************************************************************************/
/**********************************************************************************/
int ftmpi_ddt_set_size ( MPI_Datatype type, int *size )
{
  FTMPI_DDT_R_ITEM * temp = NULL;
  *size = -1;
  switch(type){
    case MPI_CHAR:
      *size = sizeof(char);
      break;
    case MPI_UNSIGNED_CHAR:
      *size = sizeof(unsigned char);
      break;
    case MPI_PACKED:
    case MPI_BYTE:
      *size = sizeof(char);
      break;
    case MPI_SHORT:
      *size = sizeof(short);
      break;
    case MPI_UNSIGNED_SHORT:
      *size = sizeof(unsigned short);
      break;
    case MPI_LOGICAL:
    case MPI_INTEGER:
    case MPI_INT:
      *size = sizeof(int);
      break;
    case MPI_UNSIGNED:
      *size = sizeof(unsigned int);
      break;
    case MPI_LONG:
      *size = sizeof(long);
      break;
    case MPI_UNSIGNED_LONG:
      *size = sizeof(unsigned long);
      break;
    case MPI_REAL:
    case MPI_FLOAT:
      *size = sizeof(float);
      break;
    case MPI_DOUBLE_PRECISION:
    case MPI_DOUBLE:
      *size = sizeof(double);
      break;
    case MPI_LONG_DOUBLE:
      *size = sizeof(long double);
      break;
#ifdef HAVE_LONG_LONG
    case MPI_LONG_LONG:
      *size = sizeof(long long);
      break;
#endif
    case MPI_UB:
      *size = 0;
      break;
    case MPI_LB:
      *size = 0;
      break;
    default:
      temp = ftmpi_ddt_get_root(type);
      if(temp == NULL){
        *size = -2;
        printf("NOT A VALID DATATYPE!! - S %d\n", type);
        return ( MPI_ERR_TYPE );
      }
      else {
        *size = temp->size;
      }
      break;
  } 
  if(*size == -2){
    printf("DDT NOT INITIALIZED -- %d --\n",type);
  } 
  return(MPI_SUCCESS);
}
/**********************************************************************************/
/**********************************************************************************/
/**********************************************************************************/
/**********************************************************************************/

int ftmpi_ddt_set_xdr_size ( MPI_Datatype type, int *size )
{   
  *size = -1;
  switch(type){
    case MPI_CHAR:
    case MPI_UNSIGNED_CHAR:
    case MPI_BYTE:
    case MPI_PACKED:
      *size = 1;
      break;
    case MPI_SHORT:
    case MPI_UNSIGNED_SHORT:
      *size = 2;
      break;
    case MPI_LOGICAL:
    case MPI_INTEGER:
    case MPI_INT:
    case MPI_UNSIGNED:
      *size = 4;
      break;
    case MPI_LONG:
    case MPI_UNSIGNED_LONG:
      *size = 4;
      break;
    case MPI_REAL:
    case MPI_FLOAT:
      *size = 4;
      break;
    case MPI_DOUBLE_PRECISION:
    case MPI_DOUBLE:
      *size = 8;
      break;
    case MPI_LONG_DOUBLE:
      *size = sizeof(long double);
      break;
#ifdef HAVE_LONG_LONG
    case MPI_LONG_LONG:
      *size = sizeof(long long);
      break;
#endif
    case MPI_UB:
      *size = 0;
      break;
    case MPI_LB:
      *size = 0;
      break;
    default:
      printf("NOT A VALID DATATYPE!! - Sxdr %d\n", type);
      break;
  } 
  if(*size == -2){
    printf("DDT NOT INITIALIZED -- %d --\n",type);
  } 
  return(MPI_SUCCESS);
}
/**********************************************************************/
/**********************************************************************/
/**********************************************************************/
int ftmpi_mpi_type_xdr_size(MPI_Datatype type, int * size)
{
  FTMPI_DDT_R_ITEM * temp = NULL;

  if(type < FTMPI_DDT_B_DT_MAX)
    *size = FTMPI_DDT_BASIC_DATATYPE[type]->xdr_size;
  else {
    temp = ftmpi_ddt_get_root(type);
    if(temp == NULL ){
      return ( MPI_ERR_TYPE );
    }
    *size = temp->xdr_size;
  }
  return(MPI_SUCCESS);
}
/**********************************************************************************/
/**********************************************************************************/
/**********************************************************************************/
int ftmpi_ddt_set_xdr_extent(MPI_Datatype type, int * extent)
{
  FTMPI_DDT_R_ITEM * temp = NULL;
  int size,ret_var;

  if(type < FTMPI_DDT_B_DT_MAX){
    ret_var = ftmpi_mpi_type_xdr_size(type,&size);
    if(ret_var == MPI_SUCCESS){
      *extent = size;
      return(MPI_SUCCESS);
    }
    return ( MPI_ERR_TYPE );
  } 
  else { 
    temp = ftmpi_ddt_get_root(type);
    if(temp == NULL ){
      return ( MPI_ERR_TYPE );
    }
    else {
      *extent = temp->xdr_extent;
      return(MPI_SUCCESS);
    }
  }
}

/**********************************************************************************/
/**********************************************************************************/
/**********************************************************************************/
void ftmpi_ddt_set_type(int ddt,int type)
{
  FTMPI_DDT_R_ITEM * temp = NULL;

  temp = ftmpi_ddt_get_root(ddt);
  if(temp == NULL ){
    return;
  }
  temp->root_add->create_type = type;
}
/**********************************************************************************/
/**********************************************************************************/
/**********************************************************************************/
void ftmpi_ddt_set_args(int ddt,int ** i, MPI_Aint ** a,int * d,int type)
{
  FTMPI_DDT_R_ITEM * temp = NULL;
  int j, k;

  temp = ftmpi_ddt_get_root(ddt);
  if(temp == NULL ){
    return;
  }

  if(temp->root_add->i != NULL)_FREE(temp->root_add->i);
  if(temp->root_add->a != NULL)_FREE(temp->root_add->a);
  if(temp->root_add->d != NULL)_FREE(temp->root_add->d);

  switch(type){
    /******************************************************************/
    case MPI_COMBINER_DUP: /* DONE */
      temp->root_add->i = (int *)_MALLOC(sizeof(int));
      temp->root_add->a  = (int *)_MALLOC(sizeof(int));
      temp->root_add->d = (int *)_MALLOC(sizeof(int)*2);
     
      temp->root_add->i[0] = 0;

      temp->root_add->a[0] = 0;

      temp->root_add->d[0] = 1;
      temp->root_add->d[1] = d[0];
      break;
    /******************************************************************/
    case MPI_COMBINER_CONTIGUOUS: /* DONE */
      temp->root_add->i = (int *)_MALLOC(sizeof(int)*2);
      temp->root_add->a  = (int *)_MALLOC(sizeof(int));
      temp->root_add->d = (int *)_MALLOC(sizeof(int)*2);

      temp->root_add->i[0] = 1;
      temp->root_add->i[1] = i[0][0];

      temp->root_add->a[0] = 0;

      temp->root_add->d[0] = 1;
      temp->root_add->d[1] = d[0];
      break;
    /******************************************************************/
    case MPI_COMBINER_VECTOR: /* DONE */
      temp->root_add->i = (int *)_MALLOC(sizeof(int)*4);
      temp->root_add->a  = (int *)_MALLOC(sizeof(int));
      temp->root_add->d = (int *)_MALLOC(sizeof(int)*2);

      temp->root_add->i[0] = 3;
      temp->root_add->i[1] = i[0][0];
      temp->root_add->i[2] = i[1][0];
      temp->root_add->i[3] = i[2][0];

      temp->root_add->a[0] = 0;

      temp->root_add->d[0] = 1;
      temp->root_add->d[1] = d[0];
      break;
    /******************************************************************/
    case MPI_COMBINER_HVECTOR_INTEGER: /* DONE */
    case MPI_COMBINER_HVECTOR:
      temp->root_add->i = (int *)_MALLOC(sizeof(int)*3);
      temp->root_add->a  = (int *)_MALLOC(sizeof(int)*2);
      temp->root_add->d = (int *)_MALLOC(sizeof(int)*2);

      temp->root_add->i[0] = 2;
      temp->root_add->i[1] = i[0][0];
      temp->root_add->i[2] = i[1][0];

      temp->root_add->a[0] = 1;
      temp->root_add->a[1] = a[0][0];

      temp->root_add->d[0] = 1;
      temp->root_add->d[1] = d[0];
      break;
    /******************************************************************/
    case MPI_COMBINER_INDEXED: /* DONE */
      j = 0;
      temp->root_add->i = (int *)_MALLOC(sizeof(int)*((2*i[0][0])+2));
      temp->root_add->a  = (int *)_MALLOC(sizeof(int));
      temp->root_add->d = (int *)_MALLOC(sizeof(int)*2);


      temp->root_add->i[j++] = (2*i[0][0])+1;
      temp->root_add->i[j++] = i[0][0];     
      for(k=0;k<i[0][0];k++) temp->root_add->i[j++] = i[1][k];
      for(k=0;k<i[0][0];k++) temp->root_add->i[j++] = i[2][k];

      temp->root_add->a[0] = 0;

      temp->root_add->d[0] = 1;
      temp->root_add->d[1] = d[0];
      break;
    /******************************************************************/
    case MPI_COMBINER_HINDEXED_INTEGER: /* DONE */
    case MPI_COMBINER_HINDEXED:
      temp->root_add->i = (int *)_MALLOC(sizeof(int)*(i[0][0]+2));
      temp->root_add->a  = (int *)_MALLOC(sizeof(int)*(i[0][0]+1));
      temp->root_add->d = (int *)_MALLOC(sizeof(int)*2);

      j = 0;
      temp->root_add->i[j++] = i[0][0]+1;
      temp->root_add->i[j++] = i[0][0];
      for(k=0;k<i[0][0];k++) temp->root_add->i[j++] = i[1][k];

      j = 0;
      temp->root_add->a[j++] = i[0][0];
      for(k=0;k<i[0][0];k++) temp->root_add->a[j++] = a[0][k];

      temp->root_add->d[0] = 1;
      temp->root_add->d[1] = d[0];

      break;
    /******************************************************************/
    case MPI_COMBINER_INDEXED_BLOCK: /* DONE */
      temp->root_add->i = (int *)_MALLOC(sizeof(int)*(i[0][0]+3));
      temp->root_add->a  = (int *)_MALLOC(sizeof(int));
      temp->root_add->d = (int *)_MALLOC(sizeof(int)*2);

      j=0;
      temp->root_add->i[j++] = (i[0][0])+2;
      temp->root_add->i[j++] = i[0][0]; 
      temp->root_add->i[j++] = i[1][0];
      for(k=0;k<i[0][0]+1;k++) temp->root_add->i[j++] = i[2][k];

      temp->root_add->a[0] = 0;

      temp->root_add->d[0] = 1;
      temp->root_add->d[1] = d[0];

      break;
    /******************************************************************/
    case MPI_COMBINER_STRUCT_INTEGER: /* DONE */
    case MPI_COMBINER_STRUCT:
      temp->root_add->i = (int *)_MALLOC(sizeof(int)*(i[0][0]+2));
      temp->root_add->a  = (int *)_MALLOC(sizeof(int)*(i[0][0]+1));
      temp->root_add->d = (int *)_MALLOC(sizeof(int)*(i[0][0]+1));

      j=0;
      temp->root_add->i[j++] = i[0][0]+1;
      temp->root_add->i[j++] = i[0][0];
      for(k=0;k<i[0][0];k++) temp->root_add->i[j++] = i[1][k];
     
      j = 0;
      temp->root_add->a[j++] = i[0][0];
      for(k=0;k<i[0][0];k++) temp->root_add->a[j++] = a[0][k];

      j = 0;
      temp->root_add->d[j++] = i[0][0];
      for(k=0;k<i[0][0];k++) temp->root_add->d[j++] = d[k];

      break;
    /******************************************************************/
    case MPI_COMBINER_SUBARRAY: /* DONE */
      temp->root_add->i = (int *)_MALLOC(sizeof(int)*((3*i[0][0])+3));
      temp->root_add->a  = (int *)_MALLOC(sizeof(int));
      temp->root_add->d = (int *)_MALLOC(sizeof(int)*2);

      j=0;
      temp->root_add->i[j++] = (3*i[0][0])+2;
      temp->root_add->i[j++] = i[0][0];
      for(k=0;k<i[0][0];k++) temp->root_add->i[j++] = i[1][k];
      for(k=0;k<i[0][0];k++) temp->root_add->i[j++] = i[2][k];
      for(k=0;k<i[0][0];k++) temp->root_add->i[j++] = i[3][k];
      temp->root_add->i[j++] = i[4][0];

      temp->root_add->a[0] = 0;
 
      temp->root_add->d[0] = 1;
      temp->root_add->d[1] = d[0];

      break;
    /******************************************************************/
    case MPI_COMBINER_DARRAY:
      temp->root_add->i = (int *)_MALLOC(sizeof(int)*((4*i[0][0])+5));
      temp->root_add->a  = (int *)_MALLOC(sizeof(int));
      temp->root_add->d = (int *)_MALLOC(sizeof(int)*2);

      j=0;
      temp->root_add->i[j++] = (4*i[0][0])+4;
      temp->root_add->i[j++] = i[0][0];
      temp->root_add->i[j++] = i[1][0];
      temp->root_add->i[j++] = i[2][0];

      for(k=0;k<i[2][0];k++) temp->root_add->i[j++] = i[3][k];
      for(k=0;k<i[2][0];k++) temp->root_add->i[j++] = i[4][k];
      for(k=0;k<i[2][0];k++) temp->root_add->i[j++] = i[5][k];
      for(k=0;k<i[2][0];k++) temp->root_add->i[j++] = i[6][k];
      temp->root_add->i[j++] = i[7][0];

      temp->root_add->a[0] = 0;

      temp->root_add->d[0] = 1;
      temp->root_add->d[1] = d[0];

      break;
    /******************************************************************/
    case MPI_COMBINER_F90_REAL:
    case MPI_COMBINER_F90_COMPLEX:
      temp->root_add->i = (int *)_MALLOC(sizeof(int)*3);
      temp->root_add->a  = (int *)_MALLOC(sizeof(int));
      temp->root_add->d = (int *)_MALLOC(sizeof(int));

      temp->root_add->i[0] = 2;
      temp->root_add->i[1] = i[0][0];
      temp->root_add->i[2] = i[1][0];

      temp->root_add->a[0] = 0;

      temp->root_add->d[0] = 0;
      break;
    /******************************************************************/
    case MPI_COMBINER_F90_INTEGER:
      temp->root_add->i = (int *)_MALLOC(sizeof(int)*2);
      temp->root_add->a  = (int *)_MALLOC(sizeof(int));
      temp->root_add->d = (int *)_MALLOC(sizeof(int));

      temp->root_add->i[0] = 1;
      temp->root_add->i[1] = i[0][0];

      temp->root_add->a[0] = 0;

      temp->root_add->d[0] = 0;
      break;
    /******************************************************************/
    case MPI_COMBINER_RESIZED:
      temp->root_add->i = (int *)_MALLOC(sizeof(int));
      temp->root_add->a  = (int *)_MALLOC(sizeof(int)*3);
      temp->root_add->d = (int *)_MALLOC(sizeof(int)*2);

      temp->root_add->i[0] = 0;

      temp->root_add->a[0] = 2;
      temp->root_add->a[1] = a[0][0];
      temp->root_add->a[2] = a[1][0];

      temp->root_add->d[0] = 1;
      temp->root_add->d[1] = d[0];
      break;
    /******************************************************************/
    default:
    break;
  }

  temp->root_add->create_type = type;
}
/**********************************************************************************/
/**********************************************************************************/
/**********************************************************************************/
int ftmpi_ddt_get_args(int ddt,int which,int * ci, int * i,int * ca, int * a,int * cd, 
		       int * d,int * type)
{
  FTMPI_DDT_R_ITEM * temp = NULL;
  int k;


  if(ddt < FTMPI_DDT_B_DATATYPE_TOTAL){
    switch(which){
      case 0:
        *ci = 0;
        *ca = 0;
        *cd = 0;
        *type = MPI_COMBINER_NAMED;
        break;
      default:
/*         printf("ERRROR --- INVALID OPTION  ON FTMPI_DDT_GET_ARGS\n"); */
        return MPI_ERR_INTERN;
        break;
    }
    return(MPI_SUCCESS);
  }

  temp = ftmpi_ddt_get_root(ddt);
  if(temp == NULL ){
    return MPI_ERR_TYPE;
  }

  switch(which){
    case 0:     /* GET THE LENGTHS */
      *ci = temp->root_add->i[0];
      *ca = temp->root_add->a[0];
      *cd = temp->root_add->d[0];
      *type = temp->root_add->create_type;
      break;
    case 1:     /* GET THE ARGUMENTS */
      if(*ci < temp->root_add->i[0] || *ca < temp->root_add->a[0] || *cd < temp->root_add->d[0]) return MPI_ERR_ARG;

      for(k=1;k<temp->root_add->i[0]+1;k++) i[k-1] = temp->root_add->i[k];
      for(k=1;k<temp->root_add->a[0]+1;k++) a[k-1] = temp->root_add->a[k];
      for(k=1;k<temp->root_add->d[0]+1;k++) d[k-1] = temp->root_add->d[k];

      break;
    default:
      printf("ERRROR --- INVALID OPTION  ON FTMPI_DDT_GET_ARGS\n");
      return MPI_ERR_INTERN;
      break;
  }
  return(MPI_SUCCESS);
}
/**********************************************************************************/
/**********************************************************************************/
/**********************************************************************************/
    
/* The IEEE754 floating point representation.
** The following statement are for long double numbers and of course
** are not standards (as expected). But there are the exceptional numbers in
** the logical representation for the long double numbers:
** normal number (0<e<32767): (-1)Sign2 (exponent - 16383)1.f
** subnormal number (e=0, f!=0): (-1)Sign2 (-16382)0.f
** zero (e=0, f=0): (-1)Sign0.0
** signaling NaN s=u, e=32767(max); f=.0u-uu; at least one bit must be nonzero
** quiet NaN s=u, e=32767(max); f=.1uuu-uu
** Infinity s=u, e=32767(max); f=.0000-00 (all zeroes)
*********************************************************
** Now I find 2 different implementation: Sparc and Intel. The main
** difference (except the endianess problem) is that on
** the SPARC implementation the first digit of
** the significant is implicit, but not on Intel. So on all Intel
** architectures we lose one bit of precision...
**/

#ifndef MIN
#define MIN(A,B)  ((A)<(B) ? (A) : (B))
#endif  /* MIN */

int ftmpi_ddt_copy_bits( unsigned int* dest,
         unsigned int dbit,
         unsigned int* src,
         unsigned int sbit,
         unsigned int length,
         int direction,
         int implicit,
         int implicit_bit_value )
{
  int i, sindex = 0, dindex = 0;
  unsigned int dmask, smask;

  printf( "copy %d bits direction %d implicit = %d implicit_bit = %d\n",
	  length, direction, implicit, (implicit_bit_value ? 1 : 0) );
  dmask = (1 << dbit);
  smask = (1 << sbit);
  if( implicit != 0 ) {
    if( implicit == 1 ) {
      smask >>= 1;
      if( smask == 0 ) {
	sindex += direction;
	smask = (((unsigned int)1) << 31);
      }
    } else {
      dmask >>= 1;
      if( dmask == 0 ) {
	dindex += direction;
	dmask = (((unsigned int)1) << 31);
      }
      if( implicit_bit_value ) dest[dindex] |= dmask;
    }
    length--; /* done with one */
  }
  for( i = 0; i < length; i++ ) {
    dmask >>= 1;
    if( dmask == 0 ) {
      dindex += direction;
      dmask = (((unsigned int)1) << 31);
    }
    smask >>= 1;
    if( smask == 0 ) {
      sindex += direction;
      smask = (((unsigned int)1) << 31);
    }
    if( src[sindex] & smask )
      dest[dindex] |= dmask;
  }
  return 0;
}

/* #define eprintf printf */
#define eprintf (void*)

static void dump_memory_order( unsigned char* pc, unsigned int length )
{
  unsigned int dumped = 0;
  
  while( length ) {
    printf( "%02x", *pc );
    dumped++;
    if( dumped == 4 ) {
      printf( " " ); dumped = 0;
    }
    length--; pc++;
  }
  printf( "\n" );
}

static void swap_bytes( unsigned int* pui )
{
  unsigned char *pc = (unsigned char*)pui, cswap;
  cswap = pc[0]; pc[0] = pc[3]; pc[3] = cswap;
  cswap = pc[1]; pc[1] = pc[2]; pc[2] = cswap;
}

/* This function convert between big endian and little endian
 * representations. Their only job is to put the number in memory
 * in the other representation (from big to little and of
 * course from little to big).
 */
static void swap_words( unsigned int* ptr, unsigned int length )
{
  unsigned int uitemp, i;

  length--; /* small trick to use length as the swapping index */
  /* just < to avoid double conversion for the middle word */
  for( i = 0; i < length; i++, length-- ) {
    swap_bytes( &(ptr[i]) );
    swap_bytes( &(ptr[length]) );
    uitemp = ptr[i];
    ptr[i] = ptr[length];
    ptr[length] = uitemp;
  }
  if( i == length ) swap_bytes( &(ptr[i]) );
}

#define isBigEndian(DDT_DECODE) (ftmpi_convinfo_checkmask(&((DDT_DECODE).ld_flag), FTMPI_ISBIGENDIAN))
#define useIntelRepresentation(DDT_DECODE) (ftmpi_convinfo_checkmask(&((DDT_DECODE).ld_flag), FTMPI_LDISINTEL))

int
ftmpi_ddt_convert_long_double( char* buffer,      /* source buffer */
			       long double* pld,  /* destination buffer */
			       int count,         /* number of long double */
			       FTMPI_DDT_CODE_INFO * remote_info ) /* remote info */
{
/*   ftmpi_ddt_my_longdouble -- LONG DOUBLE DEFINITION FOR MY ARCH */

  unsigned int* puir = (unsigned int*)buffer;
  unsigned int* puil = (unsigned int*)pld;
  unsigned int value, rsign, mask, lexp, rexp;
  unsigned int ldepl, rdepl;
  unsigned int rmax_exp, lmax_exp, shift_sign, i;
  int lindex, rindex, direction = 1, add_sign = 0;

  if( isBigEndian(*remote_info) != isBigEndian(ftmpi_ddt_my_longdouble) ) {
    /* if they have a different endianness */
    eprintf( "swap words => now both have the same endian\n" );
    swap_words( (unsigned int*)buffer, remote_info->size_ld / sizeof(unsigned int) );
  }
  /* check if we need to add the 1 before the significand.
   * add_sign will be 0 if we same similars architectures,
   * -1 if the remote architecture is a sparc and the local is an intel
   * and 1 if remote is intel and local sparc.
   */
  add_sign = (useIntelRepresentation(*remote_info) - 
	      useIntelRepresentation(ftmpi_ddt_my_longdouble));
  eprintf( "add_sign = %d %08x\n", add_sign, add_sign );
  dump_memory_order( (unsigned char*)buffer, remote_info->size_ld );
  /* lets compute local and remote index of exponent and shift */
  ldepl = (ftmpi_ddt_my_longdouble.ld_sigl % (sizeof(unsigned int) << 3));
  rdepl = (remote_info->ld_sigl % (sizeof(unsigned int) << 3));
  rindex = ((remote_info->size_ld << 3) - 
	    (remote_info->ld_sigl + remote_info->ld_expl)) >> 5;
  lindex = ((ftmpi_ddt_my_longdouble.size_ld << 3) - 
	    (ftmpi_ddt_my_longdouble.ld_sigl + ftmpi_ddt_my_longdouble.ld_expl)) >> 5;
  rmax_exp = (1 << (remote_info->ld_expl - 1)) - 1;
  lmax_exp = (1 << (ftmpi_ddt_my_longdouble.ld_expl - 1)) - 1;
  shift_sign = ldepl - (rdepl-1);
  if( !isBigEndian(ftmpi_ddt_my_longdouble) ) { /* if little endian */
    puir = puir + (remote_info->size_ld >> 2) - 1;
    puil = puil + (ftmpi_ddt_my_longdouble.size_ld >> 2) - 1;
   /*  rindex = -rindex; */
/*     lindex = -lindex; */
    direction = -1;
  }

  /* dump all those informations */
  eprintf( "Local  info: ldepl = %d, lindex = %d lmax_exp = %d\n", ldepl, lindex, lmax_exp );
  eprintf( "Remote info: rdepl = %d, rindex = %d rmax_exp = %d\n", rdepl, rindex, rmax_exp );

  /* where the remote exponent start */
  if( (rindex >> 5) != ((rindex - remote_info->ld_expl) >> 5) ) {
    eprintf( ">> error exponent on 2 words significand length %d bits\n",
	     remote_info->ld_sigl );
    eprintf( ">> index = %d exponent is %d bits\n", rindex, remote_info->ld_expl );
  }

  for( i = 0; i < count; i++ ) {
    /* lets compute the remote sign and exponent */
    value = puir[rindex] >> rdepl;
    mask = 1 << remote_info->ld_expl;
    rsign = value & mask; /* here we got the remote sign */
    mask--; /* create the mask for the exponent */
    eprintf( "initial data %08x mask = %08x  value = %08x (middle %x)\n",
	     puir[rindex], mask, value, rmax_exp );
    rexp = (value & mask) - rmax_exp;
    eprintf( "remote number is %s and has an exponent = %d\n",
	     (rsign ? "negatif" : "positif"), rexp );
    lexp = lmax_exp + rexp;
    eprintf( "local E = %d\n", lexp );
    /* now check if the local exp can be represented */
    /* lets copy the significand */
    /* if the local exponent is 0 we have a subnormal number so
     * we dont need to add the leading bit as it's already set to 0.
     */
    ftmpi_ddt_copy_bits( puil + lindex, ldepl,
			 puir + rindex, rdepl,
			 MIN( remote_info->ld_sigl, ftmpi_ddt_my_longdouble.ld_sigl ),
			 direction, add_sign, (lexp != 0) );
    /* lets put the sign along with the exponent */
    if( rsign ) lexp |= ( 1 << ftmpi_ddt_my_longdouble.ld_expl );
    /* now lets shitf the exponent to the right place */
    lexp <<= ldepl;
    /* and add it to the long double number in the right position */
    puil[lindex] |= lexp;
    buffer += remote_info->size_ld;
    pld += 1;
  }
  return 0;
}
